/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.oauth.process;

import org.mule.RequestContext;
import org.mule.api.MuleEvent;
import org.mule.api.construct.FlowConstructAware;
import org.mule.api.context.MuleContextAware;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.Startable;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.api.annotations.JustOnce;
import org.mule.devkit.generation.oauth.AbstractOAuthAdapterGenerator;
import org.mule.devkit.generation.oauth.OAuthClientNamingConstants;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthModule;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

@JustOnce
public class DefaultSaveAccessTokenCallbackGenerator extends AbstractOAuthAdapterGenerator {

    private final static List<Product> CONSUMES = Arrays.asList(Product.OAUTH_INTERFACES);
    private final static List<Product> PRODUCES = Arrays.asList(Product.DEFAULT_SAVE_ACCESS_TOKEN_CALLBACK);

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule && ((OAuthModule) module).getUserIdentifierMethod() == null;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass callbackClass = getDefaultSaveAccessTokenCallbackClass(module);

        GeneratedField messageProcessor = generateFieldForMessageProcessor(callbackClass, "messageProcessor");
        GeneratedField logger = generateLoggerField(callbackClass);
        GeneratedField hasBeenStarted = generateFieldForBoolean(callbackClass, "hasBeenStarted");
        GeneratedField hasBeenInitialized = generateFieldForBoolean(callbackClass, "hasBeenInitialized");

        GeneratedMethod constructor = callbackClass.constructor(Modifier.PUBLIC);
        constructor.body().assign(hasBeenStarted, ExpressionFactory.FALSE);
        constructor.body().assign(hasBeenInitialized, ExpressionFactory.FALSE);

        callbackClass.getter(messageProcessor);
        callbackClass.setter(messageProcessor);

        GeneratedMethod saveAccessTokenMethod = callbackClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "saveAccessToken");
        GeneratedVariable accessToken = saveAccessTokenMethod.param(ref(String.class), "accessToken");
        GeneratedVariable accessTokenSecret = saveAccessTokenMethod.param(ref(String.class), "accessTokenSecret");

        GeneratedVariable event = saveAccessTokenMethod.body().decl(ref(MuleEvent.class), "event", ref(RequestContext.class).staticInvoke("getEvent"));
        GeneratedConditional ifAccessTokenNotNull = saveAccessTokenMethod.body()._if(accessToken.isNotNull());
        ifAccessTokenNotNull._then().add(
                event.invoke("getMessage").invoke("setInvocationProperty").arg("OAuthAccessToken").arg(accessToken)
        );
        GeneratedConditional ifAccessTokenSecretNotNull = saveAccessTokenMethod.body()._if(accessTokenSecret.isNotNull());
        ifAccessTokenSecretNotNull._then().add(
                event.invoke("getMessage").invoke("setInvocationProperty").arg("OAuthAccessTokenSecret").arg(accessTokenSecret)
        );

        GeneratedConditional ifMuleContextAware = saveAccessTokenMethod.body()._if(Op._instanceof(messageProcessor, ref(MuleContextAware.class)));
        ifMuleContextAware._then().add(
                ExpressionFactory.cast(ref(MuleContextAware.class), messageProcessor).invoke("setMuleContext").arg(
                        ref(RequestContext.class).staticInvoke("getEventContext").invoke("getMuleContext")
                )
        );

        GeneratedConditional ifFlowConstructAware = saveAccessTokenMethod.body()._if(Op._instanceof(messageProcessor, ref(FlowConstructAware.class)));
        ifFlowConstructAware._then().add(
                ExpressionFactory.cast(ref(FlowConstructAware.class), messageProcessor).invoke("setFlowConstruct").arg(
                        ref(RequestContext.class).staticInvoke("getEventContext").invoke("getFlowConstruct")
                )
        );

        GeneratedConditional ifNotInitialized = saveAccessTokenMethod.body()._if(Op.not(hasBeenInitialized));
        GeneratedConditional ifInitialisable = ifNotInitialized._then()._if(Op._instanceof(messageProcessor, ref(Initialisable.class)));
        GeneratedTry tryToInitialize = ifInitialisable._then()._try();
        tryToInitialize.body().add(
                ExpressionFactory.cast(ref(Initialisable.class), messageProcessor).invoke("initialise")
        );
        GeneratedCatchBlock catchInitlize = tryToInitialize._catch(ref(Exception.class));
        GeneratedVariable exception = catchInitlize.param("e");
        catchInitlize.body().add(
                logger.invoke("error").arg(
                        exception.invoke("getMessage")
                ).arg(
                        exception
                )
        );
        ifNotInitialized._then().assign(hasBeenInitialized, ExpressionFactory.TRUE);

        GeneratedConditional ifNotStarted = saveAccessTokenMethod.body()._if(Op.not(hasBeenStarted));
        GeneratedConditional ifStartable = ifNotStarted._then()._if(Op._instanceof(messageProcessor, ref(Startable.class)));
        GeneratedTry tryToStart = ifStartable._then()._try();
        tryToStart.body().add(
                ExpressionFactory.cast(ref(Startable.class), messageProcessor).invoke("start")
        );
        GeneratedCatchBlock catchStart = tryToStart._catch(ref(Exception.class));
        exception = catchStart.param("e");
        catchStart.body().add(
                logger.invoke("error").arg(
                        exception.invoke("getMessage")
                ).arg(
                        exception
                )
        );

        ifNotStarted._then().assign(hasBeenStarted, ExpressionFactory.TRUE);

        GeneratedTry tryProcess = saveAccessTokenMethod.body()._try();
        tryProcess.body().add(
                messageProcessor.invoke("process").arg(event)
        );
        GeneratedCatchBlock catchProcess = tryProcess._catch(ref(Exception.class));
        exception = catchProcess.param("e");
        catchProcess.body().add(
                logger.invoke("error").arg(
                        exception.invoke("getMessage")
                ).arg(
                        exception
                )
        );
    }

    private GeneratedClass getDefaultSaveAccessTokenCallbackClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);
        GeneratedClass clazz = pkg._class(OAuthClientNamingConstants.DEFAULT_SAVE_ACCESS_TOKEN_CALLBACK_CLASS_NAME);
        clazz._implements((TypeReference) ctx().getProduct(Product.SAVE_ACCESS_TOKEN_CALLBACK_INTERFACE));
        ctx().registerProduct(Product.DEFAULT_SAVE_ACCESS_TOKEN_CALLBACK, clazz);

        return clazz;
    }
}
