/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */

package org.mule.devkit.generation.oauth.process;

import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;
import org.mule.api.processor.MessageProcessor;
import org.mule.api.routing.filter.Filter;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.ModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.api.annotations.JustOnce;
import org.mule.devkit.generation.oauth.AbstractOAuthAdapterGenerator;
import org.mule.devkit.generation.oauth.OAuthClientNamingConstants;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthModule;

import java.util.Arrays;
import java.util.List;

@JustOnce
public class ManagedAccessTokenProcessInterceptorGenerator extends AbstractOAuthAdapterGenerator implements ModuleGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(Product.CONNECTION_KEY,
            Product.OAUTH_ADAPTER,
            Product.ABSTRACT_EXPRESSION_EVALUATOR,
            Product.PROCESS_INTERFACES,
            Product.CONNECTION_INTERFACES);
    private final static List<Product> PRODUCES = Arrays.asList(Product.MANAGED_ACCESS_TOKEN_PROCESS_INTERCEPTOR);

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule && ((OAuthModule) module).getUserIdentifierMethod() != null;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        OAuthModule oauthModule = (OAuthModule) module;
        GeneratedClass oauthAdapterClass = ctx().getProduct(Product.OAUTH_ADAPTER, module);

        GeneratedClass managedAccessTokenProcessInterceptorClass = getManagedAccessTokenProcessInterceptorClass(oauthModule);
        TypeVariable t = managedAccessTokenProcessInterceptorClass.generify("T");
        managedAccessTokenProcessInterceptorClass._implements(((TypeReference) ctx().getProduct(Product.PROCESS_INTERCEPTOR_INTERFACE)).narrow(t).narrow(oauthAdapterClass));

        GeneratedField logger = generateLoggerField(managedAccessTokenProcessInterceptorClass);
        GeneratedField connectionManager = generateOAuthManagedField(oauthAdapterClass, managedAccessTokenProcessInterceptorClass);
        GeneratedField muleContext = managedAccessTokenProcessInterceptorClass.field(Modifier.FINAL | Modifier.PRIVATE, ref(MuleContext.class), "muleContext");
        GeneratedField next = managedAccessTokenProcessInterceptorClass.field(Modifier.FINAL | Modifier.PRIVATE, ((TypeReference) ctx().getProduct(Product.PROCESS_INTERCEPTOR_INTERFACE)).narrow(t).narrow(oauthAdapterClass), "next");

        generateConstructor(managedAccessTokenProcessInterceptorClass, oauthModule, next, connectionManager, muleContext);

        generateExecuteMethod(oauthModule, oauthAdapterClass, managedAccessTokenProcessInterceptorClass, t, logger, connectionManager, next, muleContext);
        generateExecuteMethodForMessage(oauthModule, oauthAdapterClass, managedAccessTokenProcessInterceptorClass, t, logger, connectionManager, next, muleContext);
    }

    private void generateExecuteMethod(OAuthModule oauthModule, GeneratedClass oauthAdapterClass, GeneratedClass managedAccessTokenProcessInterceptorClass, TypeVariable t, GeneratedField logger, GeneratedField connectionManager, GeneratedField next, GeneratedField muleContext) {
        GeneratedMethod execute = managedAccessTokenProcessInterceptorClass.method(Modifier.PUBLIC, t, "execute");
        execute._throws(ref(Exception.class));
        GeneratedVariable processCallback = execute.param(((TypeReference) ctx().getProduct(Product.PROCESS_CALLBACK_INTERFACE)).narrow(t).narrow(oauthAdapterClass), "processCallback");
        GeneratedVariable object = execute.param(oauthAdapterClass, "object");
        GeneratedVariable messageProcessor = execute.param(ref(MessageProcessor.class), "messageProcessor");
        GeneratedVariable event = execute.param(ref(MuleEvent.class), "event");

        GeneratedVariable connector = execute.body().decl(oauthAdapterClass, "connector", ExpressionFactory._null());
        GeneratedClass abstractConnectedProcessor = ctx().getProduct(Product.ABSTRACT_CONNECTED_PROCESSOR);

        execute.body()._if(Op.not(processCallback.invoke("isProtected")))._then()._return(processCallback.invoke("process").arg(connectionManager.invoke("getDefaultUnauthorizedConnector")));

        execute.body()._if(ExpressionFactory.cast(abstractConnectedProcessor, messageProcessor).invoke("getAccessTokenId").isNull())._then()._throw(ExpressionFactory._new(ref(IllegalArgumentException.class)).arg("The accessTokenId cannot be null"));

        GeneratedInvocation getGenericType = abstractConnectedProcessor.dotclass().invoke("getDeclaredField").arg(ExpressionFactory.lit("_accessTokenIdType")).invoke("getGenericType");
        GeneratedInvocation evaluateAndTransform = ExpressionFactory.invoke("evaluateAndTransform").arg(muleContext).arg(event).arg(getGenericType)
                .arg(ExpressionFactory._null()).arg(ExpressionFactory.cast(abstractConnectedProcessor, messageProcessor).invoke("getAccessTokenId"));

        GeneratedVariable tokenId = execute.body().decl(ref(String.class), "_transformedToken", ExpressionFactory.cast(ref(String.class), evaluateAndTransform));

        GeneratedTry tryCatch = execute.body()._try();

        tryCatch.body()._if(logger.invoke("isDebugEnabled"))._then().add(logger.invoke("debug").arg(Op.plus(ExpressionFactory.lit("Attempting to acquire access token using from store for [accessTokenId="), tokenId.invoke("toString"))));

        tryCatch.body().assign(connector, connectionManager.invoke("acquireAccessToken").arg(tokenId));

        GeneratedConditional isConnectionNull = tryCatch.body()._if(connector.isNull());
        isConnectionNull._then()._throw(ExpressionFactory._new((TypeReference)ctx().getProduct(Product.UNABLE_TO_ACQUIRE_CONNECTION_EXCEPTION)));
        isConnectionNull._else()._if(logger.invoke("isDebugEnabled"))._then().add(
                logger.invoke("debug").arg(Op.plus(Op.plus(ExpressionFactory.lit("Access token has been acquired for [accessTokenId="), connector.invoke(oauthModule.getUserIdentifierMethod().getName())), ExpressionFactory.lit("]"))));

        tryCatch.body()._return(next.invoke("execute").arg(processCallback).arg(connector).arg(messageProcessor).arg(event));

        GeneratedCatchBlock catchBlock = tryCatch._catch(ref(Exception.class));
        GeneratedVariable e = catchBlock.param("e");
        GeneratedForEach forEachException = catchBlock.body()._if(Op.cand(processCallback.invoke("getManagedExceptions").isNotNull(), connector.isNotNull()))._then().
                forEach(ref(Class.class), "exceptionClass", ExpressionFactory.cast(ref(List.class).narrow(Class.class), processCallback.invoke("getManagedExceptions")));
        GeneratedConditional ifIsInstance = forEachException.body()._if(forEachException.var().invoke("isInstance").arg(e));
        ifIsInstance._then()._if(logger.invoke("isDebugEnabled"))._then().add(logger.invoke("debug").arg(
                Op.plus(Op.plus(
                        Op.plus(
                                Op.plus(ExpressionFactory.lit("An exception ( "), forEachException.var().invoke("getName")),
                                ExpressionFactory.lit(") has been thrown. Destroying the access token with [accessTokenId=")),
                        connector.invoke(oauthModule.getUserIdentifierMethod().getName())), ExpressionFactory.lit("]"))
        ));

        GeneratedTry tryToDestroy = ifIsInstance._then()._try();
        tryToDestroy.body().add(connectionManager.invoke("destroyAccessToken").arg(tokenId).arg(connector));
        tryToDestroy.body().assign(connector, ExpressionFactory._null());

        GeneratedCatchBlock releaseCatch = tryToDestroy._catch(ref(Exception.class));
        GeneratedVariable innerException = releaseCatch.param("innerException");
        releaseCatch.body().add(logger.invoke("error").arg(innerException.invoke("getMessage")).arg(innerException));

        catchBlock.body()._throw(e);

        GeneratedTry tryToRelease = tryCatch._finally()._try();
        GeneratedConditional connectionIsNotNull = tryToRelease.body()._if(connector.isNotNull());
        connectionIsNotNull._then()._if(logger.invoke("isDebugEnabled"))._then().add(
                logger.invoke("debug").arg(Op.plus(Op.plus(ExpressionFactory.lit("Releasing the access token back into the pool [accessTokenId="), connector.invoke(oauthModule.getUserIdentifierMethod().getName())), ExpressionFactory.lit("]")))
        );

        connectionIsNotNull._then().add(connectionManager.invoke("releaseAccessToken").arg(tokenId).arg(connector));
        GeneratedCatchBlock catchException = tryToRelease._catch(ref(Exception.class));
        GeneratedVariable exception = catchException.param("e");
        catchException.body()._throw(ExpressionFactory._new((TypeReference)ctx().getProduct(Product.UNABLE_TO_RELEASE_CONNECTION_EXCEPTION)).arg(exception));
    }

    private void generateExecuteMethodForMessage(OAuthModule oauthModule, GeneratedClass oauthAdapterClass, GeneratedClass managedAccessTokenProcessInterceptorClass, TypeVariable t, GeneratedField logger, GeneratedField connectionManager, GeneratedField next, GeneratedField muleContext) {
        GeneratedMethod execute = managedAccessTokenProcessInterceptorClass.method(Modifier.PUBLIC, t, "execute");
        execute._throws(ref(Exception.class));
        GeneratedVariable processCallback = execute.param(((TypeReference) ctx().getProduct(Product.PROCESS_CALLBACK_INTERFACE)).narrow(t).narrow(oauthAdapterClass), "processCallback");
        GeneratedVariable object = execute.param(oauthAdapterClass, "object");
        GeneratedVariable filter = execute.param(ref(Filter.class), "filter");
        GeneratedVariable message = execute.param(ref(MuleMessage.class), "message");

        execute.body()._throw(ExpressionFactory._new(ref(UnsupportedOperationException.class)));
    }

    private void generateConstructor(GeneratedClass managedAccessTokenProcessInterceptorClass, OAuthModule module, GeneratedField nextField, GeneratedField oauthManagerField, GeneratedField muleContextField) {
        GeneratedClass oauthAdapterClass = ctx().getProduct(Product.OAUTH_ADAPTER, module);

        GeneratedMethod constructor = managedAccessTokenProcessInterceptorClass.constructor(Modifier.PUBLIC);
        GeneratedVariable next = constructor.param(((TypeReference) ctx().getProduct(Product.PROCESS_INTERCEPTOR_INTERFACE)).narrow(managedAccessTokenProcessInterceptorClass.typeParams()[0]).narrow(oauthAdapterClass), "next");
        GeneratedVariable oauthManager = constructor.param(((TypeReference) ctx().getProduct(Product.OAUTH_MANAGER_INTERFACE)).narrow(oauthAdapterClass), "oauthManager");
        GeneratedVariable muleContext = constructor.param(ref(MuleContext.class), "muleContext");

        constructor.body().assign(ExpressionFactory._this().ref(nextField), next);
        constructor.body().assign(ExpressionFactory._this().ref(oauthManagerField), oauthManager);
        constructor.body().assign(ExpressionFactory._this().ref(muleContextField), muleContext);
    }

    private GeneratedClass getManagedAccessTokenProcessInterceptorClass(OAuthModule type) {
        GeneratedClass abstractExpressionEvaluatorClass = ctx().getProduct(Product.ABSTRACT_EXPRESSION_EVALUATOR);

        GeneratedPackage pkg = ctx().getCodeModel()._package(type.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);
        GeneratedClass clazz = pkg._class(Modifier.PUBLIC, OAuthClientNamingConstants.MANAGED_CONNECTION_PROCESS_INTERCEPTOR_CLASS_NAME);
        clazz._extends(abstractExpressionEvaluatorClass);

        ctx().registerProduct(Product.MANAGED_ACCESS_TOKEN_PROCESS_INTERCEPTOR, clazz);

        return clazz;
    }

    private GeneratedField generateOAuthManagedField(GeneratedClass oauthAdapterClass, GeneratedClass processTemplateClass) {
        return processTemplateClass.field(Modifier.PRIVATE | Modifier.FINAL, ((TypeReference) ctx().getProduct(Product.OAUTH_MANAGER_INTERFACE)).narrow(oauthAdapterClass), "oauthManager");
    }
}
