/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.oauth.processors;

import org.mule.api.DefaultMuleException;
import org.mule.api.MuleEvent;
import org.mule.api.MuleException;
import org.mule.api.construct.FlowConstructAware;
import org.mule.api.context.MuleContextAware;
import org.mule.api.processor.MessageProcessor;
import org.mule.config.i18n.CoreMessages;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.oauth.AbstractOAuthAdapterGenerator;
import org.mule.devkit.generation.oauth.OAuthClientNamingConstants;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthModule;

import java.util.Arrays;
import java.util.List;

public class UnauthorizeMessageProcessorGenerator extends AbstractOAuthAdapterGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(new Product[]{Product.OAUTH_ADAPTER, Product.ABSTRACT_MESSAGE_PROCESSOR, Product.ABSTRACT_CONNECTED_PROCESSOR, Product.OAUTH_MANAGER, Product.CONNECTION_INTERFACES});
    private final static List<Product> PRODUCES = Arrays.asList(new Product[]{Product.MESSAGE_PROCESSOR});

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        // get class
        GeneratedClass messageProcessorClass;

        messageProcessorClass = getUnauthorizeMessageProcessorClass(module);

        GeneratedField logger = generateLoggerField(messageProcessorClass);

        generateProcessMethod((OAuthModule)module, messageProcessorClass, logger);
    }

    private void generateProcessMethod(OAuthModule oauthModule, GeneratedClass messageProcessorClass, GeneratedField logger) {
        // add process method
        org.mule.devkit.model.code.Type muleEvent = ref(MuleEvent.class);

        GeneratedMethod process = messageProcessorClass.method(Modifier.PUBLIC, muleEvent, "process");
        process.javadoc().add("Unauthorize the connector");
        process.javadoc().addParam("event MuleEvent to be processed");
        process.javadoc().addThrows(ref(MuleException.class));

        process._throws(MuleException.class);
        GeneratedVariable event = process.param(muleEvent, "event");

        GeneratedVariable moduleObject = process.body().decl(getModuleObject(oauthModule), "moduleObject", ExpressionFactory._null());

        GeneratedTry attempt = process.body()._try();

        GeneratedInvocation findOrCreate = ExpressionFactory.invoke("findOrCreate").arg(getModuleObject(oauthModule).dotclass());
        if (oauthModule.needsConfig()) {
            findOrCreate.arg(ExpressionFactory.FALSE);
        } else {
            findOrCreate.arg(ExpressionFactory.TRUE);
        }
        findOrCreate.arg(ExpressionFactory._null());
        attempt.body().assign(moduleObject, findOrCreate);

        GeneratedCatchBlock catchIllegalAccess = attempt._catch(ref(IllegalAccessException.class));
        catchIllegalAccess.body()._throw(ExpressionFactory._new(ref(DefaultMuleException.class)).arg(ref(CoreMessages.class).staticInvoke("failedToStart").arg("authorize")).arg(catchIllegalAccess.param("e")));

        GeneratedCatchBlock catchInstantiationException = attempt._catch(ref(InstantiationException.class));
        catchInstantiationException.body()._throw(ExpressionFactory._new(ref(DefaultMuleException.class)).arg(ref(CoreMessages.class).staticInvoke("failedToStart").arg("authorize")).arg(catchInstantiationException.param("e")));


        if( oauthModule.getUserIdentifierMethod() == null ) {
            process.body().add(moduleObject.invoke("reset"));
            process.body()._return(event);
        } else {
            GeneratedClass oauthAdapterClass = ctx().getProduct(Product.OAUTH_ADAPTER, oauthModule);
            GeneratedVariable connector = process.body().decl(oauthAdapterClass, "connector", ExpressionFactory._null());

            attempt = process.body()._try();

            GeneratedClass abstractConnectedProcessor = ctx().getProduct(Product.ABSTRACT_CONNECTED_PROCESSOR);
            GeneratedInvocation getGenericType = abstractConnectedProcessor.dotclass().invoke("getDeclaredField").arg(ExpressionFactory.lit("_accessTokenIdType")).invoke("getGenericType");
            GeneratedInvocation evaluateAndTransform = ExpressionFactory.invoke("evaluateAndTransform").arg(ExpressionFactory.invoke("getMuleContext")).arg(event).arg(getGenericType)
                    .arg(ExpressionFactory._null()).arg(ExpressionFactory.invoke("getAccessTokenId"));

            GeneratedVariable tokenId = attempt.body().decl(ref(String.class), "_transformedToken", ExpressionFactory.cast(ref(String.class), evaluateAndTransform));

            attempt.body()._if(logger.invoke("isDebugEnabled"))._then().add(logger.invoke("debug").arg(Op.plus(ExpressionFactory.lit("Attempting to acquire access token using from store for user "), tokenId.invoke("toString"))));

            attempt.body().assign(connector, moduleObject.invoke("acquireAccessToken").arg(tokenId));

            GeneratedConditional isConnectionNull = attempt.body()._if(connector.isNull());
            isConnectionNull._then()._throw(ExpressionFactory._new((TypeReference)ctx().getProduct(Product.UNABLE_TO_ACQUIRE_CONNECTION_EXCEPTION)));
            isConnectionNull._else()._if(logger.invoke("isDebugEnabled"))._then().add(
                    logger.invoke("debug").arg(Op.plus(Op.plus(ExpressionFactory.lit("Access Token has been acquired for [tokenId="), connector.invoke(oauthModule.getUserIdentifierMethod().getName())), ExpressionFactory.lit("]"))));

            isConnectionNull._else().invoke(moduleObject, "destroyAccessToken").arg(tokenId).arg(connector);

            isConnectionNull._else()._if(logger.invoke("isDebugEnabled"))._then().add(
                    logger.invoke("debug").arg(Op.plus(Op.plus(ExpressionFactory.lit("Access token for [tokenId="), connector.invoke(oauthModule.getUserIdentifierMethod().getName())), ExpressionFactory.lit("] has been successfully destroyed"))));

            GeneratedCatchBlock catchException = attempt._catch(ref(Exception.class));
            GeneratedVariable e = catchException.param("e");
            catchException.body()._throw(ExpressionFactory._new(ref(DefaultMuleException.class)).arg(ref(CoreMessages.class).staticInvoke("createStaticMessage").arg("Unable to unauthorize the connector")).arg(e));

            process.body()._return(event);
        }
    }

    private GeneratedClass getUnauthorizeMessageProcessorClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.MESSAGE_PROCESSOR_NAMESPACE);

        GeneratedClass abstractMessageProcessor = ctx().getProduct(Product.ABSTRACT_MESSAGE_PROCESSOR);

        GeneratedClass moduleObject = null;
        if( module instanceof OAuthModule && ((OAuthModule)module).getUserIdentifierMethod() != null ) {
            moduleObject = ctx().<GeneratedClass>getProduct(Product.OAUTH_MANAGER, module).topLevelClass();
        } else {
            moduleObject = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        }

        GeneratedClass clazz = pkg._class(OAuthClientNamingConstants.UNAUTHORIZE_MESSAGE_PROCESSOR_CLASS_NAME, abstractMessageProcessor.narrow(moduleObject), new Class[]{
                MessageProcessor.class,
                MuleContextAware.class,
                FlowConstructAware.class});

        ctx().registerProduct(Product.MESSAGE_PROCESSOR, module, "unauthorize", clazz);

        return clazz;
    }

    protected GeneratedClass getModuleObject(org.mule.devkit.model.Type type) {
        if (type instanceof OAuthModule && ((OAuthModule) type).getUserIdentifierMethod() != null) {
            return ctx().getProduct(Product.OAUTH_MANAGER, type);
        } else {
            if (ctx().<GeneratedClass>getProduct(Product.PROCESS_ADAPTER, type) != null) {
                return ctx().<GeneratedClass>getProduct(Product.PROCESS_ADAPTER, type).topLevelClass();
            }
        }

        return null;
    }
}
