/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.oauth;

import oauth.signpost.OAuthConsumer;
import oauth.signpost.basic.DefaultOAuthConsumer;
import oauth.signpost.exception.OAuthCommunicationException;
import oauth.signpost.exception.OAuthExpectationFailedException;
import oauth.signpost.exception.OAuthMessageSignerException;
import oauth.signpost.exception.OAuthNotAuthorizedException;
import oauth.signpost.signature.AuthorizationHeaderSigningStrategy;
import oauth.signpost.signature.HmacSha1MessageSigner;
import oauth.signpost.signature.PlainTextMessageSigner;
import oauth.signpost.signature.QueryStringSigningStrategy;
import org.mule.api.annotations.oauth.OAuth;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;

public class OAuth1ClientAdapterGenerator extends AbstractOAuthAdapterGenerator {

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule && ((OAuthModule) module).getOAuthVersion() == OAuthVersion.V10A;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass oauthAdapter = getOAuthAdapterClass(module, "OAuth1Adapter", (TypeReference) ctx().getProduct(Product.OAUTH1_ADAPTER_INTERFACE));
        OAuthModule oAuthModule = (OAuthModule) module;
        OAuth oauth = module.getAnnotation(OAuth.class);
        muleContextField(oauthAdapter);

        // logger field
        GeneratedField logger = generateLoggerField(oauthAdapter);

        GeneratedField requestToken = requestTokenField(oauthAdapter);
        GeneratedField requestTokenSecret = requestTokenSecretField(oauthAdapter);
        GeneratedField oauthVerifier = authorizationCodeField(oauthAdapter);
        GeneratedField saveAccessTokenCallback = saveAccessTokenCallbackField(oauthAdapter);
        GeneratedField restoreAccessTokenCallback = restoreAccessTokenCallbackField(oauthAdapter);
        consumerField(oauthAdapter);

        GeneratedField authorizationUrl = generateAuthorizationUrlField(oauthAdapter);
        GeneratedField accessTokenUrl = generateAccessTokenUrlField(oauthAdapter);
        GeneratedField requestTokenUrl = generateRequestTokenUrlField(oauthAdapter);

        GeneratedMethod createConsumer = generateCreateConsumerMethod(oauthAdapter, oauth, module);
        generateGetConsumerMethod(oauthAdapter);

        generateInitialiseMethod(oauthAdapter, createConsumer);

        generateOAuth1AuthorizeMethod(oauthAdapter, requestToken, requestTokenSecret, module, logger);
        generateRestoreAccessTokenMethod(oAuthModule, oauthAdapter, restoreAccessTokenCallback, logger);
        generateFetchAccessTokenMethod(oauthAdapter, requestToken, requestTokenSecret, saveAccessTokenCallback, oauthVerifier, oAuthModule, oauth, logger);

        generateHasBeenAuthorizedMethod(oAuthModule, oauthAdapter);

        generateGetProcessTemplateMethod(oAuthModule, oauthAdapter, ctx().<GeneratedClass>getProduct(Product.CAPABILITIES_ADAPTER, module));

        generateResetMethod(oAuthModule, oauthAdapter);
    }

    private GeneratedField requestTokenField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(REQUEST_TOKEN_FIELD_NAME).build();
    }

    private GeneratedField requestTokenSecretField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(REQUEST_TOKEN_SECRET_FIELD_NAME).build();
    }

    private GeneratedField consumerField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(OAuthConsumer.class).name(CONSUMER_FIELD_NAME).build();
    }

    private void generateGetConsumerMethod(GeneratedClass oauthAdapter) {
        GeneratedMethod getConsumer = oauthAdapter.method(Modifier.PUBLIC, ref(OAuthConsumer.class), "getConsumer");
        getConsumer.body()._return(oauthAdapter.fields().get(CONSUMER_FIELD_NAME));
    }

    private void generateInitialiseMethod(GeneratedClass oauthAdapter, GeneratedMethod createConsumer) {
        GeneratedMethod initialise = generateInitialiseMethod(oauthAdapter);
        initialise._throws(ref(InitialisationException.class));
        initialise.body().invoke(createConsumer);
    }


    private void generateRestoreAccessTokenMethod(OAuthModule module, GeneratedClass oauthAdapter, GeneratedField restoreAccessTokenCallbackField, GeneratedField logger) {
        GeneratedMethod restoreAccessTokenMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "restoreAccessToken");

        GeneratedConditional ifRestoreCallbackNotNull = restoreAccessTokenMethod.body()._if(restoreAccessTokenCallbackField.isNotNull());

        GeneratedConditional ifDebugEnabled = ifRestoreCallbackNotNull._then()._if(logger.invoke("isDebugEnabled"));
        GeneratedVariable messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Attempting to restore access token..."));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        GeneratedTry tryToRestore = ifRestoreCallbackNotNull._then()._try();
        tryToRestore.body().add(restoreAccessTokenCallbackField.invoke("restoreAccessToken"));

        tryToRestore.body().invoke(module.getAccessTokenField().getSetter().getName()).arg(restoreAccessTokenCallbackField.invoke("getAccessToken"));
        tryToRestore.body().invoke(module.getAccessTokenSecretField().getSetter().getName()).arg(restoreAccessTokenCallbackField.invoke("getAccessTokenSecret"));
        GeneratedField consumer = oauthAdapter.fields().get(CONSUMER_FIELD_NAME);
        tryToRestore.body().add(consumer.invoke("setTokenWithSecret").arg(restoreAccessTokenCallbackField.invoke("getAccessToken")).arg(restoreAccessTokenCallbackField.invoke("getAccessTokenSecret")));

        ifDebugEnabled = tryToRestore.body()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Access token and secret has been restored successfully "));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessToken = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(restoreAccessTokenCallbackField.invoke("getAccessToken")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessTokenSecret = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(restoreAccessTokenCallbackField.invoke("getAccessTokenSecret")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        tryToRestore.body()._return(ExpressionFactory.TRUE);

        GeneratedCatchBlock logIfCannotRestore = tryToRestore._catch(ref(Exception.class));
        GeneratedVariable e = logIfCannotRestore.param("e");
        logIfCannotRestore.body().add(logger.invoke("error").arg("Cannot restore access token, an unexpected error occurred").arg(e));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        restoreAccessTokenMethod.body()._return(ExpressionFactory.FALSE);
    }

    private void generateFetchAccessTokenMethod(GeneratedClass oauthAdapter, GeneratedField requestToken, GeneratedField requestTokenSecret, GeneratedField saveAccessTokenCallback, GeneratedField oauthVerifier, OAuthModule module, OAuth oauth, GeneratedField logger) {
        GeneratedMethod fetchAccessToken = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.FETCH_ACCESS_TOKEN_METHOD_NAME);
        GeneratedVariable requestTokenUrl = fetchAccessToken.param(ref(String.class), "requestTokenUrl");
        GeneratedVariable accessTokenUrl = fetchAccessToken.param(ref(String.class), "accessTokenUrl");
        GeneratedVariable authorizationUrl = fetchAccessToken.param(ref(String.class), "authorizationUrl");
        GeneratedVariable redirectUri = fetchAccessToken.param(ref(String.class), "redirectUri");
        fetchAccessToken._throws((TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));

        fetchAccessToken.body().invoke("restoreAccessToken");

        GeneratedConditional ifAccessTokenNullOrSecretNull = fetchAccessToken.body()._if(Op.cor(
                ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName()).isNull(),
                ExpressionFactory.invoke(module.getAccessTokenSecretField().getGetter().getName()).isNull()));

        GeneratedVariable provider = generateProvider(requestTokenUrl, accessTokenUrl, authorizationUrl, ifAccessTokenNullOrSecretNull._then(), module);
        GeneratedField consumer = oauthAdapter.fields().get(CONSUMER_FIELD_NAME);
        ifAccessTokenNullOrSecretNull._then().invoke(consumer, "setTokenWithSecret").arg(requestToken).arg(requestTokenSecret);
        GeneratedTry tryRetrieveAccessToken = ifAccessTokenNullOrSecretNull._then()._try();

        GeneratedConditional ifDebugEnabled = tryRetrieveAccessToken.body()._if(logger.invoke("isDebugEnabled"));
        GeneratedVariable messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Retrieving access token..."));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        tryRetrieveAccessToken.body().invoke(provider, "retrieveAccessToken").arg(consumer).arg(oauthVerifier);
        generateCatchAndReThrow(tryRetrieveAccessToken, OAuthMessageSignerException.class, (TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));
        generateCatchAndReThrow(tryRetrieveAccessToken, OAuthNotAuthorizedException.class, (TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));
        generateCatchAndReThrow(tryRetrieveAccessToken, OAuthExpectationFailedException.class, (TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));
        generateCatchAndReThrow(tryRetrieveAccessToken, OAuthCommunicationException.class, (TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));

        ifAccessTokenNullOrSecretNull._then().invoke(module.getAccessTokenField().getSetter().getName()).arg(consumer.invoke("getToken"));
        ifAccessTokenNullOrSecretNull._then().invoke(module.getAccessTokenSecretField().getSetter().getName()).arg(consumer.invoke("getTokenSecret"));

        ifDebugEnabled = ifAccessTokenNullOrSecretNull._then()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Access token retrieved successfully "));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessToken = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName())));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessTokenSecret = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.invoke(module.getAccessTokenSecretField().getGetter().getName())));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));

        GeneratedConditional ifSaveCallbackNotNull = ifAccessTokenNullOrSecretNull._then()._if(saveAccessTokenCallback.isNotNull());
        GeneratedInvocation saveAccessToken = saveAccessTokenCallback.invoke("saveAccessToken").arg(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName())).arg(ExpressionFactory.invoke(module.getAccessTokenSecretField().getGetter().getName()));
        GeneratedTry tryToSave = ifSaveCallbackNotNull._then()._try();

        ifDebugEnabled = ifSaveCallbackNotNull._then()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Attempting to save access token..."));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessToken = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName())));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessTokenSecret = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.invoke(module.getAccessTokenSecretField().getGetter().getName())));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        tryToSave.body().add(saveAccessToken);
        GeneratedCatchBlock logIfCannotSave = tryToSave._catch(ref(Exception.class));
        GeneratedVariable e2 = logIfCannotSave.param("e");
        logIfCannotSave.body().add(logger.invoke("error").arg("Cannot save access token, an unexpected error occurred").arg(e2));
    }

    private void generateResetMethod(OAuthModule module, GeneratedClass oauthAdapter) {
        GeneratedMethod reset = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.RESET_METHOD_NAME);
        reset.body().invoke(module.getAccessTokenField().getSetter().getName()).arg(ExpressionFactory._null());
        reset.body().invoke(module.getAccessTokenSecretField().getSetter().getName()).arg(ExpressionFactory._null());
        GeneratedField consumer = oauthAdapter.fields().get(CONSUMER_FIELD_NAME);
        reset.body().add(consumer.invoke("setTokenWithSecret").arg(ExpressionFactory._null()).arg(ExpressionFactory._null()));
    }

    protected void generateGetProcessTemplateMethod(OAuthModule module, GeneratedClass oauthAdapterClass, GeneratedClass capabilitiesAdapterClass) {
        GeneratedClass oauthProcessTemplateClass = ctx().getProduct(Product.OAUTH_PROCESS_TEMPLATE);
        GeneratedMethod getProcessTemplate = oauthAdapterClass.method(Modifier.PUBLIC, (TypeReference) ctx().getProduct(Product.PROCESS_TEMPLATE_INTERFACE), "getProcessTemplate");
        getProcessTemplate.annotate(ref(Override.class));
        TypeVariable p = getProcessTemplate.generify("P");
        getProcessTemplate.type(((TypeReference) ctx().getProduct(Product.PROCESS_TEMPLATE_INTERFACE)).narrow(p).narrow(capabilitiesAdapterClass));


        GeneratedInvocation newProcessTemplate = ExpressionFactory._new(oauthProcessTemplateClass).arg(ExpressionFactory._this());
        getProcessTemplate.body()._return(newProcessTemplate);
    }
}