/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.oauth;

import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.mule.api.annotations.oauth.OAuthConsumerKey;
import org.mule.api.annotations.oauth.OAuthConsumerSecret;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.util.IOUtils;

public class OAuth2ClientAdapterGenerator extends AbstractOAuthAdapterGenerator {

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule && ((OAuthModule) module).getOAuthVersion() == OAuthVersion.V2;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass oauthAdapter = getOAuthAdapterClass(module, "OAuth2Adapter", (TypeReference) ctx().getProduct(Product.OAUTH2_ADAPTER_INTERFACE));
        OAuthModule oAuthModule = (OAuthModule) module;

        accessTokenPatternConstant(oauthAdapter, oAuthModule);
        refreshTokenPatternConstant(oauthAdapter, oAuthModule);

        expirationPatternConstant(oauthAdapter, oAuthModule);

        muleContextField(oauthAdapter);

        authorizationCodeField(oauthAdapter);
        refreshTokenField(oauthAdapter);

        GeneratedField saveAccessTokenCallback = saveAccessTokenCallbackField(oauthAdapter);
        GeneratedField restoreAccessTokenCallback = restoreAccessTokenCallbackField(oauthAdapter);

        GeneratedField redirectUri = oauthAdapter.field(Modifier.PUBLIC, ref(String.class), "redirectUri");
        GeneratedField authorizationUrl = generateAuthorizationUrlField(oauthAdapter);
        GeneratedField accessTokenUrl = generateAccessTokenUrlField(oauthAdapter);

        expirationField(oauthAdapter, oAuthModule);

        generateInitialiseMethod(oauthAdapter);

        GeneratedField logger = FieldBuilder.newLoggerField(oauthAdapter);
        generateOAuth2AuthorizeMethod(oauthAdapter, oAuthModule, logger, authorizationUrl);
        generateRestoreAccessTokenMethod(oAuthModule, oauthAdapter, restoreAccessTokenCallback, logger);
        generateFetchAccessTokenMethod(oauthAdapter, oAuthModule);
        generateRefreshAccessTokenMethod(oauthAdapter, oAuthModule, logger);
        generateFetchAndExtractMethod(oauthAdapter, oAuthModule, saveAccessTokenCallback, logger);
        generateHasTokenExpiredMethod(oAuthModule, oauthAdapter);
        generateResetMethod(oAuthModule, oauthAdapter);
        generateHasBeenAuthorizedMethod(oAuthModule, oauthAdapter);

        generateGetProcessTemplateMethod(oauthAdapter, ctx().<GeneratedClass>getProduct(Product.CAPABILITIES_ADAPTER, module));
    }

    protected GeneratedField refreshTokenField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(REFRESH_TOKEN_FIELD_NAME).getterAndSetter().build();
    }

    private void accessTokenPatternConstant(GeneratedClass oauthAdapter, OAuthModule module) {
        new FieldBuilder(oauthAdapter).type(Pattern.class).name(AbstractOAuthAdapterGenerator.ACCESS_CODE_PATTERN_FIELD_NAME).staticField().finalField().
                initialValue(ref(Pattern.class).staticInvoke("compile").arg(module.getAccessTokenRegex())).build();
    }

    private void refreshTokenPatternConstant(GeneratedClass oauthAdapter, OAuthModule module) {
        new FieldBuilder(oauthAdapter).type(Pattern.class).name(AbstractOAuthAdapterGenerator.REFRESH_TOKEN_PATTERN_FIELD_NAME).staticField().finalField().
                initialValue(ref(Pattern.class).staticInvoke("compile").arg(module.getRefreshTokenRegex())).build();
    }

    private void expirationPatternConstant(GeneratedClass oauthAdapter, OAuthModule module) {
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            new FieldBuilder(oauthAdapter).type(Pattern.class).name(AbstractOAuthAdapterGenerator.EXPIRATION_TIME_PATTERN_FIELD_NAME).staticField().finalField().
                    initialValue(ref(Pattern.class).staticInvoke("compile").arg(module.getExpirationRegex())).build();
        }
    }

    private void expirationField(GeneratedClass oauthAdapter, OAuthModule module) {
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            new FieldBuilder(oauthAdapter).type(Date.class).name(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME).setter().build();
        }
    }

    private void generateRestoreAccessTokenMethod(OAuthModule module, GeneratedClass oauthAdapter, GeneratedField restoreAccessTokenCallbackField, GeneratedField logger) {
        GeneratedMethod restoreAccessTokenMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "restoreAccessToken");
        GeneratedConditional ifRestoreCallbackNotNull = restoreAccessTokenMethod.body()._if(restoreAccessTokenCallbackField.isNotNull());

        GeneratedConditional ifDebugEnabled = ifRestoreCallbackNotNull._then()._if(logger.invoke("isDebugEnabled"));
        GeneratedVariable messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Attempting to restore access token..."));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        GeneratedTry tryToRestore = ifRestoreCallbackNotNull._then()._try();
        tryToRestore.body().add(restoreAccessTokenCallbackField.invoke("restoreAccessToken"));

        tryToRestore.body().invoke(module.getAccessTokenField().getSetter().getName()).arg(restoreAccessTokenCallbackField.invoke("getAccessToken"));

        ifDebugEnabled = tryToRestore.body()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Access token and secret has been restored successfully "));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessToken = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(restoreAccessTokenCallbackField.invoke("getAccessToken")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        tryToRestore.body()._return(ExpressionFactory.TRUE);

        GeneratedCatchBlock logIfCannotRestore = tryToRestore._catch(ref(Exception.class));
        GeneratedVariable e = logIfCannotRestore.param("e");
        logIfCannotRestore.body().add(logger.invoke("error").arg("Cannot restore access token, an unexpected error occurred").arg(e));

        restoreAccessTokenMethod.body()._return(ExpressionFactory.FALSE);
    }

    private void generateRefreshAccessTokenMethod(GeneratedClass oauthAdapter, OAuthModule module, GeneratedField logger) {
        GeneratedMethod refreshAccessTokenMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "refreshAccessToken");
        GeneratedVariable accessTokenUrl = refreshAccessTokenMethod.param(ref(String.class), "accessTokenUrl");
        refreshAccessTokenMethod._throws((TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));

        GeneratedBlock body = refreshAccessTokenMethod.body();
        GeneratedConditional ifDebugEnabled = body._if(logger.invoke("isDebugEnabled"));
        ifDebugEnabled._then().invoke(logger, "debug").arg("Trying to refresh access token...");
        
        GeneratedConditional ifRefreshTokenIsNull = body._if(ExpressionFactory._this().ref(REFRESH_TOKEN_FIELD_NAME).isNull());
        ifRefreshTokenIsNull._then()._throw(ExpressionFactory._new(
                ref(IllegalStateException.class)).arg("Cannot refresh access token since refresh token is null"));

        GeneratedVariable builder = body.decl(ref(StringBuilder.class), "builder", ExpressionFactory._new(ref(StringBuilder.class)));

        GeneratedInvocation consumerKey = ExpressionFactory.invoke(getterMethodForFieldAnnotatedWith(module, OAuthConsumerKey.class));
        GeneratedInvocation consumerSecret = ExpressionFactory.invoke(getterMethodForFieldAnnotatedWith(module, OAuthConsumerSecret.class));

        //GeneratedTry tryBlock = body._try();
        //GeneratedBlock tryBody = tryBlock.body();

        body.invoke(builder, "append").arg("grant_type=refresh_token");

        body.invoke(builder, "append").arg("&client_id=");
        body.invoke(builder, "append").arg(consumerKey);

        body.invoke(builder, "append").arg("&client_secret=");
        body.invoke(builder, "append").arg(consumerSecret);

        body.invoke(builder, "append").arg("&refresh_token=");
        body.invoke(builder, "append").arg(oauthAdapter.fields().get(REFRESH_TOKEN_FIELD_NAME));

        if (module.getScopeField() != null) {
            body.invoke(builder, "append").arg("&scope=");
            
            GeneratedTry urlEncodeTry = body._try();
            urlEncodeTry.body().invoke(builder, "append").arg(ref(URLEncoder.class).staticInvoke("encode").arg(ExpressionFactory.invoke(module.getScopeField().getGetter().getName())).arg(AbstractOAuthAdapterGenerator.ENCODING));
            this.generateCatchAndReThrow(urlEncodeTry, UnsupportedEncodingException.class, RuntimeException.class);
        }

        //generateCatchAndReThrow(tryBlock, RuntimeException.class);
        body.invoke("setAccessToken").arg(ExpressionFactory._null());
        body.invoke("fetchAndExtract").arg(accessTokenUrl).arg(builder.invoke("toString"));
    }

    private void generateFetchAccessTokenMethod(GeneratedClass oauthAdapter, OAuthModule module) {
        GeneratedMethod fetchAccessToken = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "fetchAccessToken");
        GeneratedVariable accessTokenUrl = fetchAccessToken.param(ref(String.class), "accessTokenUrl");
        GeneratedVariable redirectUri = fetchAccessToken.param(ref(String.class), "redirectUri");
        fetchAccessToken._throws((TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));

        GeneratedInvocation consumerKey = ExpressionFactory.invoke(getterMethodForFieldAnnotatedWith(module, OAuthConsumerKey.class));
        GeneratedInvocation consumerSecret = ExpressionFactory.invoke(getterMethodForFieldAnnotatedWith(module, OAuthConsumerSecret.class));

        GeneratedVariable builder = fetchAccessToken.body().decl(ref(StringBuilder.class), "builder", ExpressionFactory._new(ref(StringBuilder.class)));

        GeneratedTry tryBlock = fetchAccessToken.body()._try();
        GeneratedBlock tryBody = tryBlock.body();

        tryBody.invoke(builder, "append").arg("code=");
        tryBody.invoke(builder, "append").arg(ref(URLEncoder.class).staticInvoke("encode").arg(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.VERIFIER_FIELD_NAME)).arg(AbstractOAuthAdapterGenerator.ENCODING));

        tryBody.invoke(builder, "append").arg("&client_id=");
        tryBody.invoke(builder, "append").arg(ref(URLEncoder.class).staticInvoke("encode").arg(consumerKey).arg(AbstractOAuthAdapterGenerator.ENCODING));

        tryBody.invoke(builder, "append").arg("&client_secret=");
        tryBody.invoke(builder, "append").arg(ref(URLEncoder.class).staticInvoke("encode").arg(consumerSecret).arg(AbstractOAuthAdapterGenerator.ENCODING));

        tryBody.invoke(builder, "append").arg("&grant_type=");
        tryBody.invoke(builder, "append").arg(ref(URLEncoder.class).staticInvoke("encode").arg(AbstractOAuthAdapterGenerator.GRANT_TYPE).arg(AbstractOAuthAdapterGenerator.ENCODING));

        tryBody.invoke(builder, "append").arg("&redirect_uri=");
        tryBody.invoke(builder, "append").arg(ref(URLEncoder.class).staticInvoke("encode").arg(redirectUri));

        generateCatchAndReThrow(tryBlock, UnsupportedEncodingException.class, RuntimeException.class);
        fetchAccessToken.body().invoke("fetchAndExtract").arg(accessTokenUrl).arg(builder.invoke("toString"));
    }

    private void generateFetchAndExtractMethod(GeneratedClass oauthAdapter, OAuthModule module, GeneratedField saveAccessTokenCallback, GeneratedField logger) {
        GeneratedMethod fetchAccessToken = oauthAdapter.method(Modifier.PRIVATE, ctx().getCodeModel().VOID, "fetchAndExtract");
        GeneratedVariable accessTokenUrl = fetchAccessToken.param(ref(String.class), "accessTokenUrl");
        GeneratedVariable requestBodyParam = fetchAccessToken.param(ref(String.class), "requestBodyParam");
        fetchAccessToken._throws((TypeReference) ctx().getProduct(Product.UNABLE_TO_ACQUIRE_ACCESS_TOKEN_EXCEPTION));

        fetchAccessToken.body().invoke("restoreAccessToken");

        GeneratedConditional ifAccessTokenNull = fetchAccessToken.body()._if(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName()).isNull());

        GeneratedTry tryStatement = ifAccessTokenNull._then()._try();

        GeneratedConditional ifDebugEnabled = tryStatement.body()._if(logger.invoke("isDebugEnabled"));
        GeneratedVariable messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Retrieving access token..."));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        GeneratedBlock body = tryStatement.body();
        GeneratedVariable conn = body.decl(ref(HttpURLConnection.class), "conn",
                ExpressionFactory.cast(ref(HttpURLConnection.class), ExpressionFactory._new(ref(URL.class)).arg(accessTokenUrl).invoke("openConnection")));

        body.invoke(conn, "setRequestMethod").arg("POST");
        body.invoke(conn, "setDoOutput").arg(ExpressionFactory.lit(true));

        ifDebugEnabled = tryStatement.body()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Sending request to ["));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(accessTokenUrl));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("] using the following as content ["));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(requestBodyParam));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("]"));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        GeneratedVariable out = body.decl(ref(OutputStreamWriter.class), "out", ExpressionFactory._new(ref(OutputStreamWriter.class)).arg(conn.invoke("getOutputStream")));
        body.invoke(out, "write").arg(requestBodyParam);
        body.invoke(out, "close");

        GeneratedVariable response = body.decl(ref(String.class), "response", ref(IOUtils.class).staticInvoke("toString").arg(conn.invoke("getInputStream")));

        ifDebugEnabled = tryStatement.body()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Received response ["));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(response));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("]"));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        GeneratedVariable matcher = body.decl(ref(Matcher.class), "matcher", oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.ACCESS_CODE_PATTERN_FIELD_NAME).invoke("matcher").arg(response));
        GeneratedConditional ifAccessTokenFound = body._if(Op.cand(matcher.invoke("find"), Op.gte(matcher.invoke("groupCount"), ExpressionFactory.lit(1))));
        GeneratedInvocation group = matcher.invoke("group").arg(ExpressionFactory.lit(1));
        ifAccessTokenFound._then().invoke(module.getAccessTokenField().getSetter().getName()).arg(ref(URLDecoder.class).staticInvoke("decode").arg(group).arg(AbstractOAuthAdapterGenerator.ENCODING));

        ifDebugEnabled = ifAccessTokenFound._then()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Access token retrieved successfully "));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessToken = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName())));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        ifAccessTokenFound._else()._throw(ExpressionFactory._new(
                ref(Exception.class)).arg(ref(String.class).staticInvoke("format").arg("OAuth access token could not be extracted from: %s").arg(response)));

        GeneratedConditional ifSaveCallbackNotNull = ifAccessTokenFound._then()._if(saveAccessTokenCallback.isNotNull());
        GeneratedInvocation saveAccessToken = saveAccessTokenCallback.invoke("saveAccessToken").arg(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName()))
                .arg(ExpressionFactory._null());
        GeneratedTry tryToSave = ifSaveCallbackNotNull._then()._try();

        ifDebugEnabled = ifSaveCallbackNotNull._then()._if(logger.invoke("isDebugEnabled"));
        messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Attempting to save access token..."));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[accessToken = ")));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName())));
        ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
        ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

        tryToSave.body().add(saveAccessToken);
        GeneratedCatchBlock logIfCannotSave = tryToSave._catch(ref(Exception.class));
        GeneratedVariable e2 = logIfCannotSave.param("e");
        logIfCannotSave.body().add(logger.invoke("error").arg("Cannot save access token, an unexpected error occurred").arg(e2));

        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            ifDebugEnabled = ifAccessTokenFound._then()._if(logger.invoke("isDebugEnabled"));
            messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Attempting to extract expiration time using "));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[expirationPattern = ")));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(module.getExpirationRegex()));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
            ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

            GeneratedVariable expirationMatcher = ifAccessTokenFound._then().decl(ref(Matcher.class), "expirationMatcher", oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.EXPIRATION_TIME_PATTERN_FIELD_NAME).invoke("matcher").arg(response));
            GeneratedConditional ifExpirationFound = ifAccessTokenFound._then()._if(Op.cand(expirationMatcher.invoke("find"), Op.gte(expirationMatcher.invoke("groupCount"), ExpressionFactory.lit(1))));
            GeneratedVariable seconds = ifExpirationFound._then().decl(ref(Long.class), "expirationSecsAhead",
                    ref(Long.class).staticInvoke("parseLong").arg(expirationMatcher.invoke("group").arg(ExpressionFactory.lit(1))));
            ifExpirationFound._then().assign(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME), ExpressionFactory._new(ref(Date.class)).arg(
                    Op.plus(ref(System.class).staticInvoke("currentTimeMillis"), Op.mul(seconds, ExpressionFactory.lit(1000)))));

            ifDebugEnabled = ifExpirationFound._then()._if(logger.invoke("isDebugEnabled"));
            messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Token expiration extracted successfully "));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[expiration = ")));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
            ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

            ifDebugEnabled = ifExpirationFound._else()._if(logger.invoke("isDebugEnabled"));
            messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Token expiration could not be extracted from "));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[response = ")));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(response));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
            ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));
        }

        if (!StringUtils.isEmpty(module.getRefreshTokenRegex())) {
            ifDebugEnabled = ifAccessTokenFound._then()._if(logger.invoke("isDebugEnabled"));
            messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Attempting to extract refresh token time using "));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[refreshTokenPattern = ")));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(module.getRefreshTokenRegex()));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
            ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

            GeneratedVariable expirationMatcher = ifAccessTokenFound._then().decl(ref(Matcher.class), "refreshTokenMatcher", oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.REFRESH_TOKEN_PATTERN_FIELD_NAME).invoke("matcher").arg(response));
            GeneratedConditional ifRefreshTokenFound = ifAccessTokenFound._then()._if(Op.cand(expirationMatcher.invoke("find"), Op.gte(expirationMatcher.invoke("groupCount"), ExpressionFactory.lit(1))));

            ifRefreshTokenFound._then().assign(oauthAdapter.fields().get(REFRESH_TOKEN_FIELD_NAME), expirationMatcher.invoke("group").arg(ExpressionFactory.lit(1)));

            ifDebugEnabled = ifRefreshTokenFound._then()._if(logger.invoke("isDebugEnabled"));
            messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Refresh token extracted successfully "));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[refresh token = ")));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.REFRESH_TOKEN_FIELD_NAME)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
            ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));

            ifDebugEnabled = ifRefreshTokenFound._else()._if(logger.invoke("isDebugEnabled"));
            messageStringBuilder = ifDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg("Refresh token could not be extracted from "));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("[response = ")));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(response));
            ifDebugEnabled._then().add(messageStringBuilder.invoke("append").arg(ExpressionFactory.lit("] ")));
            ifDebugEnabled._then().add(logger.invoke("debug").arg(messageStringBuilder.invoke("toString")));
        }

        if (module.getCallbackParameters() != null && !module.getCallbackParameters().isEmpty()) {
            generateFetchCallbackParameters(oauthAdapter, module);

            ifAccessTokenFound._then().invoke("fetchCallbackParameters").arg(response);
        }

        if (module.getPostAuthorizationMethod() != null) {
            ifAccessTokenFound._then().invoke(module.getPostAuthorizationMethod().getName());
        }

        generateCatchAndReThrow(tryStatement, Exception.class, RuntimeException.class);
    }

    private void generateHasTokenExpiredMethod(OAuthModule module, GeneratedClass oauthAdapter) {
        GeneratedMethod hasTokenExpired = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, AbstractOAuthAdapterGenerator.HAS_TOKEN_EXPIRED_METHOD_NAME);
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            GeneratedField expirationDate = oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME);
            hasTokenExpired.body()._return(Op.cand(
                    expirationDate.isNotNull(),
                    expirationDate.invoke("before").arg(ExpressionFactory._new(ref(Date.class)))));
        } else {
            hasTokenExpired.body()._return(ExpressionFactory.FALSE);
        }
    }

    private void generateResetMethod(OAuthModule module, GeneratedClass oauthAdapter) {
        GeneratedMethod reset = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.RESET_METHOD_NAME);
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            reset.body().assign(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME), ExpressionFactory._null());
        }
        reset.body().assign(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.VERIFIER_FIELD_NAME), ExpressionFactory._null());
        reset.body().invoke(module.getAccessTokenField().getSetter().getName()).arg(ExpressionFactory._null());
    }

    protected void generateGetProcessTemplateMethod(GeneratedClass oauthAdapterClass, GeneratedClass capabilitiesAdapterClass) {
        GeneratedClass oauthProcessTemplateClass = ctx().getProduct(Product.OAUTH_PROCESS_TEMPLATE);
        GeneratedMethod getProcessTemplate = oauthAdapterClass.method(Modifier.PUBLIC, (TypeReference) ctx().getProduct(Product.PROCESS_TEMPLATE_INTERFACE), "getProcessTemplate");
        getProcessTemplate.annotate(ref(Override.class));
        TypeVariable p = getProcessTemplate.generify("P");
        getProcessTemplate.type(((TypeReference) ctx().getProduct(Product.PROCESS_TEMPLATE_INTERFACE)).narrow(p).narrow(capabilitiesAdapterClass));


        GeneratedInvocation newProcessTemplate = ExpressionFactory._new(oauthProcessTemplateClass).arg(ExpressionFactory._this());
        getProcessTemplate.body()._return(newProcessTemplate);
    }
}