/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.oauth;

import oauth.signpost.OAuthConsumer;
import oauth.signpost.basic.DefaultOAuthConsumer;
import oauth.signpost.signature.AuthorizationHeaderSigningStrategy;
import oauth.signpost.signature.HmacSha1MessageSigner;
import oauth.signpost.signature.PlainTextMessageSigner;
import oauth.signpost.signature.QueryStringSigningStrategy;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.pool.KeyedPoolableObjectFactory;
import org.apache.commons.pool.impl.GenericKeyedObjectPool;
import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.annotations.oauth.OAuth;
import org.mule.api.config.MuleProperties;
import org.mule.api.construct.FlowConstruct;
import org.mule.api.construct.FlowConstructAware;
import org.mule.api.context.MuleContextAware;
import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.lifecycle.Startable;
import org.mule.api.lifecycle.Stoppable;
import org.mule.api.store.ObjectStore;
import org.mule.api.store.ObjectStoreException;
import org.mule.config.i18n.CoreMessages;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.ModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedCast;
import org.mule.devkit.model.code.GeneratedCatchBlock;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedConditional;
import org.mule.devkit.model.code.GeneratedField;
import org.mule.devkit.model.code.GeneratedFieldReference;
import org.mule.devkit.model.code.GeneratedInvocation;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedPackage;
import org.mule.devkit.model.code.GeneratedTry;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.Op;
import org.mule.devkit.model.code.TypeReference;
import org.mule.devkit.model.code.TypeVariable;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthCallbackParameterField;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;

import java.util.Arrays;
import java.util.List;

public class OAuthManagerGenerator extends AbstractOAuthAdapterGenerator implements ModuleGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(Product.CAPABILITIES_ADAPTER,
            Product.LIFECYCLE_ADAPTER,
            Product.OAUTH_ADAPTER,
            Product.INJECTION_ADAPTER,
            Product.HTTP_CALLBACK_ADAPTER,
            Product.OAUTH_ADAPTER,
            Product.MANAGED_ACCESS_TOKEN_PROCESS_TEMPLATE,
            Product.METADATA_ADAPTER,
            Product.PROCESS_INTERFACES,
            Product.BASICS_INTERFACE);
    private final static List<Product> PRODUCES = Arrays.asList(Product.OAUTH_MANAGER);


    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule && ((OAuthModule) module).getUserIdentifierMethod() != null;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        OAuthModule oauthModule = (OAuthModule) module;

        GeneratedClass oauthManagerClass = getOAuthManagerClass(oauthModule);

        // generate factory
        GeneratedClass oauthClientFactoryClass = getOAuthClientFactoryClass(module);
        GeneratedField oauthManagerInFactory = oauthClientFactoryClass.field(Modifier.PRIVATE,
                oauthManagerClass, "oauthManager");

        GeneratedField logger = generateLoggerField(oauthManagerClass);

        generateOAuthClientFactoryConstructor(oauthManagerClass, oauthClientFactoryClass, oauthManagerInFactory);
        generateMakeObjectMethod(module, oauthClientFactoryClass, oauthManagerInFactory);
        generateDestroyObjectMethod(module, oauthClientFactoryClass);
        generateValidateObjectMethod(module, oauthClientFactoryClass, oauthManagerInFactory);
        generateActivateObjectMethod(oauthClientFactoryClass);
        generatePassivateObjectMethod(module, oauthClientFactoryClass, oauthManagerInFactory);

        // generate manager
        GeneratedField defaultUnauthorizedConnector = generateDefaultUnauthorizedConnectorField(module, oauthManagerClass);

        // generate fields for each configurable field
        for (Field field : module.getConfigurableFields()) {
            GeneratedField configField = oauthManagerClass.field(Modifier.PRIVATE, ref(field.asTypeMirror()), field.getName());
            oauthManagerClass.setter(configField);
            oauthManagerClass.getter(configField);
        }

        // standard fields
        GeneratedField muleContext = generateMuleContextField(oauthManagerClass, defaultUnauthorizedConnector);
        GeneratedField flowConstruct = generateFlowConstructField(oauthManagerClass, defaultUnauthorizedConnector);
        GeneratedField accessTokenObjectStore = oauthManagerClass.field(Modifier.PRIVATE, ref(ObjectStore.class), "accessTokenObjectStore");
        oauthManagerClass.getter(accessTokenObjectStore);
        oauthManagerClass.setter(accessTokenObjectStore);

        // config urls
        GeneratedField authorizationUrl = generateAuthorizationUrlField(oauthManagerClass);
        GeneratedField accessTokenUrl = generateAccessTokenUrlField(oauthManagerClass);
        if( ((OAuthModule)module).getOAuthVersion() == OAuthVersion.V10A ) {
            GeneratedField requestTokenUrl = generateRequestTokenUrlField(oauthManagerClass);
        }

        // generate field for connection pool
        GeneratedField accessTokenPoolFactory = generateFieldForAccessTokenPoolFactory(oauthManagerClass);
        GeneratedField accessTokenPool = generateFieldForAccessTokenPool(oauthManagerClass);

        generateInitialiseMethod(oauthManagerClass, accessTokenPool, accessTokenPoolFactory, accessTokenObjectStore, oauthClientFactoryClass, defaultUnauthorizedConnector, oauthModule);
        generateStartMethod(oauthManagerClass, defaultUnauthorizedConnector);
        generateStopMethod(oauthManagerClass, defaultUnauthorizedConnector);
        generateDisposeMethod(oauthManagerClass, defaultUnauthorizedConnector);

        generateCreateAccessTokenMethod(module, oauthManagerClass);
        generateAcquireAccessTokenMethod(module, oauthManagerClass, accessTokenPool, logger);
        generateReleaseAccessTokenMethod(module, oauthManagerClass, accessTokenPool, logger);
        generateDestroyAccessTokenMethod(module, oauthManagerClass, accessTokenPool, logger);

        generateIsCapableOfMethod(module, oauthManagerClass);

        generateGetProcessTemplateMethod(oauthModule, oauthManagerClass, ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass());

        generateMetadataConstantsAndGetters(module, oauthManagerClass);

        if (oauthModule.getOAuthVersion() == OAuthVersion.V2) {
            generateOAuth2AuthorizeMethod(oauthManagerClass, oauthModule, logger, authorizationUrl);
        } else if(oauthModule.getOAuthVersion() == OAuthVersion.V10A) {
            consumerField(oauthManagerClass);
            generateCreateConsumerMethod(oauthManagerClass, module.getAnnotation(OAuth.class), module);
            generateOAuth1AuthorizeMethod(oauthManagerClass, null, null, oauthModule, logger);
        }

    }
    private GeneratedField consumerField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(OAuthConsumer.class).name(CONSUMER_FIELD_NAME).build();
    }

    private void generateStartMethod(GeneratedClass oauthManagerClass, GeneratedField defaultUnauthorizedConnector) {
        GeneratedMethod start = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "start");
        start._throws(ref(MuleException.class));
        start.body()._if(Op._instanceof(defaultUnauthorizedConnector, ref(Startable.class)))._then().add(ExpressionFactory.cast(ref(Startable.class), defaultUnauthorizedConnector).invoke("start"));
    }

    private void generateStopMethod(GeneratedClass oauthManagerClass, GeneratedField defaultUnauthorizedConnector) {
        GeneratedMethod stop = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "stop");
        stop._throws(ref(MuleException.class));
        stop.body()._if(Op._instanceof(defaultUnauthorizedConnector, ref(Stoppable.class)))._then().add(ExpressionFactory.cast(ref(Stoppable.class), defaultUnauthorizedConnector).invoke("stop"));
    }

    private void generateDisposeMethod(GeneratedClass oauthManagerClass, GeneratedField defaultUnauthorizedConnector) {
        GeneratedMethod dispose = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "dispose");
        dispose.body()._if(Op._instanceof(defaultUnauthorizedConnector, ref(Disposable.class)))._then().add(ExpressionFactory.cast(ref(Disposable.class), defaultUnauthorizedConnector).invoke("dispose"));
    }

    public GeneratedField generateFlowConstructField(GeneratedClass oauthManagerClass, GeneratedField defaultUnauthorizedConnector) {
        GeneratedField flowConstruct = oauthManagerClass.field(Modifier.PROTECTED, ctx().getCodeModel().ref(FlowConstruct.class), GeneratedClass.FLOW_CONSTRUCT_FIELD_NAME);
        flowConstruct.javadoc().add(GeneratedClass.FLOW_CONSTRUCT_JAVADOC);
        oauthManagerClass.getter(flowConstruct);

        GeneratedMethod setFlowConstruct = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setFlowConstruct");
        GeneratedVariable newFlowConstruct = setFlowConstruct.param(ref(FlowConstruct.class), "flowConstruct");
        setFlowConstruct.body().assign(ExpressionFactory._this().ref(flowConstruct), newFlowConstruct);
        setFlowConstruct.body()._if(Op._instanceof(defaultUnauthorizedConnector, ref(FlowConstructAware.class)))._then().add(ExpressionFactory.cast(ref(FlowConstructAware.class), defaultUnauthorizedConnector).invoke("setFlowConstruct").arg(newFlowConstruct));

        return flowConstruct;
    }

    public GeneratedField generateMuleContextField(GeneratedClass oauthManagerClass, GeneratedField defaultUnauthorizedConnector) {
        GeneratedField muleContext = oauthManagerClass.field(Modifier.PROTECTED, ctx().getCodeModel().ref(MuleContext.class), GeneratedClass.MULE_CONTEXT_FIELD_NAME);
        muleContext.javadoc().add(GeneratedClass.MULE_CONTEXT_FIELD_NAME);
        oauthManagerClass.getter(muleContext);

        GeneratedMethod setMuleContext = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setMuleContext");
        GeneratedVariable newMuleContext = setMuleContext.param(ref(MuleContext.class), "muleContext");
        setMuleContext.body().assign(ExpressionFactory._this().ref(muleContext), newMuleContext);
        setMuleContext.body()._if(Op._instanceof(defaultUnauthorizedConnector, ref(MuleContextAware.class)))._then().add(ExpressionFactory.cast(ref(MuleContextAware.class), defaultUnauthorizedConnector).invoke("setMuleContext").arg(newMuleContext));

        return muleContext;
    }

    private GeneratedField generateDefaultUnauthorizedConnectorField(Module module, GeneratedClass oauthManagerClass) {
        GeneratedClass oauthAdapterClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedField defaultUnauthorizedConnector = oauthManagerClass.field(Modifier.PRIVATE,
                oauthAdapterClass, "defaultUnauthorizedConnector");

        oauthManagerClass.getter(defaultUnauthorizedConnector);

        return defaultUnauthorizedConnector;
    }

    private void generateOAuthClientFactoryConstructor(GeneratedClass oauthManagerClass, GeneratedClass oauthClientFactoryClass, GeneratedField oauthManagerInFactory) {
        GeneratedMethod oauthClientFactoryConstructor = oauthClientFactoryClass.constructor(Modifier.PUBLIC);
        GeneratedVariable constructorOAuthManager = oauthClientFactoryConstructor.param(oauthManagerClass, "oauthManager");
        oauthClientFactoryConstructor.body().assign(ExpressionFactory._this().ref(oauthManagerInFactory),
                constructorOAuthManager);
    }

    private GeneratedClass getOAuthManagerClass(OAuthModule module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);
        GeneratedClass classToExtend = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();

        TypeReference previous = ctx().getProduct(Product.HTTP_CALLBACK_ADAPTER, module);

        GeneratedClass oauthManagerClass = pkg._class(module.getClassName() + OAuthClientNamingConstants.OAUTH_MANAGER_CLASS_NAME_SUFFIX, previous);
        oauthManagerClass._implements(ref(Initialisable.class));
        oauthManagerClass._implements((TypeReference)ctx().getProduct(Product.CAPABILITIES_INTERFACE));
        oauthManagerClass._implements(ref(MuleContextAware.class));
        oauthManagerClass._implements(((TypeReference) ctx().getProduct(Product.OAUTH_MANAGER_INTERFACE)).narrow(classToExtend));
        oauthManagerClass._implements(((TypeReference) ctx().getProduct(Product.PROCESS_ADAPTER_INTERFACE)).narrow(classToExtend));
        oauthManagerClass._implements(((TypeReference)ctx().getProduct(Product.METADATA_AWARE_INTERFACE)));

        ctx().registerProduct(Product.OAUTH_MANAGER, module, oauthManagerClass);

        oauthManagerClass.javadoc().add("A {@code " + oauthManagerClass.name() + "} is a wrapper around ");
        oauthManagerClass.javadoc().add(ref(module.asTypeMirror()));
        oauthManagerClass.javadoc().add(" that adds access token management capabilities to the pojo.");

        return oauthManagerClass;
    }

    private GeneratedField generateFieldForAccessTokenPool(GeneratedClass oauthManagerClass) {
        GeneratedField accessTokenPool = oauthManagerClass.field(Modifier.PRIVATE, ref(GenericKeyedObjectPool.class), "accessTokenPool");
        accessTokenPool.javadoc().add("Access Token Pool");

        return accessTokenPool;
    }

    private GeneratedField generateFieldForAccessTokenPoolFactory(GeneratedClass oauthManagerClass) {
        GeneratedField accessTokenPoolFactory = oauthManagerClass.field(Modifier.PRIVATE, ref(KeyedPoolableObjectFactory.class), "accessTokenPoolFactory");
        accessTokenPoolFactory.javadoc().add("Access Token Pool Factory");
        oauthManagerClass.getter(accessTokenPoolFactory);

        return accessTokenPoolFactory;
    }

    private void generateInitialiseMethod(GeneratedClass oauthManagerClass, GeneratedField accessTokenPool, GeneratedField accessTokenPoolFactory,  GeneratedField accessTokenObjectStore, GeneratedClass oauthClientFactoryClass, GeneratedField defaultUnauthorizedConnector, OAuthModule module) {
        GeneratedMethod initialisableMethod = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "initialise");
        initialisableMethod._throws(ref(InitialisationException.class));

        GeneratedVariable config = initialisableMethod.body().decl(ref(GenericKeyedObjectPool.Config.class), "config",
                ExpressionFactory._new(ref(GenericKeyedObjectPool.Config.class)));

        initialisableMethod.body().assign(config.ref("testOnBorrow"), ExpressionFactory.TRUE);

        if( module.getOAuthVersion() == OAuthVersion.V10A ) {
            initialisableMethod.body().add(ExpressionFactory.invoke("createConsumer"));
        }

        GeneratedConditional ifNotNull = initialisableMethod.body()._if(accessTokenObjectStore.isNull());

        ifNotNull._then().assign(accessTokenObjectStore, oauthManagerClass.muleContextField().invoke("getRegistry").invoke("lookupObject").arg(ref(MuleProperties.class).staticRef("DEFAULT_USER_OBJECT_STORE_NAME")));
        ifNotNull._then()._if(accessTokenObjectStore.isNull())._then()._throw(ExpressionFactory._new(ref(InitialisationException.class)).arg(
                ref(CoreMessages.class).staticInvoke("createStaticMessage").arg("There is no default user object store on this Mule instance.")
        ).arg(ExpressionFactory._this()));

        initialisableMethod.body().assign(accessTokenPoolFactory, ExpressionFactory._new(oauthClientFactoryClass).arg(ExpressionFactory._this()));
        initialisableMethod.body().assign(accessTokenPool, ExpressionFactory._new(ref(GenericKeyedObjectPool.class)).arg(
                accessTokenPoolFactory
        ).arg(config));

        initialisableMethod.body().assign(defaultUnauthorizedConnector, ExpressionFactory._new(defaultUnauthorizedConnector.type()));
        initialisableMethod.body()._if(Op._instanceof(defaultUnauthorizedConnector, ref(Initialisable.class)))._then().add(ExpressionFactory.cast(ref(Initialisable.class), defaultUnauthorizedConnector).invoke("initialise"));

    }


    private GeneratedClass getOAuthClientFactoryClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);
        GeneratedClass objectFactory = pkg._class(Modifier.PUBLIC, module.getClassName() + OAuthClientNamingConstants.OAUTH_CLIENT_FACTORY_CLASS_NAME_SUFFIX);
        objectFactory._implements(KeyedPoolableObjectFactory.class);
        return objectFactory;
    }

    private void generateMakeObjectMethod(Module module, GeneratedClass oauthClientFactoryClass, GeneratedField oauthManagerInFactory) {
        GeneratedClass connectorClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedClass stateClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_STATE, module);

        GeneratedMethod makeObject = oauthClientFactoryClass.method(Modifier.PUBLIC, Object.class, "makeObject");
        makeObject._throws(ref(Exception.class));
        GeneratedVariable key = makeObject.param(Object.class, "key");
        GeneratedConditional ifNotKey = makeObject.body()._if(Op.not(Op._instanceof(key, ref(String.class))));
        ifNotKey._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Invalid key type"));

        GeneratedVariable state = makeObject.body().decl(stateClass, "state", ExpressionFactory._null());
        makeObject.body()._if(Op.not(oauthManagerInFactory.invoke("getAccessTokenObjectStore").invoke("contains").arg(ExpressionFactory.cast(ref(String.class), key))))
                ._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(
                Op.plus(Op.plus(ExpressionFactory.lit("There is no access token stored under the key "), ExpressionFactory.cast(ref(String.class), key)), ExpressionFactory.lit(". You need to call the <authorize> message processor. The key will be given to you via a flow variable after the OAuth dance is completed. You can extract it using flowVars['tokenId'].")
                )));

        makeObject.body().assign(state, ExpressionFactory.cast(stateClass, oauthManagerInFactory.invoke("getAccessTokenObjectStore").invoke("retrieve").arg(ExpressionFactory.cast(ref(String.class), key))));

        GeneratedVariable connector = makeObject.body().decl(connectorClass, "connector", ExpressionFactory._new(connectorClass));
        for (Field field : module.getConfigurableFields()) {
            makeObject.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName()))
                    .arg(oauthManagerInFactory.invoke("get" + StringUtils.capitalize(field.getName()))));
        }

        makeObject.body().add(connector.invoke("setAccessToken").arg(
                state.invoke("getAccessToken")
        ));
        makeObject.body().add(connector.invoke("setAuthorizationUrl").arg(
                state.invoke("getAuthorizationUrl")
        ));
        makeObject.body().add(connector.invoke("setAccessTokenUrl").arg(
                state.invoke("getAccessTokenUrl")
        ));

        if (((OAuthModule) module).getOAuthVersion() == OAuthVersion.V10A) {
            makeObject.body().add(connector.invoke("setAccessTokenSecret").arg(
                    state.invoke("getAccessTokenSecret")
            ));
            makeObject.body().add(connector.invoke("setRequestTokenUrl").arg(
                    state.invoke("getRequestTokenUrl")
            ));
        }

        if (((OAuthModule) module).getOAuthVersion() == OAuthVersion.V2) {
            makeObject.body().add(connector.invoke("setRefreshToken").arg(
                    state.invoke("getRefreshToken")
            ));
        }

        for (OAuthCallbackParameterField field : ((OAuthModule) module).getCallbackParameters()) {
            makeObject.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName())).arg(
                    state.invoke("get" + StringUtils.capitalize(field.getName()))
            ));
        }

        makeObject.body()._if(Op._instanceof(connector, ref(Initialisable.class)))._then().add(ExpressionFactory.cast(ref(Initialisable.class), connector).invoke("initialise"));
        makeObject.body()._if(Op._instanceof(connector, ref(MuleContextAware.class)))._then().add(ExpressionFactory.cast(ref(MuleContextAware.class), connector).invoke("setMuleContext").arg(oauthManagerInFactory.invoke("getMuleContext")));
        makeObject.body()._if(Op._instanceof(connector, ref(Startable.class)))._then().add(ExpressionFactory.cast(ref(Startable.class), connector).invoke("start"));

        if( ((OAuthModule)module).getPostAuthorizationMethod() != null ) {
            makeObject.body().add(connector.invoke(((OAuthModule)module).getPostAuthorizationMethod().getName()));
        }

        makeObject.body()._return(connector);
    }

    private void generateDestroyObjectMethod(Module module, GeneratedClass oauthClientFactoryClass) {
        GeneratedClass oauthAdapterClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();

        GeneratedMethod destroyObject = oauthClientFactoryClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "destroyObject");
        destroyObject._throws(ref(Exception.class));
        GeneratedVariable key = destroyObject.param(Object.class, "key");
        GeneratedVariable obj = destroyObject.param(Object.class, "obj");
        GeneratedConditional ifNotKey = destroyObject.body()._if(Op.not(Op._instanceof(key, ref(String.class))));
        ifNotKey._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Invalid key type"));

        GeneratedConditional ifNotObj = destroyObject.body()._if(Op.not(Op._instanceof(obj, oauthAdapterClass)));
        ifNotObj._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Invalid connector type"));

        GeneratedCast casterConnector = ExpressionFactory.cast(oauthAdapterClass, obj);
        GeneratedTry tryDisconnect = destroyObject.body()._try();
        GeneratedCatchBlock catchAndRethrow = tryDisconnect._catch(ref(Exception.class));
        GeneratedVariable e = catchAndRethrow.param("e");
        catchAndRethrow.body()._throw(e);
        tryDisconnect._finally()._if(Op._instanceof(casterConnector, ref(Stoppable.class)))._then().add(ExpressionFactory.cast(ref(Stoppable.class), obj).invoke("stop"));
        tryDisconnect._finally()._if(Op._instanceof(casterConnector, ref(Disposable.class)))._then().add(ExpressionFactory.cast(ref(Disposable.class), obj).invoke("dispose"));
    }

    private void generateValidateObjectMethod(Module module, GeneratedClass oauthClientFactoryClass, GeneratedField oauthManagerInFactory) {
        GeneratedClass oauthAdapterClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedClass stateClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_STATE, module);

        GeneratedMethod validateObject = oauthClientFactoryClass.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "validateObject");
        GeneratedVariable key = validateObject.param(Object.class, "key");
        GeneratedVariable obj = validateObject.param(Object.class, "obj");

        GeneratedConditional ifNotKey = validateObject.body()._if(Op.not(Op._instanceof(key, ref(String.class))));
        ifNotKey._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Invalid key type"));

        GeneratedConditional ifNotObj = validateObject.body()._if(Op.not(Op._instanceof(obj, oauthAdapterClass)));
        ifNotObj._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Invalid connector type"));


        GeneratedVariable state = validateObject.body().decl(stateClass, "state", ExpressionFactory._null());
        GeneratedTry  attempt = validateObject.body()._try();
        attempt.body()._if(Op.not(oauthManagerInFactory.invoke("getAccessTokenObjectStore").invoke("contains").arg(ExpressionFactory.cast(ref(String.class), key))))
                ._then()._return(ExpressionFactory.FALSE);

        attempt.body().assign(state, ExpressionFactory.cast(stateClass, oauthManagerInFactory.invoke("getAccessTokenObjectStore").invoke("retrieve").arg(ExpressionFactory.cast(ref(String.class), key))));

        attempt.body()._if(ExpressionFactory.cast(oauthAdapterClass, obj).invoke("getAccessToken").isNull())._then()._return(ExpressionFactory.FALSE);
        attempt.body()._if(Op.not(ExpressionFactory.cast(oauthAdapterClass, obj).invoke("getAccessToken").invoke("equals").arg(state.invoke("getAccessToken"))))._then()._return(ExpressionFactory.FALSE);

        attempt.body()._if(Op.cand(ExpressionFactory.cast(oauthAdapterClass, obj).invoke("getRefreshToken").isNull(), state.invoke("getRefreshToken").isNotNull()))._then()._return(ExpressionFactory.FALSE);
        attempt.body()._if(Op.cand(ExpressionFactory.cast(oauthAdapterClass, obj).invoke("getRefreshToken").isNotNull(), Op.not(ExpressionFactory.cast(oauthAdapterClass, obj).invoke("getRefreshToken").invoke("equals").arg(state.invoke("getRefreshToken")))))._then()._return(ExpressionFactory.FALSE);


        attempt._catch(ref(ObjectStoreException.class)).body()._return(ExpressionFactory.FALSE);

        validateObject.body()._return(ExpressionFactory.TRUE);
    }

    private void generateActivateObjectMethod(GeneratedClass oauthClientFactoryClass) {
        GeneratedMethod activateObject = oauthClientFactoryClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "activateObject");
        activateObject._throws(ref(Exception.class));
        GeneratedVariable key = activateObject.param(Object.class, "key");
        GeneratedVariable obj = activateObject.param(Object.class, "obj");
    }

    private void generatePassivateObjectMethod(Module module, GeneratedClass oauthClientFactoryClass, GeneratedField oauthManager) {
        GeneratedMethod passivateObject = oauthClientFactoryClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "passivateObject");
        passivateObject._throws(ref(Exception.class));
        GeneratedVariable key = passivateObject.param(Object.class, "key");
        GeneratedVariable obj = passivateObject.param(Object.class, "obj");

        GeneratedConditional ifNotKey = passivateObject.body()._if(Op.not(Op._instanceof(key, ref(String.class))));
        ifNotKey._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Invalid key type"));

        GeneratedClass connectorClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedClass stateClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_STATE, module);
        GeneratedConditional ifNotObj = passivateObject.body()._if(Op.not(Op._instanceof(obj, connectorClass)));
        ifNotObj._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Invalid connector type"));

        GeneratedVariable state = passivateObject.body().decl(stateClass, "state", ExpressionFactory._null());
        GeneratedConditional ifStateExists = passivateObject.body()._if(oauthManager.invoke("getAccessTokenObjectStore").invoke("contains").arg(ExpressionFactory.cast(ref(String.class), key)));
        ifStateExists._then().assign(state, ExpressionFactory.cast(stateClass, oauthManager.invoke("getAccessTokenObjectStore").invoke("retrieve").arg(ExpressionFactory.cast(ref(String.class), key))));
        ifStateExists._then().add(oauthManager.invoke("getAccessTokenObjectStore").invoke("remove").arg(ExpressionFactory.cast(ref(String.class), key)));

        passivateObject.body()._if(state.isNull())._then().assign(state, ExpressionFactory._new(stateClass));

        passivateObject.body().add(state.invoke("setAccessToken").arg(
                ExpressionFactory.cast(connectorClass, obj).invoke("getAccessToken")
        ));
        passivateObject.body().add(state.invoke("setAccessTokenUrl").arg(
                ExpressionFactory.cast(connectorClass, obj).invoke("getAccessTokenUrl")
        ));
        passivateObject.body().add(state.invoke("setAuthorizationUrl").arg(
                ExpressionFactory.cast(connectorClass, obj).invoke("getAuthorizationUrl")
        ));

        if (((OAuthModule) module).getOAuthVersion() == OAuthVersion.V10A) {
            passivateObject.body().add(state.invoke("setAccessTokenSecret").arg(
                    ExpressionFactory.cast(connectorClass, obj).invoke("getAccessTokenSecret")
            ));
            passivateObject.body().add(state.invoke("setRequestTokenUrl").arg(
                    ExpressionFactory.cast(connectorClass, obj).invoke("getRequestTokenUrl")
            ));
        }

        if (((OAuthModule) module).getOAuthVersion() == OAuthVersion.V2) {
            passivateObject.body().add(state.invoke("setRefreshToken").arg(
                    ExpressionFactory.cast(connectorClass, obj).invoke("getRefreshToken")
            ));
        }

        for (OAuthCallbackParameterField field : ((OAuthModule) module).getCallbackParameters()) {
            passivateObject.body().add(state.invoke("set" + StringUtils.capitalize(field.getName())).arg(
                    ExpressionFactory.cast(connectorClass, obj).invoke("get" + StringUtils.capitalize(field.getName()))
            ));
        }

        passivateObject.body().add(oauthManager.invoke("getAccessTokenObjectStore").invoke("store").arg(ExpressionFactory.cast(ref(String.class), key)).arg(state));
    }

    private void generateAcquireAccessTokenMethod(Module module, GeneratedClass oauthManagerClass, GeneratedField accessTokenPool, GeneratedField logger) {
        GeneratedClass connectorClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedMethod acquireAccessTokenMethod = oauthManagerClass.method(Modifier.PUBLIC, connectorClass, "acquireAccessToken");
        GeneratedVariable userId = acquireAccessTokenMethod.param(ref(String.class), "userId");
        acquireAccessTokenMethod._throws(ref(Exception.class));

        GeneratedConditional isDebugEnabled = acquireAccessTokenMethod.body()._if(logger.invoke("isDebugEnabled"));
        GeneratedVariable mesasgeStringBuilder = isDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("Pool Statistics before acquiring [key "));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(userId));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [active="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumActive").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [idle="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumIdle").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("]"));
        isDebugEnabled._then().add(logger.invoke("debug").arg(mesasgeStringBuilder.invoke("toString")));

        GeneratedVariable object = acquireAccessTokenMethod.body().decl(connectorClass, "object",
                ExpressionFactory.cast(connectorClass,
                        accessTokenPool.invoke("borrowObject").arg(
                                userId
                        ))
        );

        isDebugEnabled = acquireAccessTokenMethod.body()._if(logger.invoke("isDebugEnabled"));
        mesasgeStringBuilder = isDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("Pool Statistics after acquiring [key "));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(userId));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [active="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumActive").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [idle="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumIdle").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("]"));
        isDebugEnabled._then().add(logger.invoke("debug").arg(mesasgeStringBuilder.invoke("toString")));

        acquireAccessTokenMethod.body()._return(object);
    }

    private void generateCreateAccessTokenMethod(Module module, GeneratedClass oauthManagerClass) {
        GeneratedClass connectorClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedMethod acquireAccessTokenMethod = oauthManagerClass.method(Modifier.PUBLIC, connectorClass, "createAccessToken");
        GeneratedVariable verifier = acquireAccessTokenMethod.param(ref(String.class), "verifier");
        acquireAccessTokenMethod._throws(ref(Exception.class));

        GeneratedVariable connector = acquireAccessTokenMethod.body().decl(connectorClass, "connector", ExpressionFactory._new(connectorClass));
        acquireAccessTokenMethod.body().add(connector.invoke("setOauthVerifier").arg(verifier));

        acquireAccessTokenMethod.body().add(connector.invoke("setAuthorizationUrl").arg(ExpressionFactory.invoke("getAuthorizationUrl")));
        acquireAccessTokenMethod.body().add(connector.invoke("setAccessTokenUrl").arg(ExpressionFactory.invoke("getAccessTokenUrl")));

        if( ((OAuthModule)module).getOAuthVersion() == OAuthVersion.V10A ) {
            acquireAccessTokenMethod.body().add(connector.invoke("setRequestTokenUrl").arg(ExpressionFactory.invoke("getRequestTokenUrl")));
        }

        for (Field field : module.getConfigurableFields()) {
            acquireAccessTokenMethod.body().add(connector.invoke(field.getSetter().getName()).arg(ExpressionFactory.invoke(field.getGetter().getName())));
        }

        acquireAccessTokenMethod.body()._if(Op._instanceof(connector, ref(MuleContextAware.class)))._then().add(connector.invoke("setMuleContext").arg(ExpressionFactory.ref("muleContext")));

        acquireAccessTokenMethod.body()._if(Op._instanceof(connector, ref(Initialisable.class)))._then().add(connector.invoke("initialise"));

        acquireAccessTokenMethod.body()._if(Op._instanceof(connector, ref(Startable.class)))._then().add(connector.invoke("start"));

        acquireAccessTokenMethod.body()._return(connector);
    }

    private void generateReleaseAccessTokenMethod(Module module, GeneratedClass oauthManagerClass, GeneratedField accessTokenPool, GeneratedField logger) {
        GeneratedClass connectorClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedMethod releaseAccessToken = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "releaseAccessToken");
        GeneratedVariable userId = releaseAccessToken.param(ref(String.class), "userId");
        releaseAccessToken._throws(ref(Exception.class));
        GeneratedVariable connector = releaseAccessToken.param(connectorClass, "connector");

        GeneratedConditional isDebugEnabled = releaseAccessToken.body()._if(logger.invoke("isDebugEnabled"));
        GeneratedVariable mesasgeStringBuilder = isDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("Pool Statistics before releasing [key "));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(userId));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [active="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumActive").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [idle="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumIdle").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("]"));
        isDebugEnabled._then().add(logger.invoke("debug").arg(mesasgeStringBuilder.invoke("toString")));

        releaseAccessToken.body().add(
                accessTokenPool.invoke("returnObject").arg(
                        userId
                ).arg(connector)
        );

        isDebugEnabled = releaseAccessToken.body()._if(logger.invoke("isDebugEnabled"));
        mesasgeStringBuilder = isDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("Pool Statistics after releasing [key "));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(userId));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [active="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumActive").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [idle="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumIdle").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("]"));
        isDebugEnabled._then().add(logger.invoke("debug").arg(mesasgeStringBuilder.invoke("toString")));
    }


    private void generateDestroyAccessTokenMethod(Module module, GeneratedClass oauthManagerClass, GeneratedField accessTokenPool, GeneratedField logger) {
        GeneratedClass connectorClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedMethod destroyConnector = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "destroyAccessToken");
        GeneratedVariable userId = destroyConnector.param(ref(String.class), "userId");
        destroyConnector._throws(ref(Exception.class));
        GeneratedVariable connector = destroyConnector.param(connectorClass, "connector");

        GeneratedConditional isDebugEnabled = destroyConnector.body()._if(logger.invoke("isDebugEnabled"));
        GeneratedVariable mesasgeStringBuilder = isDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("Pool Statistics before destroying [key "));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(userId));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [active="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumActive").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [idle="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumIdle").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("]"));
        isDebugEnabled._then().add(logger.invoke("debug").arg(mesasgeStringBuilder.invoke("toString")));

        destroyConnector.body().add(
                accessTokenPool.invoke("invalidateObject").arg(
                        userId
                ).arg(connector)
        );

        isDebugEnabled = destroyConnector.body()._if(logger.invoke("isDebugEnabled"));
        mesasgeStringBuilder = isDebugEnabled._then().decl(ref(StringBuilder.class), "messageStringBuilder", ExpressionFactory._new(ref(StringBuilder.class)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("Pool Statistics after destroying [key "));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(userId));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [active="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumActive").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("] [idle="));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg(accessTokenPool.invoke("getNumIdle").arg(userId)));
        isDebugEnabled._then().add(mesasgeStringBuilder.invoke("append").arg("]"));
        isDebugEnabled._then().add(logger.invoke("debug").arg(mesasgeStringBuilder.invoke("toString")));
    }

    protected void generateIsCapableOfMethod(Module module, GeneratedClass capabilitiesAdapter) {
        GeneratedMethod isCapableOf = capabilitiesAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "isCapableOf");
        GeneratedVariable capability = isCapableOf.param(((TypeReference)ctx().getProduct(Product.CAPABILITY_ENUM)), "capability");
        isCapableOf.javadoc().add("Returns true if this module implements such capability");

        addCapability(isCapableOf, capability, ((TypeReference)ctx().getProduct(Product.CAPABILITY_ENUM)).staticRef("LIFECYCLE_CAPABLE"));

        if (module instanceof OAuthModule && ((OAuthModule) module).getOAuthVersion() == OAuthVersion.V2) {
            addCapability(isCapableOf, capability, ((TypeReference)ctx().getProduct(Product.CAPABILITY_ENUM)).staticRef("OAUTH2_CAPABLE"));
        }

        if (module instanceof OAuthModule && ((OAuthModule) module).getOAuthVersion() == OAuthVersion.V10A) {
            addCapability(isCapableOf, capability, ((TypeReference)ctx().getProduct(Product.CAPABILITY_ENUM)).staticRef("OAUTH1_CAPABLE"));
        }

        if (module instanceof OAuthModule && ((OAuthModule) module).getUserIdentifierMethod() != null) {
            addCapability(isCapableOf, capability, ((TypeReference)ctx().getProduct(Product.CAPABILITY_ENUM)).staticRef("OAUTH_ACCESS_TOKEN_MANAGEMENT_CAPABLE"));
        }

        isCapableOf.body()._return(ExpressionFactory.FALSE);
    }

    private void addCapability(GeneratedMethod capableOf, GeneratedVariable capability, GeneratedFieldReference capabilityToCheckFor) {
        GeneratedConditional isCapable = capableOf.body()._if(Op.eq(capability, capabilityToCheckFor));
        isCapable._then()._return(ExpressionFactory.TRUE);
    }

    protected void generateMetadataConstantsAndGetters(Module module, GeneratedClass generatedClass) {
        String connectorName = module.getFriendlyName();
        if (org.mule.util.StringUtils.isEmpty(connectorName)) {
            connectorName = module.getModuleName();
        }

        GeneratedField moduleName = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("MODULE_NAME").initialValue(connectorName).build();
        GeneratedField moduleVersion = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("MODULE_VERSION").initialValue(ctx().getMavenInformation().getVersion()).build();
        GeneratedField devkitVersion = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("DEVKIT_VERSION").initialValue(ctx().getManifest().getProductVersion()).build();
        GeneratedField devkitBuild = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("DEVKIT_BUILD").initialValue(ctx().getManifest().getBuildNumber()).build();

        generateGetter(generatedClass, moduleName, "getModuleName");
        generateGetter(generatedClass, moduleVersion, "getModuleVersion");
        generateGetter(generatedClass, devkitVersion, "getDevkitVersion");
        generateGetter(generatedClass, devkitBuild, "getDevkitBuild");
    }

    private void generateGetter(GeneratedClass metadataAdapter, GeneratedField field, String getterMethodName) {
        GeneratedMethod getter = metadataAdapter.method(Modifier.PUBLIC, ref(String.class), getterMethodName);
        getter.body()._return(field);
    }

    private void generateGetProcessTemplateMethod(OAuthModule module, GeneratedClass oauthAdapterClass, GeneratedClass capabilitiesAdapterClass) {
        GeneratedClass managedOAuthProcessTemplateClass = ctx().getProduct(Product.MANAGED_ACCESS_TOKEN_PROCESS_TEMPLATE);
        GeneratedMethod getProcessTemplate = oauthAdapterClass.method(Modifier.PUBLIC, (TypeReference) ctx().getProduct(Product.PROCESS_TEMPLATE_INTERFACE), "getProcessTemplate");
        getProcessTemplate.annotate(ref(Override.class));
        TypeVariable p = getProcessTemplate.generify("P");
        getProcessTemplate.type(((TypeReference) ctx().getProduct(Product.PROCESS_TEMPLATE_INTERFACE)).narrow(p).narrow(capabilitiesAdapterClass));

        GeneratedInvocation newProcessTemplate = ExpressionFactory._new(managedOAuthProcessTemplateClass).arg(ExpressionFactory._this());
        newProcessTemplate.arg(ExpressionFactory.invoke("getMuleContext"));
        getProcessTemplate.body()._return(newProcessTemplate);
    }
}