/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.oauth;

import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.ModuleAnnotationVerifier;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Type;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.oauth.OAuthModule;

import java.lang.annotation.Annotation;

import javax.lang.model.type.TypeKind;

public abstract class AbstractOAuthClientAnnotationVerifier implements ModuleAnnotationVerifier {

    @Override
    public boolean shouldVerify(Module module) {
        return module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC;
    }

    protected abstract void verify(OAuthModule oAuthModule) throws AnnotationVerificationException ;

    @Override
    public void verify(Module module) throws AnnotationVerificationException {
        if (module instanceof OAuthModule && ((OAuthModule) module).getUserIdentifierMethod() != null) {
            if (!((OAuthModule) module).getUserIdentifierMethod().getReturnType().toString().equals("java.lang.String")) {
                throw new AnnotationVerificationException(module, "A @OAuthAccessTokenIdentifier must return java.lang.String.");
            }
            if (!((OAuthModule) module).getUserIdentifierMethod().isPublic()) {
                throw new AnnotationVerificationException(module, "A @OAuthAccessTokenIdentifier method must be public.");
            }
            if (((OAuthModule) module).getUserIdentifierMethod().isStatic()) {
                throw new AnnotationVerificationException(module, "A @OAuthAccessTokenIdentifier cannot be static.");
            }
            if (!((OAuthModule) module).getUserIdentifierMethod().getParameters().isEmpty()) {
                throw new AnnotationVerificationException(module, "The @OAuthAccessTokenIdentifier method cannot receive any arguments");
            }
        }

        verify((OAuthModule)module);
    }

    protected void verifyPostAuthorizationMethod(OAuthModule module) throws AnnotationVerificationException {
        if (module.getPostAuthorizationMethod() != null &&
                module.getPostAuthorizationMethod().isPrivate()) {
            throw new AnnotationVerificationException(module, "@OAuthPostAuthorization-annotated method must be public or protected");
        }

        if (module.getPostAuthorizationMethod() != null &&
                module.getPostAuthorizationMethod().getParameters().size() > 0) {
            throw new AnnotationVerificationException(module, "@OAuthPostAuthorization-annotated method cannot have any parameter");
        }

        if (module.getPostAuthorizationMethod() != null &&
                module.getPostAuthorizationMethod().isStatic()) {
            throw new AnnotationVerificationException(module, "@OAuthPostAuthorization-annotated method cannot be static");
        }

        if (module.getPostAuthorizationMethod() != null &&
                module.getPostAuthorizationMethod().getReturnType().getKind() != TypeKind.VOID) {
            throw new AnnotationVerificationException(module, "@OAuthPostAuthorization-annotated method must be void");
        }
    }

    protected boolean classHasMethodWithParameterAnnotated(Module module, Class<? extends Annotation> annotation) {
        for (Method<Type> method : module.getMethods()) {
            for (Parameter parameter : method.getParameters()) {
                if (parameter.getAnnotation(annotation) != null) {
                    return true;
                }
            }
        }
        return false;
    }
}