/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.oauth;

import org.mule.api.devkit.ProcessTemplate;
import org.mule.common.security.oauth.AuthorizationParameter;
import org.mule.common.security.oauth.exception.NotAuthorizedException;
import org.mule.common.security.oauth.exception.UnableToAcquireAccessTokenException;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedBlock;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedField;
import org.mule.devkit.model.code.GeneratedInvocation;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.Op;
import org.mule.devkit.model.code.TypeVariable;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthAuthorizationParameter;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.security.oauth.OAuth2Adapter;
import org.mule.security.oauth.OAuth2Manager;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;

public class OAuth2ClientAdapterGenerator extends AbstractOAuthAdapterGenerator {

    @Override
    public boolean shouldGenerate(Module module) {
        return false;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass oauthAdapter = getOAuthAdapterClass(module, "OAuth2Adapter", ref(OAuth2Adapter.class));
        oauthAdapter._implements(OAuth2Adapter.class);
        OAuthModule oAuthModule = (OAuthModule) module;

        GeneratedClass oauthManagerClass = ctx().getProduct(Product.OAUTH_MANAGER, module);
        GeneratedField oauthManager = oauthAdapter.field(Modifier.PRIVATE,
                ref(OAuth2Manager.class).narrow(OAuth2Adapter.class), "oauthManager");

        generateConstructor(oauthAdapter, oauthManager, OAuth2Manager.class, OAuth2Adapter.class);

        accessTokenPatternConstant(oauthAdapter, oAuthModule);
        refreshTokenPatternConstant(oauthAdapter, oAuthModule);
        expirationPatternConstant(oauthAdapter, oAuthModule);

        GeneratedField name = oauthAdapter.field(Modifier.PRIVATE, ref(String.class),"name");
        oauthAdapter.getter(name);
        oauthAdapter.setterOverride(name);

        generateOnNoTokenPolicyField(oauthAdapter);

        authorizationCodeField(oauthAdapter);
        refreshTokenField(oauthAdapter);

        GeneratedField redirectUri = oauthAdapter.field(Modifier.PUBLIC, ref(String.class), "redirectUri");
        GeneratedField authorizationUrl = generateAuthorizationUrlField(oauthAdapter);
        GeneratedField accessTokenUrl = generateAccessTokenUrlField(oauthAdapter);

        expirationField(oauthAdapter, oAuthModule);

        generateInitialiseMethod(oauthAdapter);

        GeneratedField logger = FieldBuilder.newLoggerField(oauthAdapter);
        generateOAuth2AuthorizeMethod(oauthAdapter, oauthManager);
        generateFetchAccessTokenMethod(oauthAdapter, oauthManager);
        generateRefreshAccessTokenMethod(oauthAdapter, oauthManager);
        generateHasTokenExpiredMethod(oauthAdapter, oAuthModule);
        generateResetMethod(oAuthModule, oauthAdapter);
        generateHasBeenAuthorizedMethod(oAuthModule, oauthAdapter);

        generateRegexpGetters(oauthAdapter);

        generateOAuth2ConnectorInterfaceImpl(oAuthModule,oauthAdapter);

        generateGetProcessTemplateMethod(oauthAdapter, ctx().<GeneratedClass>getProduct(Product.CAPABILITIES_ADAPTER, module),oauthManager);
    }


    protected void generateHasBeenAuthorizedMethod(OAuthModule module, GeneratedClass oauthAdapter) {
        GeneratedMethod hasBeenAuthorized = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "hasBeenAuthorized");
        hasBeenAuthorized._throws(ref(NotAuthorizedException.class));
        GeneratedBlock ifAccessTokenIsNull = hasBeenAuthorized.body()._if(ExpressionFactory.invoke(module.getAccessTokenField().getGetter().getName()).isNull())._then();

        GeneratedInvocation newNotAuthorizedException = ExpressionFactory._new(ref(NotAuthorizedException.class));
        newNotAuthorizedException.arg("This connector has not yet been authorized, please authorize by calling \"authorize\".");

        ifAccessTokenIsNull._throw(newNotAuthorizedException);
    }

    private void generateOAuth2ConnectorInterfaceImpl(OAuthModule oAuthModule, GeneratedClass oauthAdapter) {

        GeneratedMethod getAuthorizationParameters = oauthAdapter.method(Modifier.PUBLIC,ref(Set.class).narrow(ref(AuthorizationParameter.class).narrow(ctx().getCodeModel().wildcard())),"getAuthorizationParameters");
        getAuthorizationParameters.annotate(Override.class);


        List<GeneratedVariable> generatedVariableList = new ArrayList<GeneratedVariable>();

        for (OAuthAuthorizationParameter parameter: oAuthModule.getAuthorizationParameters()) {
            GeneratedVariable param = getAuthorizationParameters.body().decl(ref(AuthorizationParameter.class).narrow(String.class),parameter.getName(),ExpressionFactory._new(ref(AuthorizationParameter.class).narrow(String.class))
                    .arg(ExpressionFactory.lit(parameter.getName()))
                    .arg(ExpressionFactory.lit(parameter.getDescription()))
                    .arg(ExpressionFactory.lit(parameter.isOptional()))
                    .arg(ExpressionFactory.lit(parameter.getDefaultValue()))
                    .arg(ref(parameter.getType().asTypeMirror()).dotclass()));
            generatedVariableList.add(param);
        }

        GeneratedVariable result = getAuthorizationParameters.body().decl(ref(Set.class).narrow(ref(AuthorizationParameter.class).narrow(ctx().getCodeModel().wildcard()))
                ,"result"
                ,ExpressionFactory._new(ref(HashSet.class).narrow(ref(AuthorizationParameter.class).narrow(ctx().getCodeModel().wildcard()))));

        for (GeneratedVariable param: generatedVariableList) {
            getAuthorizationParameters.body().add(result.invoke("add").arg(param));
        }

        getAuthorizationParameters.body()._return(result);


        if (!oAuthModule.hasMethodWithName("getConsumerKey")) {
            GeneratedMethod getConsumerKeyMethod = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getConsumerKey");
            getConsumerKeyMethod.annotate(Override.class);
            getConsumerKeyMethod.body()._return(ExpressionFactory._this().invoke(oAuthModule.getConsumerKeyField().getGetter().getName()));
        }

        if (!oAuthModule.hasMethodWithName("setConsumerKey")) {
            GeneratedMethod setConsumerKey = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setConsumerKey");
            GeneratedVariable value = setConsumerKey.param(ref(String.class),"value");
            setConsumerKey.annotate(Override.class);
            setConsumerKey.body().add(ExpressionFactory._this().invoke(oAuthModule.getConsumerKeyField().getSetter().getName()).arg(value));
        }

        if (!oAuthModule.hasMethodWithName("getConsumerSecret")) {
            GeneratedMethod getConsumerSecret = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getConsumerSecret");
            getConsumerSecret.annotate(Override.class);
            getConsumerSecret.body()._return(ExpressionFactory._this().invoke(oAuthModule.getConsumerSecretField().getGetter().getName()));
        }

        if (!oAuthModule.hasMethodWithName("setConsumerSecret")) {
            GeneratedMethod setConsumerSecret = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setConsumerSecret");
            GeneratedVariable value = setConsumerSecret.param(ref(String.class),"value");
            setConsumerSecret.annotate(Override.class);
            setConsumerSecret.body().add(ExpressionFactory._this().invoke(oAuthModule.getConsumerSecretField().getSetter().getName()).arg(value));
        }


        if (!oAuthModule.hasMethodWithName("getAccessToken")) {
            GeneratedMethod getAccessToken = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getAccessToken");
            getAccessToken.annotate(Override.class);
            getAccessToken.body()._return(ExpressionFactory._this().invoke(oAuthModule.getAccessTokenField().getGetter().getName()));
        }

        if (!oAuthModule.hasMethodWithName("setAccessToken")) {
            GeneratedMethod setAccessToken = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setAccessToken");
            GeneratedVariable value = setAccessToken.param(ref(String.class),"value");
            setAccessToken.annotate(Override.class);
            setAccessToken.body().add(ExpressionFactory._this().invoke(oAuthModule.getAccessTokenField().getSetter().getName()).arg(value));
        }


        if (!oAuthModule.hasMethodWithName("getScope")) {
            GeneratedMethod getScope = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getScope");
            getScope.annotate(Override.class);
            if (oAuthModule.getScopeField() != null){
                getScope.body()._return(ExpressionFactory._this().invoke(oAuthModule.getScopeField().getGetter().getName()));
            } else {
                getScope.body()._return(ExpressionFactory.lit(""));
            }
        }

        if (!oAuthModule.hasMethodWithName("setScope") && oAuthModule.getScopeField() != null) {
            GeneratedMethod setScope = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setScope");
            GeneratedVariable value = setScope.param(ref(String.class),"value");
            setScope.body().add(ExpressionFactory._this().invoke(oAuthModule.getScopeField().getSetter().getName()).arg(value));
        }


        if (!oAuthModule.hasMethodWithName("postAuth")) {
            GeneratedMethod postAuth = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"postAuth");
            postAuth.annotate(Override.class);

            if (oAuthModule.getPostAuthorizationMethod() != null) {
                //TODO: on mule fix uncomment
                postAuth._throws(Exception.class);
                postAuth.body().add(ExpressionFactory._this().invoke(oAuthModule.getPostAuthorizationMethod().getName()));
            }
        }

    }

    private void generateRegexpGetters(GeneratedClass oauthAdapter) {
        oauthAdapter.direct("@Override\n" +
                "    public String getAccessTokenRegex() {\n" +
                "        return ACCESS_CODE_PATTERN.pattern();\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public String getExpirationRegex() {\n" +
                "        return EXPIRATION_TIME_PATTERN.pattern();\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public String getRefreshTokenRegex() {\n" +
                "        return REFRESH_TOKEN_PATTERN.pattern();\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public String getVerifierRegex() {\n" +
                "        return oauthVerifier;\n" +
                "    }");
    }



    protected GeneratedField refreshTokenField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(REFRESH_TOKEN_FIELD_NAME).getterAndSetter().build();
    }

    private void accessTokenPatternConstant(GeneratedClass oauthAdapter, OAuthModule module) {
        GeneratedField accessTokenPattern = new FieldBuilder(oauthAdapter).type(Pattern.class).name(AbstractOAuthAdapterGenerator.ACCESS_CODE_PATTERN_FIELD_NAME).staticField().finalField().
                initialValue(ref(Pattern.class).staticInvoke("compile").arg(module.getAccessTokenRegex())).build();
        GeneratedMethod getter = oauthAdapter.getter(accessTokenPattern);
        getter.name("getAccessCodePattern");
        getter.annotate(Override.class);
    }

    private void refreshTokenPatternConstant(GeneratedClass oauthAdapter, OAuthModule module) {
        GeneratedField refreshTokenPattern = new FieldBuilder(oauthAdapter).type(Pattern.class).name(AbstractOAuthAdapterGenerator.REFRESH_TOKEN_PATTERN_FIELD_NAME).staticField().finalField().
                initialValue(ref(Pattern.class).staticInvoke("compile").arg(module.getRefreshTokenRegex())).build();
        GeneratedMethod getter = oauthAdapter.getter(refreshTokenPattern);
        getter.name("getRefreshTokenPattern");
        getter.annotate(Override.class);
    }

    private void expirationPatternConstant(GeneratedClass oauthAdapter, OAuthModule module) {
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            GeneratedField expirationPattern = new FieldBuilder(oauthAdapter).type(Pattern.class).name(AbstractOAuthAdapterGenerator.EXPIRATION_TIME_PATTERN_FIELD_NAME).staticField().finalField().
                    initialValue(ref(Pattern.class).staticInvoke("compile").arg(module.getExpirationRegex())).build();
            GeneratedMethod getter = oauthAdapter.getter(expirationPattern);
            getter.name("getExpirationTimePattern");
            getter.annotate(Override.class);
        }
    }

    private void expirationField(GeneratedClass oauthAdapter, OAuthModule module) {
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            new FieldBuilder(oauthAdapter).type(Date.class).name(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME).setter().build();
        }
    }

    private void generateRefreshAccessTokenMethod(GeneratedClass oauthAdapter,GeneratedField oauthManager) {
        GeneratedMethod refreshAccessTokenMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "refreshAccessToken");
        GeneratedVariable accessTokenId = refreshAccessTokenMethod.param(String.class, "accessTokenId");
        refreshAccessTokenMethod._throws(ref(UnableToAcquireAccessTokenException.class));
        refreshAccessTokenMethod.annotate(Override.class);
//        GeneratedVariable accessTokenUrl = refreshAccessTokenMethod.param(ref(String.class), "accessTokenUrl");
        refreshAccessTokenMethod.body().add(oauthManager.invoke("refreshAccessToken").arg(ExpressionFactory._this()).arg(accessTokenId));
    }

    private void generateFetchAccessTokenMethod(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod fetchAccessToken = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "fetchAccessToken");
        GeneratedVariable redirectUri = fetchAccessToken.param(ref(String.class), "redirectUri");
        fetchAccessToken._throws(ref(UnableToAcquireAccessTokenException.class));
        fetchAccessToken.body().add(oauthManager.invoke("fetchAccessToken").arg(ExpressionFactory._this()).arg(redirectUri));

    }
    private void generateHasTokenExpiredMethod(GeneratedClass oauthAdapter, OAuthModule module) {
        GeneratedMethod hasTokenExpired = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, AbstractOAuthAdapterGenerator.HAS_TOKEN_EXPIRED_METHOD_NAME);
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            GeneratedField expirationDate = oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME);
            hasTokenExpired.body()._return(Op.cand(
                    expirationDate.isNotNull(),
                    expirationDate.invoke("before").arg(ExpressionFactory._new(ref(Date.class)))));
        } else {
            hasTokenExpired.body()._return(ExpressionFactory.FALSE);
        }
    }

    private void generateResetMethod(OAuthModule module, GeneratedClass oauthAdapter) {
        GeneratedMethod reset = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.RESET_METHOD_NAME);
        if (!StringUtils.isEmpty(module.getExpirationRegex())) {
            reset.body().assign(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.EXPIRATION_FIELD_NAME), ExpressionFactory._null());
        }
        reset.body().assign(oauthAdapter.fields().get(AbstractOAuthAdapterGenerator.VERIFIER_FIELD_NAME), ExpressionFactory._null());
        reset.body().invoke(module.getAccessTokenField().getSetter().getName()).arg(ExpressionFactory._null());
    }

    protected void generateGetProcessTemplateMethod(GeneratedClass oauthAdapterClass, GeneratedClass capabilitiesAdapterClass, GeneratedField oauthManager) {
        GeneratedMethod getProcessTemplate = oauthAdapterClass.method(Modifier.PUBLIC, ref(ProcessTemplate.class), "getProcessTemplate");
        getProcessTemplate.annotate(ref(Override.class));
        TypeVariable p = getProcessTemplate.generify("P");
        getProcessTemplate.type(ref(ProcessTemplate.class).narrow(p).narrow(capabilitiesAdapterClass));

        getProcessTemplate.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)));
    }
}