/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.oauth.processors;

import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.oauth.AbstractOAuthAdapterGenerator;
import org.mule.devkit.generation.oauth.OAuthClientNamingConstants;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedField;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedPackage;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthAuthorizationParameter;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.model.schema.SchemaTypeConversion;
import org.mule.security.oauth.OAuth1Adapter;
import org.mule.security.oauth.processor.BaseOAuth1AuthorizeMessageProcessor;
import org.mule.security.oauth.processor.BaseOAuth2AuthorizeMessageProcessor;

import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;

public class AuthorizeMessageProcessorGenerator extends AbstractOAuthAdapterGenerator {
    private static final String HTTP_STATUS_PROPERTY = "http.status";
    private static final String REDIRECT_HTTP_STATUS = "302";
    private static final String LOCATION_PROPERTY = "Location";
    private static final String CALLBACK_FIELD_NAME = "oauthCallback";
    private final static List<Product> CONSUMES = Arrays.asList(new Product[]{Product.OAUTH_ADAPTER, Product.OAUTH_MANAGER, Product.OAUTH_EXTRACT_AUTHORIZATION_CODE_MESSAGE_PROCESSOR, Product.OAUTH_FETCH_ACCESS_TOKEN_MESSAGE_PROCESSOR});
    private final static List<Product> PRODUCES = Arrays.asList(new Product[]{Product.MESSAGE_PROCESSOR});

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        // get class
        GeneratedClass messageProcessorClass;
        messageProcessorClass = getAuthorizeMessageProcessorClass(module);
        GeneratedField authorizeCodePattern = authorizationCodePatternConstant(messageProcessorClass, ((OAuthModule) module).getVerifierRegex());
        generateExtraParameterFields((OAuthModule) module, messageProcessorClass);
        generateGetAuthCodeRegexMethod(messageProcessorClass, authorizeCodePattern);

        if (((OAuthModule) module).getOAuthVersion().equals(OAuthVersion.V2)) {
            generateGetOAuthManagerClass(module,messageProcessorClass);
        } else {
            generateGetAdapterClass(module,messageProcessorClass);
        }
    }

    private void generateGetAdapterClass(Module module, GeneratedClass messageProcessorClass) {
        GeneratedMethod getAdapterClassMethod = messageProcessorClass.method(Modifier.PROTECTED, ref(Class.class).narrow(ref(OAuth1Adapter.class).wildcard()),"getAdapterClass");
        getAdapterClassMethod.annotate(Override.class);

        if (module.hasRestCalls()) {
            getAdapterClassMethod.body()._return(ExpressionFactory.direct(module.getPackage().getName() + ".adapters." + module.getClassName() + "RestClientAdapter.class"));
        } else {
            GeneratedClass oauthAdapter = ctx().getProduct(Product.OAUTH_ADAPTER,module);
            getAdapterClassMethod.body()._return(oauthAdapter.dotclass());
        }
    }

    private void generateGetOAuthManagerClass(Module module, GeneratedClass messageProcessorClass) {
        GeneratedClass oauthManager = ctx().<GeneratedClass>getProduct(Product.OAUTH_MANAGER, module);
        GeneratedMethod getOAuthManagerClassMethod = messageProcessorClass.method(Modifier.PROTECTED, ref(Class.class).narrow(oauthManager),"getOAuthManagerClass");
        getOAuthManagerClassMethod.annotate(Override.class);
        getOAuthManagerClassMethod.body()._return(oauthManager.dotclass());
    }

    private void generateGetAuthCodeRegexMethod(GeneratedClass messageProcessorClass, GeneratedField authorizeCodePattern) {
        GeneratedMethod getAuthCodeRegexMethod = messageProcessorClass.method(Modifier.PROTECTED,ref(String.class),"getAuthCodeRegex");
        getAuthCodeRegexMethod.annotate(Override.class);
        getAuthCodeRegexMethod.body()._return(authorizeCodePattern.invoke("pattern"));
    }


    private void generateExtraParameterFields(OAuthModule module, GeneratedClass messageProcessorClass) {
        if (module.getAuthorizationParameters() != null) {
            for (OAuthAuthorizationParameter parameter : module.getAuthorizationParameters()) {
                if (SchemaTypeConversion.isSupported(parameter.getType().asTypeMirror().toString()) || parameter.getType().isEnum()) {
                    GeneratedField field = messageProcessorClass.field(Modifier.PRIVATE, ref(Object.class), parameter.getName());
                    messageProcessorClass.setter(field);
                    GeneratedField fieldType = messageProcessorClass.field(Modifier.PRIVATE, ref(parameter.getType().asTypeMirror()), "_" + parameter.getName() + "Type");
                }
            }
        }
    }

    private GeneratedClass getAuthorizeMessageProcessorClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.MESSAGE_PROCESSOR_NAMESPACE);

        GeneratedClass moduleObject = ctx().<GeneratedClass>getProduct(Product.OAUTH_MANAGER, module);
        OAuthModule oauthModule = (OAuthModule) module;

        GeneratedClass clazz = null;
        if (oauthModule.getOAuthVersion().equals(OAuthVersion.V2)) {
            clazz = pkg._class(OAuthClientNamingConstants.AUTHORIZE_MESSAGE_PROCESSOR_CLASS_NAME, ref(BaseOAuth2AuthorizeMessageProcessor.class).narrow(moduleObject));
        } else {
            clazz = pkg._class(OAuthClientNamingConstants.AUTHORIZE_MESSAGE_PROCESSOR_CLASS_NAME, ref(BaseOAuth1AuthorizeMessageProcessor.class));
        }

        ctx().registerProduct(Product.MESSAGE_PROCESSOR, module, "authorize", clazz);

        return clazz;
    }


    protected GeneratedField authorizationCodePatternConstant(GeneratedClass oauthAdapter, String regex) {
        return new FieldBuilder(oauthAdapter).type(Pattern.class).name(AUTH_CODE_PATTERN_FIELD_NAME).staticField().finalField().
                initialValue(ref(Pattern.class).staticInvoke("compile").arg(regex)).build();
    }
}
