/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.oauth;

import org.mule.DefaultMuleMessage;
import org.mule.api.MuleMessage;
import org.mule.api.expression.ExpressionManager;
import org.mule.api.store.ObjectStore;
import org.mule.common.security.oauth.OAuthState;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.ModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedBlock;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedExpression;
import org.mule.devkit.model.code.GeneratedField;
import org.mule.devkit.model.code.GeneratedInvocation;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedPackage;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.oauth.OAuthCallbackParameterField;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.security.oauth.BaseOAuth1Manager;
import org.mule.security.oauth.BaseOAuth2Manager;
import org.mule.security.oauth.BaseOAuthClientFactory;
import org.mule.security.oauth.OAuth2Adapter;
import org.mule.security.oauth.OAuth2Manager;
import org.mule.security.oauth.OnNoTokenPolicy;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.lang.model.type.DeclaredType;

import oauth.signpost.OAuthConsumer;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.pool.KeyedPoolableObjectFactory;

public class OAuthManagerGenerator extends AbstractOAuthAdapterGenerator implements ModuleGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(Product.CAPABILITIES_ADAPTER,
            Product.LIFECYCLE_ADAPTER,
            Product.INJECTION_ADAPTER,
            Product.MANAGED_ACCESS_TOKEN_PROCESS_TEMPLATE,
            Product.METADATA_ADAPTER,
            Product.PROCESS_INTERFACES
            );
    private final static List<Product> PRODUCES = Arrays.asList(Product.OAUTH_MANAGER, Product.OAUTH_ADAPTER);


    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        OAuthModule oauthModule = (OAuthModule) module;

        GeneratedClass oauthManagerClass = getOAuthManagerClass(oauthModule);
        GeneratedField logger = generateLoggerField(oauthManagerClass);
        getLoggerMethod(oauthManagerClass,logger);

        ModuleGenerator oauthAdapterGenerator = null;

        //TODO: make this with a double dispatch
        if (((OAuthModule) module).getOAuthVersion().equals(OAuthVersion.V2)) {
            oauthAdapterGenerator = new OAuth2ClientAdapterGenerator();
            oauthAdapterGenerator.setCtx(ctx());
            oauthAdapterGenerator.generate(module);
        } else {
            oauthAdapterGenerator = new OAuth1ClientAdapterGenerator();
            oauthAdapterGenerator.setCtx(ctx());
            oauthAdapterGenerator.generate(module);
        }

        // generate other stuff only on oauth2
        if (((OAuthModule) module).getOAuthVersion().equals(OAuthVersion.V10A)) {
            return;
        }

        GeneratedClass oauthClientFactoryClass = getOAuthClientFactoryClass(module);
        GeneratedClass adapterClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedClass stateClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_STATE, module);

        GeneratedField oauthManager = generateOAuthManagerClientFactoryField(oauthClientFactoryClass,oauthManagerClass);
        generateOAuthClientFactoryConstructor(oauthManagerClass, oauthClientFactoryClass);
        generateGetAdapterClassMethod(module, oauthClientFactoryClass, adapterClass);
        generateSetCustomAdapterPropertiesMethod(module, oauthClientFactoryClass, adapterClass, stateClass, oauthManager);
        generateSetCustomStateProperties(module, oauthClientFactoryClass, adapterClass, stateClass);

        for (Field field : module.getConfigurableFields()) {

            if (((OAuthModule) module).getConsumerKeyField().equals(field)) {
                GeneratedMethod setter = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "set" + StringUtils.capitalize(field.getName()));
                setter.javadoc().add("Sets " + field.getName());
                setter.javadoc().addParam("key to set");
                GeneratedVariable value = setter.param(ref(String.class), "value");
                setter.body().add(ExpressionFactory._super().invoke("setConsumerKey").arg(value));
                continue;
            } else if (((OAuthModule) module).getConsumerSecretField().equals(field)) {
                GeneratedMethod setter = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "set" + StringUtils.capitalize(field.getName()));
                setter.javadoc().add("Sets " + field.getName());
                setter.javadoc().addParam("secret to set");
                GeneratedVariable value = setter.param(ref(String.class), "value");
                setter.body().add(ExpressionFactory._super().invoke("setConsumerSecret").arg(value));
                continue;
            } else if (((OAuthModule) module).getScopeField() != null && ((OAuthModule) module).getScopeField().equals(field)) {
                GeneratedMethod setter = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "set" + StringUtils.capitalize(field.getName()));
                setter.javadoc().add("Sets " + field.getName());
                setter.javadoc().addParam("scope to set");
                GeneratedVariable value = setter.param(ref(String.class), "value");
                setter.body().add(ExpressionFactory._super().invoke("setScope").arg(value));
                continue;
            }

            GeneratedMethod setter = oauthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "set" + StringUtils.capitalize(field.getName()));
            setter.javadoc().add("Sets " + field.getName());
            setter.javadoc().addParam("scope to set");
            GeneratedVariable value = setter.param(ref(field.asTypeMirror()), "value");
            GeneratedVariable setterConnector = setter.body().decl(adapterClass,"connector",ExpressionFactory.cast(adapterClass,ExpressionFactory._this().invoke("getDefaultUnauthorizedConnector")));
            setter.body().add(setterConnector.invoke(setter.name()).arg(value));

            GeneratedMethod getter = oauthManagerClass.method(Modifier.PUBLIC, ref(field.asTypeMirror()), "get" + StringUtils.capitalize(field.getName()));
            getter.javadoc().add("Retrieves " + field.getName());
            GeneratedVariable getterConnector = getter.body().decl(adapterClass,"connector",ExpressionFactory.cast(adapterClass,ExpressionFactory._this().invoke("getDefaultUnauthorizedConnector")));
            getter.body()._return(getterConnector.invoke(getter.name()));
        }

        generateMetadataConstantsAndGetters(module, oauthManagerClass);

        if (oauthModule.getOAuthVersion() == OAuthVersion.V2) {
            generateInstantiateMethod(module, oauthManagerClass);
            generateCreatePoolFactoryMethod(oauthClientFactoryClass, oauthManagerClass);
            generateSetCustomPropertiesMethod(module, adapterClass, oauthManagerClass);
            generateFetchCallbackParameters(oauthModule, oauthManagerClass, adapterClass);
            generateSetOnNoTokenPolicy(oauthManagerClass);
            generateRefreshAccessTokenOn(oauthModule, oauthManagerClass);
        }

    }

    private GeneratedField generateOAuthManagerClientFactoryField(GeneratedClass oauthClientFactoryClass, GeneratedClass oauthManagerClass) {
        return oauthClientFactoryClass.field(Modifier.PRIVATE,oauthManagerClass,"oauthManager");
    }

    private void generateSetOnNoTokenPolicy(GeneratedClass oauthManagerClass) {
        /**
         * this.getDefaultUnauthorizedConnector().setOnNoTokenPolicy(policy);
         */
        GeneratedMethod setOnNoTokenPolicyMethod = oauthManagerClass.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setOnNoToken");
        GeneratedVariable policy = setOnNoTokenPolicyMethod.param(ref(OnNoTokenPolicy.class),"policy");
        setOnNoTokenPolicyMethod.body().add(ExpressionFactory._this().invoke("getDefaultUnauthorizedConnector").invoke("setOnNoTokenPolicy").arg(policy));
    }

    private void getLoggerMethod(GeneratedClass oauthManagerClass, GeneratedField logger) {
        GeneratedMethod getLoggerMethod = oauthManagerClass.method(Modifier.PROTECTED, ref(org.slf4j.Logger.class),"getLogger");
        getLoggerMethod.annotate(Override.class);
        getLoggerMethod.body()._return(logger);
    }

    private void generateRefreshAccessTokenOn(OAuthModule module, GeneratedClass oauthManagerClass) {
        Set<DeclaredType> exceptions = new HashSet<DeclaredType>();

        for (ProcessorMethod processor : module.getProcessorMethods()) {
            if (processor.invalidateAccessTokenOn() != null) {
                exceptions.add(processor.invalidateAccessTokenOn());
            }
        }

        if (!exceptions.isEmpty()) {
            GeneratedMethod method = oauthManagerClass.method(Modifier.PROTECTED, ref(Set.class).narrow(ref(Class.class).narrow(ref(Exception.class).wildcard())), "refreshAccessTokenOn");
            method.annotate(Override.class);

            GeneratedExpression set = method.body().decl(ref(Set.class).narrow(ref(Class.class).narrow(ref(Exception.class).wildcard())), "types", ExpressionFactory._new(ref(HashSet.class).narrow(ref(Class.class).narrow(ref(Exception.class).wildcard()))));

            for (DeclaredType type : exceptions) {
                method.body().invoke(set, "add").arg(ExpressionFactory.direct(type + ".class"));
            }

            method.body()._return(set);
        }
    }

    private void generateFetchCallbackParameters(OAuthModule module, GeneratedClass oauthManagerClass, GeneratedClass adapterClass) {
        GeneratedMethod fetchCallbackParameters = oauthManagerClass.method(Modifier.PROTECTED, context.getCodeModel().VOID, "fetchCallbackParameters");
        GeneratedVariable adapter = fetchCallbackParameters.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable response = fetchCallbackParameters.param(ref(String.class), "response");

        GeneratedVariable connector = fetchCallbackParameters.body().decl(adapterClass,"connector", ExpressionFactory.cast(adapterClass,adapter));

        GeneratedBlock body = fetchCallbackParameters.body();
        GeneratedVariable expressionManager = body.decl(ref(ExpressionManager.class), "expressionManager", ExpressionFactory.direct("muleContext.getExpressionManager()"));

        GeneratedVariable muleMessage = body.decl(ref(MuleMessage.class), "muleMessage",
                ExpressionFactory._new(ref(DefaultMuleMessage.class)).arg(response).arg(ExpressionFactory.direct("muleContext")));

        for (OAuthCallbackParameterField field : module.getCallbackParameters()) {

            GeneratedInvocation invocation = connector.invoke("set" + StringUtils.capitalize(field.getName())).arg(
                    ExpressionFactory.cast(ref(field.asTypeMirror()), expressionManager.invoke("evaluate").arg(field.getExpression()).arg(muleMessage)));
            body.add(invocation);
        }
    }

    private void generateSetCustomPropertiesMethod(Module module, GeneratedClass adapterClass, GeneratedClass oauthManagerClass) {
        GeneratedMethod setCustomPropertiesMethod = oauthManagerClass.method(Modifier.PROTECTED,ctx().getCodeModel().VOID,"setCustomProperties");
        setCustomPropertiesMethod.annotate(Override.class);
        GeneratedVariable adapter = setCustomPropertiesMethod.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable connector = setCustomPropertiesMethod.body().decl(adapterClass, "connector", ExpressionFactory.cast(adapterClass, adapter));

        for (Field field : module.getConfigurableFields()) {
            if (((OAuthModule) module).getConsumerKeyField().equals(field)) {
                setCustomPropertiesMethod.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName()))
                        .arg(ExpressionFactory.invoke("getConsumerKey")));
            } else if (((OAuthModule) module).getConsumerSecretField().equals(field)) {
                setCustomPropertiesMethod.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName()))
                        .arg(ExpressionFactory.invoke("getConsumerSecret")));
            } else if (((OAuthModule) module).getScopeField() != null && ((OAuthModule) module).getScopeField().equals(field)) {
                setCustomPropertiesMethod.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName()))
                        .arg(ExpressionFactory.invoke("getScope")));
            } else {
                setCustomPropertiesMethod.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName()))
                        .arg(ExpressionFactory.invoke("get" + StringUtils.capitalize(field.getName()))));
            }
        }

    }

    private void generateCreatePoolFactoryMethod(GeneratedClass oauthClientFactoryClass, GeneratedClass oauthManagerClass) {
        GeneratedMethod createPoolFactoryMethod = oauthManagerClass.method(Modifier.PROTECTED,ref(KeyedPoolableObjectFactory.class),"createPoolFactory");
        createPoolFactoryMethod.annotate(Override.class);
        GeneratedVariable oauthManager = createPoolFactoryMethod.param(ref(OAuth2Manager.class).narrow(OAuth2Adapter.class),"oauthManager");
        GeneratedVariable objectStore = createPoolFactoryMethod.param(ref(ObjectStore.class).narrow(Serializable.class),"objectStore");
        createPoolFactoryMethod.body()._return(ExpressionFactory._new(oauthClientFactoryClass).arg(oauthManager).arg(objectStore));
    }

    private void generateInstantiateMethod(Module module,GeneratedClass oauthManagerClass) {
        GeneratedMethod instantiateMethod = oauthManagerClass.method(Modifier.PROTECTED,ref(OAuth2Adapter.class),"instantiateAdapter");
        instantiateMethod.annotate(Override.class);
        GeneratedClass oauthAdapterClass;
        if (module.hasRestCalls()) {
            instantiateMethod.body().directStatement("return new " +module.getPackage().getName() + ".adapters."+ module.getClassName() + "RestClientAdapter(this);");
        } else {
            oauthAdapterClass = ctx().getProduct(Product.OAUTH_ADAPTER,module);
            instantiateMethod.body()._return(ExpressionFactory._new(oauthAdapterClass.topLevelClass()).arg(ExpressionFactory._this()));
        }

    }

    private void generateSetCustomStateProperties(Module module, GeneratedClass oauthClientFactoryClass, GeneratedClass adapterClass, GeneratedClass stateClass) {
        GeneratedMethod setCustomStatePropertiesMethod = oauthClientFactoryClass.method(Modifier.PROTECTED, ctx().getCodeModel().VOID,"setCustomStateProperties");
        setCustomStatePropertiesMethod.annotate(Override.class);
        GeneratedVariable adapter = setCustomStatePropertiesMethod.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable state = setCustomStatePropertiesMethod.param(ref(OAuthState.class), "state");
        GeneratedVariable connector = setCustomStatePropertiesMethod.body().decl(adapterClass,"connector",ExpressionFactory.cast(adapterClass,adapter));

        for (OAuthCallbackParameterField field : ((OAuthModule) module).getCallbackParameters()) {
            setCustomStatePropertiesMethod.body().add(state.invoke("setCustomProperty").arg(field.getName()).arg(connector.invoke("get" + StringUtils.capitalize(field.getName()))));

        }
    }

    private void generateSetCustomAdapterPropertiesMethod(Module module, GeneratedClass oauthClientFactoryClass, GeneratedClass adapterClass, GeneratedClass stateClass, GeneratedField oauthManager) {
        GeneratedMethod setCustomAdapterPropertiesMethod = oauthClientFactoryClass.method(Modifier.PROTECTED,ctx().getCodeModel().VOID,"setCustomAdapterProperties");
        setCustomAdapterPropertiesMethod.annotate(Override.class);
        GeneratedVariable adapter = setCustomAdapterPropertiesMethod.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable state = setCustomAdapterPropertiesMethod.param(ref(OAuthState.class), "state");
        GeneratedVariable connector = setCustomAdapterPropertiesMethod.body().decl(adapterClass,"connector",ExpressionFactory.cast(adapterClass,adapter));

        for (OAuthCallbackParameterField field : ((OAuthModule) module).getCallbackParameters()) {
            setCustomAdapterPropertiesMethod.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName())).arg(
                    state.invoke("getCustomProperty").arg(field.getName())));
        }

        for (Field field: module.getConfigurableFields()) {
            if (((OAuthModule) module).getConsumerKeyField().equals(field) || ((OAuthModule) module).getConsumerSecretField().equals(field)) continue;
            setCustomAdapterPropertiesMethod.body().add(connector.invoke("set" + StringUtils.capitalize(field.getName())).arg(oauthManager.invoke("get" + StringUtils.capitalize(field.getName()))));
        }

    }



    private void generateGetAdapterClassMethod(Module module, GeneratedClass oauthClientFactoryClass, GeneratedClass adapterClass) {
        GeneratedMethod getAdapterClassMethod = oauthClientFactoryClass.method(Modifier.PROTECTED,ref(Class.class).narrow(ref(OAuth2Adapter.class).wildcard()),"getAdapterClass");
        getAdapterClassMethod.annotate(Override.class);
        if (module.hasRestCalls()) {
            getAdapterClassMethod.body()._return(ExpressionFactory.direct(module.getPackage().getName() + ".adapters."+ module.getClassName() + "RestClientAdapter.class"));
        } else {
            getAdapterClassMethod.body()._return(adapterClass.dotclass());
        }

    }

    private GeneratedField consumerField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(OAuthConsumer.class).name(CONSUMER_FIELD_NAME).build();
    }

    private void generateOAuthClientFactoryConstructor(GeneratedClass oauthManagerClass, GeneratedClass oauthClientFactoryClass) {
        GeneratedMethod oauthClientFactoryConstructor = oauthClientFactoryClass.constructor(Modifier.PUBLIC);
        GeneratedVariable constructorOAuthManager = oauthClientFactoryConstructor.param(ref(OAuth2Manager.class).narrow(ref(OAuth2Adapter.class)), "oauthManager");
        GeneratedVariable constructorObjectStore = oauthClientFactoryConstructor.param(ref(ObjectStore.class).narrow(ref(Serializable.class)),"objectStore");
        oauthClientFactoryConstructor.body().directStatement("super(oauthManager, objectStore);");
        oauthClientFactoryConstructor.body().directStatement(String.format("this.oauthManager = (%s) oauthManager;", oauthManagerClass.name()));
    }

    private GeneratedClass getOAuthManagerClass(OAuthModule module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);

        GeneratedClass oauthManagerClass;
        if (OAuthVersion.V2.equals(module.getOAuthVersion())) {
             oauthManagerClass = pkg._class(module.getClassName() + OAuthClientNamingConstants.OAUTH_MANAGER_CLASS_NAME_SUFFIX, ref(BaseOAuth2Manager.class).narrow(OAuth2Adapter.class));
        } else {
            oauthManagerClass = pkg._class(module.getClassName() + OAuthClientNamingConstants.OAUTH_MANAGER_CLASS_NAME_SUFFIX, ref(BaseOAuth1Manager.class));
        }

        ctx().registerProduct(Product.OAUTH_MANAGER, module, oauthManagerClass);

        oauthManagerClass.javadoc().add("A {@code " + oauthManagerClass.name() + "} is a wrapper around ");
        oauthManagerClass.javadoc().add(ref(module.asTypeMirror()));
        oauthManagerClass.javadoc().add(" that adds access token management capabilities to the pojo.");

        return oauthManagerClass;
    }

    private GeneratedClass getOAuthClientFactoryClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);
        GeneratedClass objectFactory = pkg._class(Modifier.PUBLIC, module.getClassName() + OAuthClientNamingConstants.OAUTH_CLIENT_FACTORY_CLASS_NAME_SUFFIX);
        objectFactory._extends(ref(BaseOAuthClientFactory.class));
        return objectFactory;
    }

    protected void generateMetadataConstantsAndGetters(Module module, GeneratedClass generatedClass) {
        String connectorName = module.getFriendlyName();
        if (org.mule.util.StringUtils.isEmpty(connectorName)) {
            connectorName = module.getModuleName();
        }

        GeneratedField moduleName = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("MODULE_NAME").initialValue(connectorName).build();
        GeneratedField moduleVersion = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("MODULE_VERSION").initialValue(ctx().getMavenInformation().getVersion()).build();
        GeneratedField devkitVersion = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("DEVKIT_VERSION").initialValue(ctx().getManifest().getProductVersion()).build();
        GeneratedField devkitBuild = FieldBuilder.newConstantFieldBuilder(generatedClass).type(String.class).name("DEVKIT_BUILD").initialValue(ctx().getManifest().getBuildNumber()).build();

        generateGetter(generatedClass, moduleName, "getModuleName");
        generateGetter(generatedClass, moduleVersion, "getModuleVersion");
        generateGetter(generatedClass, devkitVersion, "getDevkitVersion");
        generateGetter(generatedClass, devkitBuild, "getDevkitBuild");
    }

    private void generateGetter(GeneratedClass metadataAdapter, GeneratedField field, String getterMethodName) {
        GeneratedMethod getter = metadataAdapter.method(Modifier.PUBLIC, ref(String.class), getterMethodName);
        getter.body()._return(field);
    }
}