/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.oauth;

import org.mule.api.annotations.oauth.OAuth;
import org.mule.api.devkit.ProcessTemplate;
import org.mule.common.security.oauth.exception.UnableToAcquireAccessTokenException;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedField;
import org.mule.devkit.model.code.GeneratedInvocation;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.TypeVariable;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.security.oauth.OAuth1Adapter;
import org.mule.security.oauth.OAuth1Manager;
import org.mule.security.oauth.OnNoTokenPolicy;
import org.mule.security.oauth.process.OAuthProcessTemplate;

import oauth.signpost.OAuthConsumer;
import oauth.signpost.signature.AuthorizationHeaderSigningStrategy;
import oauth.signpost.signature.HmacSha1MessageSigner;
import oauth.signpost.signature.OAuthMessageSigner;
import oauth.signpost.signature.SigningStrategy;

public class OAuth1ClientAdapterGenerator extends AbstractOAuthAdapterGenerator {



    @Override
    public boolean shouldGenerate(Module module) {
        return false;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass oauthAdapter = getOAuthAdapterClass(module, "OAuth1Adapter", ref(OAuth1Adapter.class));
        oauthAdapter._implements(OAuth1Adapter.class);

        OAuthModule oAuthModule = (OAuthModule) module;
        OAuth oauth = module.getAnnotation(OAuth.class);

        GeneratedClass oauthManagerClass = ctx().getProduct(Product.OAUTH_MANAGER, module);
        GeneratedField oauthManager = createOauthManager(oauthAdapter);
        GeneratedField requestToken = requestTokenField(oauthAdapter);
        GeneratedField requestTokenSecret = requestTokenSecretField(oauthAdapter);
        GeneratedField oauthVerifier = authorizationCodeField(oauthAdapter);
        GeneratedField saveAccessTokenCallback = saveAccessTokenCallbackField(oauthAdapter);
        GeneratedField restoreAccessTokenCallback = restoreAccessTokenCallbackField(oauthAdapter);

        GeneratedField authorizationUrl = generateAuthorizationUrlField(oauthAdapter);
        GeneratedField accessTokenUrl = generateAccessTokenUrlField(oauthAdapter);
        GeneratedField requestTokenUrl = generateRequestTokenUrlField(oauthAdapter);
        GeneratedField onNoTokenPolicy = generateOnNoTokenPolicy(oauthAdapter);

        generateAdapterInterfaceImplementations(oauthAdapter, module);

        generateCallbackGettersAndSetters(oauthAdapter);
        generateName(oauthAdapter);


        consumerField(oauthAdapter);


        // Constructor
        GeneratedMethod constructor = oauthAdapter.constructor(Modifier.PUBLIC);
        GeneratedVariable oauthManagerParam = constructor.param(ref(OAuth1Manager.class), "oauthManager");
        constructor.body().assign(ExpressionFactory._this().ref(oauthManager),oauthManagerParam);



        generateOAuth1AuthorizeMethod(oauthAdapter, oauthManager);
        generateRestoreAccessTokenMethod(oauthAdapter, oauthManager);
        generateFetchAccessTokenMethod(oauthAdapter, oauthManager);
        generateGetOAuth1ManagerMethod(oauthAdapter,oauthManager);

        generateHasBeenAuthorizedMethod(oAuthModule, oauthAdapter);

        generateGetProcessTemplateMethod(oAuthModule, oauthAdapter, ctx().<GeneratedClass>getProduct(Product.CAPABILITIES_ADAPTER, module));

        generateResetMethod(oAuthModule, oauthAdapter);
    }

    private void generateAdapterInterfaceImplementations(GeneratedClass oauthAdapter, Module module) {
        OAuthModule oAuthModule = (OAuthModule) module;
        if (!oAuthModule.hasMethodWithName("getConsumerKey")) {
            GeneratedMethod getConsumerKeyMethod = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getConsumerKey");
            getConsumerKeyMethod.annotate(Override.class);
            getConsumerKeyMethod.body()._return(ExpressionFactory._this().invoke(oAuthModule.getConsumerKeyField().getGetter().getName()));
        }

        if (!oAuthModule.hasMethodWithName("getConsumerSecret")) {
            GeneratedMethod getConsumerSecret = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getConsumerSecret");
            getConsumerSecret.annotate(Override.class);
            getConsumerSecret.body()._return(ExpressionFactory._this().invoke(oAuthModule.getConsumerSecretField().getGetter().getName()));
        }


        if (!oAuthModule.hasMethodWithName("getAccessToken")) {
            GeneratedMethod getAccessToken = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getAccessToken");
            getAccessToken.annotate(Override.class);
            getAccessToken.body()._return(ExpressionFactory._this().ref(oAuthModule.getAccessTokenField().getGetter().getName()));
        }


        if (!oAuthModule.hasMethodWithName("getScope")) {
            GeneratedMethod getScope = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getScope");
            getScope.annotate(Override.class);
            if (oAuthModule.getScopeField() != null){
                getScope.body()._return(ExpressionFactory._this().invoke(oAuthModule.getScopeField().getGetter().getName()));
            } else {
                getScope.body()._return(ExpressionFactory.lit(""));
            }
        }

        if (!oAuthModule.hasMethodWithName("getVerifierRegex")) {
            GeneratedMethod getVerifierRegex = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getVerifierRegex");
            getVerifierRegex.annotate(Override.class);
            getVerifierRegex.body()._return(ExpressionFactory.lit(oAuthModule.getVerifierRegex()));
        }

        if (!oAuthModule.hasMethodWithName("getMessageSigner")) {
            GeneratedMethod getMessageSigner = oauthAdapter.method(Modifier.PUBLIC, ref(OAuthMessageSigner.class), "getMessageSigner");
            getMessageSigner.annotate(Override.class);
            getMessageSigner.body()._return(ExpressionFactory._new(ref(HmacSha1MessageSigner.class)));
        }

        if (!oAuthModule.hasMethodWithName("getSigningStrategy")) {
            GeneratedMethod getSigningStrategy = oauthAdapter.method(Modifier.PUBLIC, ref(SigningStrategy.class), "getSigningStrategy");
            getSigningStrategy.annotate(Override.class);
            getSigningStrategy.body()._return(ExpressionFactory._new(ref(AuthorizationHeaderSigningStrategy.class)));
        }

    }

    private void generateName(GeneratedClass oauthAdapter) {
        oauthAdapter.direct("String name;\n" +
                "    \n" +
                "    public String getName() {\n" +
                "        return name;\n" +
                "    }\n" +
                "\n" +
                "    public void setName(String name) {\n" +
                "        this.name = name;\n" +
                "    }");
    }

    private void generateCallbackGettersAndSetters(GeneratedClass oauthAdapter) {
        oauthAdapter.direct(" public Integer getLocalPort() {\n" +
                "        return oauthManager.getLocalPort();\n" +
                "    }\n" +
                "\n" +
                "    public Integer getRemotePort() {\n" +
                "        return oauthManager.getRemotePort();\n" +
                "    }\n" +
                "\n" +
                "    public String getDomain() {\n" +
                "        return oauthManager.getDomain();\n" +
                "    }\n" +
                "\n" +
                "    public String getPath() {\n" +
                "        return oauthManager.getPath();\n" +
                "    }\n" +
                "\n" +
                "    public org.mule.api.transport.Connector getConnector() {\n" +
                "        return oauthManager.getConnector();\n" +
                "    }\n" +
                "\n" +
                "    public void setConnector(org.mule.api.transport.Connector value) {\n" +
                "        oauthManager.setConnector(value);\n" +
                "    }\n" +
                "\n" +
                "    public Boolean getAsync() {\n" +
                "        return oauthManager.getAsync();\n" +
                "    }\n" +
                "\n" +
                "    public void setAsync(Boolean async) {\n" +
                "        oauthManager.setAsync(async);\n" +
                "    }\n" +
                "\n" +
                "    public void setLocalPort(Integer value) {\n" +
                "        oauthManager.setLocalPort(value);\n" +
                "    }\n" +
                "\n" +
                "    public void setRemotePort(Integer value) {\n" +
                "        oauthManager.setRemotePort(value);\n" +
                "    }\n" +
                "\n" +
                "    public void setDomain(String value) {\n" +
                "        oauthManager.setDomain(value);\n" +
                "    }\n" +
                "\n" +
                "    public void setPath(String value) {\n" +
                "        oauthManager.setPath(value);\n" +
                "    }");
    }

    private void generatePathSetter(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod setPathMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setPath");
        GeneratedVariable value = setPathMethod.param(ref(String.class),"value");
        setPathMethod.body().add(oauthManager.invoke("setPath").arg(value));
    }

    private void generateSetDomain(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod setDomainMethod = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setDomain");
        GeneratedVariable value = setDomainMethod.param(ref(String.class),"value");
        setDomainMethod.body().add(oauthManager.invoke("setDomain").arg(value));
    }

    private void generateSetRemotePort(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod setRemotePortMethod = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setRemotePort");
        GeneratedVariable value = setRemotePortMethod.param(ref(Integer.class),"value");
        setRemotePortMethod.body().add(oauthManager.invoke("setRemotePort").arg(value));
    }

    private void generateSetLocalPort(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod setLocalPortMethod = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setLocalPort");
        GeneratedVariable value = setLocalPortMethod.param(ref(Integer.class),"value");
        setLocalPortMethod.body().add(oauthManager.invoke("setLocalPort").arg(value));
    }

    private void generateGetOAuth1ManagerMethod(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod getOAuth1ManagerMethod = oauthAdapter.method(Modifier.PUBLIC,ref(OAuth1Manager.class),"getOauth1Manager");
        getOAuth1ManagerMethod.body()._return(oauthManager);
        getOAuth1ManagerMethod.annotate(Override.class);
    }

    private GeneratedField createOauthManager(GeneratedClass oauthAdapter)
    {
        GeneratedField oauthManager = oauthAdapter.field(Modifier.PRIVATE, ref(OAuth1Manager.class), "oauthManager");
        oauthAdapter.getter(oauthManager);
        oauthAdapter.setter(oauthManager);
        return oauthManager;
    }

    private GeneratedField generateOnNoTokenPolicy(GeneratedClass oauthAdapter)
    {
        FieldBuilder fieldBuilder = new FieldBuilder(oauthAdapter);
        fieldBuilder.initialValue(ref(OnNoTokenPolicy.class).staticRef("EXCEPTION"));
        GeneratedMethod setOnNoToken = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID,"setOnNoToken");
        GeneratedField result = fieldBuilder.type(OnNoTokenPolicy.class).name(ON_NO_TOKEN_POLICY_FIELD_NAME).getterAndSetter().build();
        GeneratedVariable value = setOnNoToken.param(ref(OnNoTokenPolicy.class),"value");
        setOnNoToken.body().add(ExpressionFactory.invoke("setOnNoTokenPolicy").arg(value));
        return result;
    }

    private GeneratedField requestTokenField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(REQUEST_TOKEN_FIELD_NAME).getterAndSetter().build();
    }

    private GeneratedField requestTokenSecretField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(REQUEST_TOKEN_SECRET_FIELD_NAME).getterAndSetter().build();
    }

    private GeneratedField consumerField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(OAuthConsumer.class).name(CONSUMER_FIELD_NAME).getterAndSetter().build();
    }



    private void generateRestoreAccessTokenMethod( GeneratedClass oauthAdapter,GeneratedField oauthManager) {
        GeneratedMethod restoreAccessTokenMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, "restoreAccessToken");

        restoreAccessTokenMethod.body()._return(oauthManager.invoke("restoreAccessToken").arg(ExpressionFactory._this()));
    }

    private void generateFetchAccessTokenMethod(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod fetchAccessToken = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.FETCH_ACCESS_TOKEN_METHOD_NAME);
        GeneratedVariable requestTokenUrl = fetchAccessToken.param(ref(String.class), "requestTokenUrl");
        GeneratedVariable accessTokenUrl = fetchAccessToken.param(ref(String.class), "accessTokenUrl");
        GeneratedVariable authorizationUrl = fetchAccessToken.param(ref(String.class), "authorizationUrl");
        GeneratedVariable redirectUri = fetchAccessToken.param(ref(String.class), "redirectUri");
        fetchAccessToken._throws(ref(UnableToAcquireAccessTokenException.class));

        fetchAccessToken.body().add(oauthManager.invoke("fetchAccessToken")
                .arg(ExpressionFactory._this())
                .arg(requestTokenUrl)
                .arg(accessTokenUrl)
                .arg(authorizationUrl)
                .arg(redirectUri));
//        fetchAccessToken.body().invoke("restoreAccessToken");
    }

    private void generateResetMethod(OAuthModule module, GeneratedClass oauthAdapter) {
        GeneratedMethod reset = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, AbstractOAuthAdapterGenerator.RESET_METHOD_NAME);
        reset.body().invoke(module.getAccessTokenField().getSetter().getName()).arg(ExpressionFactory._null());
        reset.body().invoke(module.getAccessTokenSecretField().getSetter().getName()).arg(ExpressionFactory._null());
        GeneratedField consumer = oauthAdapter.fields().get(CONSUMER_FIELD_NAME);
        reset.body().add(consumer.invoke("setTokenWithSecret").arg(ExpressionFactory._null()).arg(ExpressionFactory._null()));
    }

    protected void generateGetProcessTemplateMethod(OAuthModule module, GeneratedClass oauthAdapterClass, GeneratedClass capabilitiesAdapterClass) {
        GeneratedMethod getProcessTemplate = oauthAdapterClass.method(Modifier.PUBLIC, ref(ProcessTemplate.class), "getProcessTemplate");
        getProcessTemplate.annotate(ref(Override.class));
        TypeVariable p = getProcessTemplate.generify("P");
        getProcessTemplate.type(ref(ProcessTemplate.class).narrow(p).narrow(capabilitiesAdapterClass));


        GeneratedInvocation newProcessTemplate = ExpressionFactory._new(ref(OAuthProcessTemplate.class)).arg(ExpressionFactory._this());
        getProcessTemplate.body()._return(newProcessTemplate);
    }
}