package org.mule.devkit.oauth.generation.manager.factory;

import org.mule.api.store.ObjectStore;
import org.mule.common.security.oauth.OAuthState;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthCallbackParameterField;
import org.mule.devkit.model.module.oauth.OAuthCapability;
import org.mule.devkit.oauth.generation.AbstractOAuthAdapterGenerator;
import org.mule.devkit.oauth.generation.OAuthClientNamingConstants;
import org.mule.devkit.generation.utils.OAuth2StrategyUtilsResolver;
import org.mule.devkit.utils.NameUtils;
import org.mule.security.oauth.BaseOAuthClientFactory;
import org.mule.security.oauth.OAuth2Adapter;
import org.mule.security.oauth.OAuth2Manager;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;

/**
 * OAuthClientFactory generator (methods and fields for the class generation)
 *
 * It assumes that, at least, the generated classes for Adapters and Managers are already in place
 * (check the {@link #generate(org.mule.devkit.model.module.Module)} method when looks for adapter and manager classes) .
 *
 * As the time of writing this javadoc, this generator could be called when using @OAuth2 annotation with the @Connector,
 * or when using @OAuth2 annotation in a different component (check the calls to OAuth2StrategyUtilsResolver object).
 *
 */
public class OAuthClientFactoryGenerator extends AbstractOAuthAdapterGenerator {

    @Override
    public boolean shouldGenerate(Module module) {
        return false;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        OAuthCapability oAuthCapability = OAuth2StrategyUtilsResolver.getOAuthCapability(module);

        GeneratedClass adapterClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedClass oAuthManagerClass =ctx().getProduct(Product.OAUTH_MANAGER, module);

        GeneratedClass oauthClientFactoryClass = getOAuthClientFactoryClass(module);
        GeneratedField oauthManager = generateOAuthManagerClientFactoryField(oauthClientFactoryClass,oAuthManagerClass);
        generateOAuthClientFactoryConstructor(oAuthManagerClass, oauthClientFactoryClass);
        generateGetAdapterClassMethod(module, oauthClientFactoryClass, adapterClass);
        generateSetCustomAdapterPropertiesMethod(module, oauthClientFactoryClass, adapterClass, oauthManager, oAuthCapability);
        generateSetCustomStateProperties(module, oauthClientFactoryClass, adapterClass, oAuthCapability);
    }

    /**
     * OAuthClientFactory generator.
     * Class declaration
     */
    private GeneratedClass getOAuthClientFactoryClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);
        GeneratedClass objectFactory = pkg._class(Modifier.PUBLIC, module.getClassName() + OAuthClientNamingConstants.OAUTH_CLIENT_FACTORY_CLASS_NAME_SUFFIX);
        objectFactory._extends(ref(BaseOAuthClientFactory.class));

        ctx().registerProduct(Product.OAUTH2_FACTORY, module, objectFactory);
        return objectFactory;
    }

    /**
     * OAuthClientFactory generator.
     * Constructor generation to fulfil the inheritance of {@link BaseOAuthClientFactory#BaseOAuthClientFactory(org.mule.security.oauth.OAuth2Manager, org.mule.api.store.ObjectStore)}
     */
    private void generateOAuthClientFactoryConstructor(GeneratedClass oauthManagerClass, GeneratedClass oauthClientFactoryClass) {
        GeneratedMethod oauthClientFactoryConstructor = oauthClientFactoryClass.constructor(Modifier.PUBLIC);
        GeneratedVariable constructorOAuthManager = oauthClientFactoryConstructor.param(ref(OAuth2Manager.class).narrow(ref(OAuth2Adapter.class)), "oauthManager");
        GeneratedVariable constructorObjectStore = oauthClientFactoryConstructor.param(ref(ObjectStore.class).narrow(ref(Serializable.class)),"objectStore");
        oauthClientFactoryConstructor.body().directStatement("super(oauthManager, objectStore);");
        oauthClientFactoryConstructor.body().directStatement(String.format("this.oauthManager = (%s) oauthManager;", oauthManagerClass.name()));
    }

    /**
     * OAuthClientFactory generator.
     * Field to contain the concrete manager that has picked up all the values from the XML (thanks to the Definition Parser)
     */
    private GeneratedField generateOAuthManagerClientFactoryField(GeneratedClass oauthClientFactoryClass, GeneratedClass oauthManagerClass) {
        return oauthClientFactoryClass.field(Modifier.PRIVATE,oauthManagerClass,"oauthManager");
    }

    /**
     * OAuthClientFactory generator.
     * Constructor generation to fulfil template of {@link BaseOAuthClientFactory#setCustomStateProperties(org.mule.security.oauth.OAuth2Adapter, org.mule.common.security.oauth.OAuthState)}
     */
    private void generateSetCustomStateProperties(Module module, GeneratedClass oauthClientFactoryClass, GeneratedClass adapterClass, OAuthCapability oAuthCapability) {
        GeneratedMethod setCustomStatePropertiesMethod = oauthClientFactoryClass.method(Modifier.PROTECTED, ctx().getCodeModel().VOID,"setCustomStateProperties");
        setCustomStatePropertiesMethod.annotate(Override.class);
        GeneratedVariable adapter = setCustomStatePropertiesMethod.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable state = setCustomStatePropertiesMethod.param(ref(OAuthState.class), "state");
        GeneratedVariable connector = setCustomStatePropertiesMethod.body().decl(adapterClass,"connector",ExpressionFactory.cast(adapterClass,adapter));

        for (OAuthCallbackParameterField field : oAuthCapability.getCallbackParameters()) {
            GeneratedExpression castedConnector = OAuth2StrategyUtilsResolver.getOAuthConcreteComponent(module, connector, ctx());
            setCustomStatePropertiesMethod.body().add(state.invoke("setCustomProperty").arg(field.getName()).arg(castedConnector.invoke(NameUtils.buildGetter(field.getName()))));
        }
    }

    /**
     * OAuthClientFactory generator.
     * Constructor generation to fulfil template of {@link BaseOAuthClientFactory#setCustomAdapterProperties(org.mule.security.oauth.OAuth2Adapter, org.mule.common.security.oauth.OAuthState)}
     */
    private void generateSetCustomAdapterPropertiesMethod(Module module, GeneratedClass oauthClientFactoryClass, GeneratedClass adapterClass, GeneratedField oauthManager, OAuthCapability oAuthCapability) {
        GeneratedMethod setCustomAdapterPropertiesMethod = oauthClientFactoryClass.method(Modifier.PROTECTED,ctx().getCodeModel().VOID,"setCustomAdapterProperties");
        setCustomAdapterPropertiesMethod.annotate(Override.class);
        GeneratedVariable adapter = setCustomAdapterPropertiesMethod.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable state = setCustomAdapterPropertiesMethod.param(ref(OAuthState.class), "state");
        GeneratedVariable connector = setCustomAdapterPropertiesMethod.body().decl(adapterClass,"connector",ExpressionFactory.cast(adapterClass,adapter));

        for (OAuthCallbackParameterField field : oAuthCapability.getCallbackParameters()) {
            GeneratedExpression castedConnector = OAuth2StrategyUtilsResolver.getOAuthConcreteComponent(module, connector, ctx());
            setCustomAdapterPropertiesMethod.body().add(castedConnector.invoke(NameUtils.buildSetter(field.getName())).arg(
                    state.invoke("getCustomProperty").arg(field.getName())));
        }

        /**
         * Setting the configurables of the @Connector
         */
        for (Field field: module.getConfigurableFields()) {
            setConfigurableOn(connector, oAuthCapability, setCustomAdapterPropertiesMethod, oauthManager, field);
        }
        if (OAuth2StrategyUtilsResolver.hasOAuth2Component(module)){
            for (Field field: module.manager().oauth2Component().get().getConfigurableFields()){
                GeneratedExpression castedConnector = OAuth2StrategyUtilsResolver.getOAuthConcreteComponent(module, connector, ctx());
                setConfigurableOn(castedConnector, oAuthCapability, setCustomAdapterPropertiesMethod, oauthManager, field);
            }
        }
    }

    /**
     * Given a {@code connectorOrStrategy} object, it will set the property of the @Configurable field, whether if
     * it's for the current module (aka: @Connector) or if it's for the component (OAuth strategy).
     *
     * The only constraints in this method is it will not set the properties as a normal {@link org.mule.api.annotations.Configurable}
     * if the specified field is a {@link org.mule.api.annotations.oauth.OAuthConsumerKey} or a {@link org.mule.api.annotations.oauth.OAuthConsumerSecret}
     * as both fields must be filled by the OAuth contract under {@link org.mule.security.oauth.BaseOAuth2Manager#createAdapter(String)}, rather
     * than in the generated factory this generator is creating.
     *
     * @param connectorOrStrategy current module (@Connector/@Module) or strategy (@Oauth2)
     * @param oAuthCapability
     * @param setCustomAdapterPropertiesMethod
     * @param oauthManager
     * @param field
     */
    private void setConfigurableOn(GeneratedExpression connectorOrStrategy, OAuthCapability oAuthCapability, GeneratedMethod setCustomAdapterPropertiesMethod, GeneratedField oauthManager, Field field) {
        Set<Field> skippedFields = new HashSet<Field>();
        skippedFields.add(oAuthCapability.getConsumerKeyField());
        skippedFields.add(oAuthCapability.getConsumerSecretField());

        if (skippedFields.contains(field)) {
            //if the field to work with is the consumer key or the consumer secret, then we abort adding it to the factory
            return;
        }
        GeneratedInvocation oauthManagerConfigurableInvoker = oauthManager.invoke(NameUtils.buildGetter(field.getName()));
        if (field.equals(oAuthCapability.getScopeField())){
            /**
             * We need to support the same as {@link org.mule.devkit.oauth.generation.manager.AbstractOAuth2ManagerGenerator#generateConfigurableFields(org.mule.devkit.model.module.Module, org.mule.devkit.model.code.GeneratedClass, OAuthCapability, org.mule.devkit.model.code.GeneratedClass)}
             */
            oauthManagerConfigurableInvoker = oauthManager.invoke("getScope");
        }

        setCustomAdapterPropertiesMethod.body().add(
                connectorOrStrategy.invoke(NameUtils.buildSetter(field.getName())).arg(oauthManagerConfigurableInvoker));

    }

    /**
     * OAuthClientFactory generator.
     * Implementation of {@link org.mule.security.oauth.BaseOAuthClientFactory#getAdapterClass()}
     */
    private void generateGetAdapterClassMethod(Module module, GeneratedClass oauthClientFactoryClass, GeneratedClass adapterClass) {
        GeneratedMethod getAdapterClassMethod = oauthClientFactoryClass.method(Modifier.PROTECTED,ref(Class.class).narrow(ref(OAuth2Adapter.class).wildcard()),"getAdapterClass");
        getAdapterClassMethod.annotate(Override.class);
        if (module.hasRestCalls()) {
            getAdapterClassMethod.body()._return(ExpressionFactory.direct(module.getPackage().getName() + ".adapters."+ module.getClassName() + "RestClientAdapter.class"));
        } else {
            getAdapterClassMethod.body()._return(adapterClass.dotclass());
        }
    }

}
