package org.mule.devkit.oauth.generation.manager;

import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.ModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.oauth.generation.AbstractOAuthAdapterGenerator;
import org.mule.devkit.oauth.generation.OAuthClientNamingConstants;

import java.util.Arrays;
import java.util.List;

/**
 * Abstract generator that specifies the template of the generated managers for a given connector (which could be OAuth1/OAuth2)
 */
public abstract class AbstractOAuthManagerGenerator extends AbstractOAuthAdapterGenerator implements ModuleGenerator {

    private final static List<Product> CONSUMES = Arrays.asList(Product.CAPABILITIES_ADAPTER,
            Product.LIFECYCLE_ADAPTER,
            Product.MANAGED_ACCESS_TOKEN_PROCESS_TEMPLATE,
            Product.METADATA_ADAPTER,
            Product.PROCESS_ADAPTER
    );
    private final static List<Product> PRODUCES = Arrays.asList(Product.OAUTH_MANAGER, Product.OAUTH_ADAPTER);


    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    /**
     * Template method that subclasses must not override
     *
     * @param module
     * @throws GenerationException
     */
    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass oAuthManagerClass = getOAuthManagerClass(module, getExtendManagerClass());
        generateLoggerMethod(oAuthManagerClass);
        callGenerator(module, getAdapterGenerator());
        doGenerate(module, oAuthManagerClass);
    }

    /**
     * Returns the concrete class to be used a superclass of the generated Manager (either for OAuth1 or OAuth2)
     *
     * @return
     */
    protected abstract TypeReference getExtendManagerClass();

    /**
     * Returns the concrete object to generate the adapter (either for OAuth1 or OAuth2)
     * TODO: It must be refactored to avoid calling a generator from another generator, but this will imply adding more dependencies in the CONSUMES/PRODUCES mechanism
     * @return
     */
    protected abstract AbstractOAuthAdapterGenerator getAdapterGenerator();

    /**
     * Hook method to implement the custom core behaviour of the manager anything the concrete object to generate the adapter (either for OAuth1 or OAuth2)
     *
     * @param module
     * @param oAuthManagerClass
     * @throws GenerationException
     */
    protected abstract void doGenerate(Module module, GeneratedClass oAuthManagerClass) throws GenerationException;


    /**
     *  Generation of the adapter, code lives here to reduce the dependency between generators (Adapter vs Manager..).
     *  TODO: It must be refactored to avoid calling a generator from another generator, but this will imply adding more dependencies in the CONSUMES/PRODUCES mechanism
     *
     * @param module
     * @param oauthAdapterGenerator
     * @throws GenerationException
     */
    protected void callGenerator(Module module, AbstractOAuthAdapterGenerator oauthAdapterGenerator) throws GenerationException {
        ctx().warn("Calling a generator from within a generator. This is not a good practice and should be changed");
        oauthAdapterGenerator.setCtx(ctx());
        oauthAdapterGenerator.generate(module);
    }

    /**
     * Given a module with OAuth capabilities, the class that the generated manager must extends varies if it's OAuth1 or OAuth2
     * implementation. Thus, each hook method {@link #getExtendManagerClass()} must return the specific class the manager is
     * going to extend.
     *
     * @param module
     * @param superClass hook provided from {@link #getExtendManagerClass()}
     * @return
     */
    private GeneratedClass getOAuthManagerClass(Module module, TypeReference superClass) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);

        GeneratedClass oauthManagerClass;
        oauthManagerClass = pkg._class(module.getClassName() + OAuthClientNamingConstants.OAUTH_MANAGER_CLASS_NAME_SUFFIX, superClass);

        ctx().registerProduct(Product.OAUTH_MANAGER, module, oauthManagerClass);

        oauthManagerClass.javadoc().add("A {@code " + oauthManagerClass.name() + "} is a wrapper around ");
        oauthManagerClass.javadoc().add(ref(module.asTypeMirror()));
        oauthManagerClass.javadoc().add(" that adds access token management capabilities to the pojo.");

        return oauthManagerClass;
    }

    /**
     * Implementation in the generated manager class to fulfill the contract within
     * {@link org.mule.security.oauth.BaseOAuth2Manager#getLogger()} if using {@link org.mule.api.annotations.oauth.OAuth2},
     * or {@link org.mule.security.oauth.BaseOAuth1Manager#getLogger()} if using {@link org.mule.api.annotations.oauth.OAuth}
     *
     * @param oauthManagerClass generated subclass of {@link org.mule.security.oauth.BaseOAuth2Manager} or {@link org.mule.security.oauth.BaseOAuth1Manager}
     */
    protected void generateLoggerMethod(GeneratedClass oauthManagerClass) {
        GeneratedField logger = generateLoggerField(oauthManagerClass);

        GeneratedMethod getLoggerMethod = oauthManagerClass.method(Modifier.PROTECTED, ref(org.slf4j.Logger.class),"getLogger");
        getLoggerMethod.annotate(Override.class);
        getLoggerMethod.body()._return(logger);
    }

}
