/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.devkit.oauth.generation.adapter;

import org.apache.commons.lang.StringUtils;

import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.context.MuleContextAware;
import org.mule.api.devkit.ProcessTemplate;
import org.mule.api.lifecycle.Startable;
import org.mule.api.lifecycle.Stoppable;
import org.mule.common.security.oauth.AuthorizationParameter;
import org.mule.common.security.oauth.exception.NotAuthorizedException;
import org.mule.common.security.oauth.exception.UnableToAcquireAccessTokenException;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.utils.OAuth2StrategyUtilsResolver;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.code.builders.FieldBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.components.connection.OAuth2Component;
import org.mule.devkit.model.module.oauth.OAuthAuthorizationParameter;
import org.mule.devkit.model.module.oauth.OAuthCapability;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.oauth.generation.AbstractOAuthAdapterGenerator;
import org.mule.devkit.utils.NameUtils;
import org.mule.security.oauth.OAuth2Adapter;
import org.mule.security.oauth.OAuth2Manager;

import com.google.common.base.Optional;

import java.util.*;
import java.util.regex.Pattern;

public class OAuth2ClientAdapterGenerator extends AbstractOAuthAdapterGenerator {

    @Override
    public boolean shouldGenerate(Module module) {
        return false;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass oauthAdapter = getOAuthAdapterClass(module, "OAuth2Adapter", ref(OAuth2Adapter.class));
        oauthAdapter._implements(OAuth2Adapter.class);

        OAuthCapability oAuthCapability;
        if (module.manager().oauth2Component().isPresent()){
            OAuth2Component oAuth2Component = module.manager().oauth2Component().get();
            GeneratedExpression oauth2ComponentReference = OAuth2StrategyUtilsResolver.getOAuthConcreteComponent(module, ExpressionFactory._super(), ctx());
            generateSetMuleContext(oauth2ComponentReference, oAuth2Component, oauthAdapter);
            generateStartable(oauth2ComponentReference, oAuth2Component, oauthAdapter);
            generateStoppable(oauth2ComponentReference, oAuth2Component, oauthAdapter);

            oAuthCapability = oAuth2Component;
        }else {
            oAuthCapability = (OAuthModule) module;
        }

        GeneratedClass oauthManagerClass = ctx().getProduct(Product.OAUTH_MANAGER, module);
        GeneratedField oauthManager = oauthAdapter.field(Modifier.PRIVATE,
                ref(OAuth2Manager.class).narrow(OAuth2Adapter.class), "oauthManager");

        generateConstructor(module, oauthAdapter, oauthManager, OAuth2Manager.class, OAuth2Adapter.class);

        accessTokenPatternConstant(oauthAdapter, oAuthCapability);
        refreshTokenPatternConstant(oauthAdapter, oAuthCapability);
        expirationPatternConstant(oauthAdapter, oAuthCapability);

        GeneratedField name = oauthAdapter.field(Modifier.PRIVATE, ref(String.class),"name");
        oauthAdapter.getter(name);
        oauthAdapter.setterOverride(name);

        generateOnNoTokenPolicyField(oauthAdapter);

        authorizationCodeField(oauthAdapter);
        refreshTokenField(oauthAdapter);

        GeneratedField redirectUri = oauthAdapter.field(Modifier.PUBLIC, ref(String.class), "redirectUri");
        GeneratedField authorizationUrl = generateAuthorizationUrlField(oauthAdapter);
        GeneratedField accessTokenUrl = generateAccessTokenUrlField(oauthAdapter);

        expirationField(oauthAdapter, oAuthCapability);

        generateInitialiseMethod(oauthAdapter);

        GeneratedField logger = FieldBuilder.newLoggerField(oauthAdapter);
        generateOAuth2AuthorizeMethod(oauthAdapter, oauthManager);
        generateFetchAccessTokenMethod(oauthAdapter, oauthManager);
        generateRefreshAccessTokenMethod(oauthAdapter, oauthManager);
        generateHasTokenExpiredMethod(oauthAdapter, oAuthCapability);
        generateResetMethod(oAuthCapability, oauthAdapter);
        generateHasBeenAuthorizedMethod(oAuthCapability, oauthAdapter);

        generateRegexpGetters(oauthAdapter);

        generateOAuth2ConnectorInterfaceImpl(module, oAuthCapability, oauthAdapter);

        generateGetProcessTemplateMethod(oauthAdapter, ctx().<GeneratedClass>getProduct(Product.CAPABILITIES_ADAPTER, module),oauthManager);
    }

    private void generateSetMuleContext(GeneratedExpression oauth2ComponentReference, OAuth2Component oAuth2Component, GeneratedClass oauthAdapter)
    {
        Optional<Field> muleContext = oAuth2Component.setMuleContext();
        if (muleContext.isPresent()){
            GeneratedMethod setMuleContextMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setMuleContext");
            setMuleContextMethod.annotate(Override.class);
            GeneratedVariable muleContextParam = setMuleContextMethod.param(ref(MuleContext.class), "muleContext");
            setMuleContextMethod.body().add(oauth2ComponentReference.invoke(muleContext.get().getSetter().getName()).arg(muleContextParam));
            setMuleContextMethod.body().add(ExpressionFactory._super().invoke("setMuleContext").arg(muleContextParam));
        }
    }

    private void generateStartable(GeneratedExpression oauth2ComponentReference, OAuth2Component oAuth2Component, GeneratedClass oauthAdapter)
    {
        Optional<Method> startable = oAuth2Component.startable();
        if (startable.isPresent()){
            GeneratedMethod startMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "start");
            startMethod.annotate(Override.class);
            startMethod._throws(ref(MuleException.class));
            startMethod.body().add(oauth2ComponentReference.invoke(startable.get().getName()));
            startMethod.body().add(ExpressionFactory._super().invoke("start"));
        }
    }

    private void generateStoppable(GeneratedExpression oauth2ComponentReference, OAuth2Component oAuth2Component, GeneratedClass oauthAdapter)
    {
        Optional<Method> stoppable = oAuth2Component.stoppable();
        if (stoppable.isPresent()){
            GeneratedMethod stopMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "stop");
            stopMethod.annotate(Override.class);
            stopMethod._throws(ref(MuleException.class));
            stopMethod.body().add(oauth2ComponentReference.invoke(stoppable.get().getName()));
            stopMethod.body().add(ExpressionFactory._super().invoke("stop"));
        }
    }

    protected void generateConstructor(Module module, GeneratedClass oauthAdapter, GeneratedField oauthManager, Class OAuthManager, Class OAuthAdapter) {
        GeneratedMethod constructor = oauthAdapter.constructor(Modifier.PUBLIC);
        GeneratedVariable oauthManagerParam = constructor.param(ref(OAuthManager).narrow(ref(OAuthAdapter)),"oauthManager");
        constructor.body().assign(ExpressionFactory._this().ref(oauthManager),oauthManagerParam);

        if (OAuth2StrategyUtilsResolver.hasOAuth2Component(module)) {
            constructor.body().invoke(NameUtils.buildSetter(module.getConfigStrategy().get().getName()))
                    .arg(ExpressionFactory._new(ref(module.manager().oauth2Component().get().asTypeMirror())));
        }
    }


    protected void generateHasBeenAuthorizedMethod(OAuthCapability oAuthCapability, GeneratedClass oauthAdapter) {
        GeneratedMethod hasBeenAuthorized = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "hasBeenAuthorized");
        hasBeenAuthorized._throws(ref(NotAuthorizedException.class));
        GeneratedBlock ifAccessTokenIsNull = hasBeenAuthorized.body()._if(ExpressionFactory.invoke("getAccessToken").isNull())._then();

        GeneratedInvocation newNotAuthorizedException = ExpressionFactory._new(ref(NotAuthorizedException.class));
        newNotAuthorizedException.arg("This connector has not yet been authorized, please authorize by calling \"authorize\".");

        ifAccessTokenIsNull._throw(newNotAuthorizedException);
    }

    private void generateOAuth2ConnectorInterfaceImpl(Module module, OAuthCapability oAuthModule, GeneratedClass oauthAdapter) {

        /**
         * impl of #getAuthorizationParameters()
         */
        GeneratedMethod getAuthorizationParameters = oauthAdapter.method(Modifier.PUBLIC,ref(Set.class).narrow(ref(AuthorizationParameter.class).narrow(ctx().getCodeModel().wildcard())),"getAuthorizationParameters");
        getAuthorizationParameters.annotate(Override.class);


        List<GeneratedVariable> generatedVariableList = new ArrayList<GeneratedVariable>();

        for (OAuthAuthorizationParameter parameter: oAuthModule.getAuthorizationParameters()) {
            GeneratedVariable param = getAuthorizationParameters.body().decl(ref(AuthorizationParameter.class).narrow(String.class),parameter.getName(),ExpressionFactory._new(ref(AuthorizationParameter.class).narrow(String.class))
                    .arg(ExpressionFactory.lit(parameter.getName()))
                    .arg(ExpressionFactory.lit(parameter.getDescription()))
                    .arg(ExpressionFactory.lit(parameter.isOptional()))
                    .arg(ExpressionFactory.lit(parameter.getDefaultValue()))
                    .arg(ref(parameter.getType().asTypeMirror()).dotclass()));
            generatedVariableList.add(param);
        }

        GeneratedVariable result = getAuthorizationParameters.body().decl(ref(Set.class).narrow(ref(AuthorizationParameter.class).narrow(ctx().getCodeModel().wildcard()))
                ,"result"
                ,ExpressionFactory._new(ref(HashSet.class).narrow(ref(AuthorizationParameter.class).narrow(ctx().getCodeModel().wildcard()))));

        for (GeneratedVariable param: generatedVariableList) {
            getAuthorizationParameters.body().add(result.invoke("add").arg(param));
        }

        getAuthorizationParameters.body()._return(result);


        GeneratedMethod getConsumerKeyMethod = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getConsumerKey");
        getConsumerKeyMethod.annotate(Override.class);

        GeneratedExpression oauth2ComponentReference = OAuth2StrategyUtilsResolver.getOAuthConcreteComponent(module, ExpressionFactory._super(), ctx());

        getConsumerKeyMethod.body()._return(oauth2ComponentReference.invoke(oAuthModule.getConsumerKeyField().getGetter().getName()));

        GeneratedMethod setConsumerKey = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setConsumerKey");
        GeneratedVariable valueCK = setConsumerKey.param(ref(String.class),"value");
        setConsumerKey.annotate(Override.class);
        setConsumerKey.body().add(oauth2ComponentReference.invoke(oAuthModule.getConsumerKeyField().getSetter().getName()).arg(valueCK));

        GeneratedMethod getConsumerSecret = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getConsumerSecret");
        getConsumerSecret.annotate(Override.class);
        getConsumerSecret.body()._return(oauth2ComponentReference.invoke(oAuthModule.getConsumerSecretField().getGetter().getName()));

        GeneratedMethod setConsumerSecret = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setConsumerSecret");
        GeneratedVariable valueCS = setConsumerSecret.param(ref(String.class),"value");
        setConsumerSecret.annotate(Override.class);
        setConsumerSecret.body().add(oauth2ComponentReference.invoke(oAuthModule.getConsumerSecretField().getSetter().getName()).arg(valueCS));

        GeneratedMethod getAccessToken = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getAccessToken");
        getAccessToken.annotate(Override.class);
        getAccessToken.body()._return(oauth2ComponentReference.invoke(oAuthModule.getAccessTokenField().getGetter().getName()));

        GeneratedMethod setAccessToken = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setAccessToken");
        GeneratedVariable valueAT = setAccessToken.param(ref(String.class),"value");
        setAccessToken.annotate(Override.class);
        setAccessToken.body().add(oauth2ComponentReference.invoke(oAuthModule.getAccessTokenField().getSetter().getName()).arg(valueAT));

        GeneratedMethod getScope = oauthAdapter.method(Modifier.PUBLIC,ref(String.class),"getScope");
        getScope.annotate(Override.class);
        if (oAuthModule.getScopeField() != null){
            getScope.body()._return(oauth2ComponentReference.invoke(oAuthModule.getScopeField().getGetter().getName()));
        } else {
            getScope.body()._return(ExpressionFactory.lit(""));
        }

        if (oAuthModule.getScopeField() != null) {
            GeneratedMethod setScope = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setScope");
            GeneratedVariable valueS = setScope.param(ref(String.class),"value");
            setScope.body().add(oauth2ComponentReference.invoke(oAuthModule.getScopeField().getSetter().getName()).arg(valueS));
        }

        /**
         * impl of #postAuth()
         */
        GeneratedMethod postAuth = oauthAdapter.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"postAuth");
        postAuth.annotate(Override.class);
        if (oAuthModule.getPostAuthorizationMethod() != null) {
            //TODO: on mule fix uncomment
            postAuth._throws(Exception.class);
            postAuth.body().add(oauth2ComponentReference.invoke(oAuthModule.getPostAuthorizationMethod().getName()));
        }
    }

    private void generateRegexpGetters(GeneratedClass oauthAdapter) {
        oauthAdapter.direct("@Override\n" +
                "    public String getAccessTokenRegex() {\n" +
                "        return ACCESS_CODE_PATTERN.pattern();\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public String getExpirationRegex() {\n" +
                "        return EXPIRATION_TIME_PATTERN.pattern();\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public String getRefreshTokenRegex() {\n" +
                "        return REFRESH_TOKEN_PATTERN.pattern();\n" +
                "    }\n" +
                "\n" +
                "    @Override\n" +
                "    public String getVerifierRegex() {\n" +
                "        return oauthVerifier;\n" +
                "    }");
    }



    protected GeneratedField refreshTokenField(GeneratedClass oauthAdapter) {
        return new FieldBuilder(oauthAdapter).type(String.class).name(REFRESH_TOKEN_FIELD_NAME).getterAndSetter().build();
    }

    private void accessTokenPatternConstant(GeneratedClass oauthAdapter, OAuthCapability oAuthCapability) {
        GeneratedField accessTokenPattern = new FieldBuilder(oauthAdapter).type(Pattern.class).name(ACCESS_CODE_PATTERN_FIELD_NAME).staticField().finalField().
                initialValue(ref(Pattern.class).staticInvoke("compile").arg(oAuthCapability.getAccessTokenRegex())).build();
        GeneratedMethod getter = oauthAdapter.getter(accessTokenPattern);
        getter.name("getAccessCodePattern");
        getter.annotate(Override.class);
    }

    private void refreshTokenPatternConstant(GeneratedClass oauthAdapter, OAuthCapability oAuthCapability) {
        GeneratedField refreshTokenPattern = new FieldBuilder(oauthAdapter).type(Pattern.class).name(REFRESH_TOKEN_PATTERN_FIELD_NAME).staticField().finalField().
                initialValue(ref(Pattern.class).staticInvoke("compile").arg(oAuthCapability.getRefreshTokenRegex())).build();
        GeneratedMethod getter = oauthAdapter.getter(refreshTokenPattern);
        getter.name("getRefreshTokenPattern");
        getter.annotate(Override.class);
    }

    private void expirationPatternConstant(GeneratedClass oauthAdapter, OAuthCapability oAuthCapability) {
        if (!StringUtils.isEmpty(oAuthCapability.getExpirationRegex())) {
            GeneratedField expirationPattern = new FieldBuilder(oauthAdapter).type(Pattern.class).name(EXPIRATION_TIME_PATTERN_FIELD_NAME).staticField().finalField().
                    initialValue(ref(Pattern.class).staticInvoke("compile").arg(oAuthCapability.getExpirationRegex())).build();
            GeneratedMethod getter = oauthAdapter.getter(expirationPattern);
            getter.name("getExpirationTimePattern");
            getter.annotate(Override.class);
        }
    }

    private void expirationField(GeneratedClass oauthAdapter, OAuthCapability oAuthCapability) {
        if (!StringUtils.isEmpty(oAuthCapability.getExpirationRegex())) {
            new FieldBuilder(oauthAdapter).type(Date.class).name(EXPIRATION_FIELD_NAME).setter().build();
        }
    }

    private void generateRefreshAccessTokenMethod(GeneratedClass oauthAdapter,GeneratedField oauthManager) {
        GeneratedMethod refreshAccessTokenMethod = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "refreshAccessToken");
        GeneratedVariable accessTokenId = refreshAccessTokenMethod.param(String.class, "accessTokenId");
        refreshAccessTokenMethod._throws(ref(UnableToAcquireAccessTokenException.class));
        refreshAccessTokenMethod.annotate(Override.class);
//        GeneratedVariable accessTokenUrl = refreshAccessTokenMethod.param(ref(String.class), "accessTokenUrl");
        refreshAccessTokenMethod.body().add(oauthManager.invoke("refreshAccessToken").arg(ExpressionFactory._this()).arg(accessTokenId));
    }

    private void generateFetchAccessTokenMethod(GeneratedClass oauthAdapter, GeneratedField oauthManager) {
        GeneratedMethod fetchAccessToken = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "fetchAccessToken");
        GeneratedVariable redirectUri = fetchAccessToken.param(ref(String.class), "redirectUri");
        fetchAccessToken._throws(ref(UnableToAcquireAccessTokenException.class));
        fetchAccessToken.body().add(oauthManager.invoke("fetchAccessToken").arg(ExpressionFactory._this()).arg(redirectUri));

    }
    private void generateHasTokenExpiredMethod(GeneratedClass oauthAdapter, OAuthCapability oAuthCapability) {
        GeneratedMethod hasTokenExpired = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().BOOLEAN, HAS_TOKEN_EXPIRED_METHOD_NAME);
        if (!StringUtils.isEmpty(oAuthCapability.getExpirationRegex())) {
            GeneratedField expirationDate = oauthAdapter.fields().get(EXPIRATION_FIELD_NAME);
            hasTokenExpired.body()._return(Op.cand(
                    expirationDate.isNotNull(),
                    expirationDate.invoke("before").arg(ExpressionFactory._new(ref(Date.class)))));
        } else {
            hasTokenExpired.body()._return(ExpressionFactory.FALSE);
        }
    }

    private void generateResetMethod(OAuthCapability oAuthCapability, GeneratedClass oauthAdapter) {
        GeneratedMethod reset = oauthAdapter.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, RESET_METHOD_NAME);
        if (!StringUtils.isEmpty(oAuthCapability.getExpirationRegex())) {
            reset.body().assign(oauthAdapter.fields().get(EXPIRATION_FIELD_NAME), ExpressionFactory._null());
        }
        reset.body().assign(oauthAdapter.fields().get(VERIFIER_FIELD_NAME), ExpressionFactory._null());
        reset.body().invoke("setAccessToken").arg(ExpressionFactory._null());
    }

    protected void generateGetProcessTemplateMethod(GeneratedClass oauthAdapterClass, GeneratedClass capabilitiesAdapterClass, GeneratedField oauthManager) {
        GeneratedMethod getProcessTemplate = oauthAdapterClass.method(Modifier.PUBLIC, ref(ProcessTemplate.class), "getProcessTemplate");
        getProcessTemplate.annotate(ref(Override.class));
        TypeVariable p = getProcessTemplate.generify("P");
        getProcessTemplate.type(ref(ProcessTemplate.class).narrow(p).narrow(capabilitiesAdapterClass));

        getProcessTemplate.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)));
    }
}