/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.oauth.generation.config;

import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.module.oauth.OAuthCapability;
import org.mule.devkit.oauth.generation.AbstractOAuthAdapterGenerator;
import org.mule.devkit.oauth.generation.OAuthClientNamingConstants;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedPackage;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthAuthorizationParameter;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.model.schema.SchemaTypeConversion;
import org.mule.devkit.generation.utils.OAuth2StrategyUtilsResolver;
import org.mule.security.oauth.config.AbstractDevkitBasedDefinitionParser;

import java.util.Arrays;
import java.util.List;

import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;

public class AuthorizeBeanDefinitionParserGenerator extends AbstractOAuthAdapterGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(Product.MESSAGE_PROCESSOR);
    private final static List<Product> PRODUCES = Arrays.asList(Product.BEAN_DEFINITION_PARSER);

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule
                || OAuth2StrategyUtilsResolver.hasOAuth2Component(module);
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass beanDefinitionParser = getAuthorizeBeanDefinitionParserClass(module);
        GeneratedClass messageProcessorClass = ctx().getProduct(Product.MESSAGE_PROCESSOR, module, "authorize");

        GeneratedMethod parse = beanDefinitionParser.method(Modifier.PUBLIC, ref(BeanDefinition.class), "parse");
        GeneratedVariable element = parse.param(ref(org.w3c.dom.Element.class), "element");
        GeneratedVariable parserContext = parse.param(ref(ParserContext.class), "parserContent");

        GeneratedVariable builder = parse.body().decl(ref(BeanDefinitionBuilder.class), "builder",
                ref(BeanDefinitionBuilder.class).staticInvoke("rootBeanDefinition").arg(messageProcessorClass.dotclass().invoke("getName")));

        parse.body().invoke("parseConfigRef").arg(element).arg(builder);

        OAuthCapability oauthCapability = OAuth2StrategyUtilsResolver.getOAuthCapability(module);
        if (oauthCapability.getOAuthVersion() == OAuthVersion.V2) {
            parse.body().invoke("parseProperty").arg(builder).arg(element).arg("state");

        }

        if (oauthCapability.getAuthorizationParameters() != null) {
            for (OAuthAuthorizationParameter parameter : oauthCapability.getAuthorizationParameters()) {
                if (SchemaTypeConversion.isSupported(parameter.getType().asTypeMirror().toString()) || parameter.getType().isEnum()) {
                    parse.body().invoke("parseProperty").arg(builder).arg(element).arg(parameter.getName());
                }
            }
        }

        parse.body().invoke("parseProperty").arg(builder).arg(element).arg("authorizationUrl");
        parse.body().invoke("parseProperty").arg(builder).arg(element).arg("accessTokenUrl");
        parse.body().invoke("parseProperty").arg(builder).arg(element).arg("accessTokenId");

        if( oauthCapability.getOAuthVersion() == OAuthVersion.V10A ) {
            parse.body().invoke("parseProperty").arg(builder).arg(element).arg("requestTokenUrl");
        }

        GeneratedVariable definition = parse.body().decl(ref(BeanDefinition.class), "definition", builder.invoke("getBeanDefinition"));

        parse.body().invoke("setNoRecurseOnDefinition").arg(definition);

        parse.body().invoke("attachProcessorDefinition").arg(parserContext).arg(definition);

        parse.body()._return(definition);
    }

    private GeneratedClass getAuthorizeBeanDefinitionParserClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.CONFIG_NAMESPACE);
        GeneratedClass clazz = pkg._class(OAuthClientNamingConstants.AUTHORIZE_DEFINITION_PARSER_CLASS_NAME, ref(AbstractDevkitBasedDefinitionParser.class));
        ctx().registerProduct(Product.BEAN_DEFINITION_PARSER, module, "authorize", clazz);

        return clazz;
    }
}
