/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.oauth.generation.manager;

import org.apache.commons.pool.KeyedPoolableObjectFactory;
import org.mule.DefaultMuleMessage;
import org.mule.api.MuleMessage;
import org.mule.api.expression.ExpressionManager;
import org.mule.api.store.ObjectStore;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.utils.OAuth2StrategyUtilsResolver;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.oauth.OAuthCallbackParameterField;
import org.mule.devkit.model.module.oauth.OAuthCapability;
import org.mule.devkit.oauth.generation.AbstractOAuthAdapterGenerator;
import org.mule.devkit.oauth.generation.adapter.OAuth2ClientAdapterGenerator;
import org.mule.devkit.oauth.generation.manager.factory.OAuthClientFactoryGenerator;
import org.mule.devkit.utils.NameUtils;
import org.mule.security.oauth.*;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public abstract class AbstractOAuth2ManagerGenerator extends AbstractOAuthManagerGenerator{



    @Override
    protected TypeReference getExtendManagerClass() {
        return ref(BaseOAuth2Manager.class).narrow(OAuth2Adapter.class);
    }

    @Override
    protected AbstractOAuthAdapterGenerator getAdapterGenerator() {
        return new OAuth2ClientAdapterGenerator();
    }

    @Override
    public void doGenerate(Module module, GeneratedClass oAuthManagerClass) throws GenerationException {

        OAuthCapability oAuthCapability = getOAuthCapability(module);

        /**
         * Invoke the factory generator to fulfil the {@link BaseOAuthClientFactory} extends contract
         */
        callGenerator(module, getOAuthFactoryGenerator());

        GeneratedClass adapterClass = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        GeneratedClass oauthClientFactoryClass = ctx().getProduct(Product.OAUTH2_FACTORY, module);

        generateConfigurableFields(module, oAuthManagerClass, oAuthCapability, adapterClass);
        generateInstantiateMethod(module, oAuthManagerClass);
        generateCreatePoolFactoryMethod(oauthClientFactoryClass, oAuthManagerClass);
        generateSetCustomPropertiesMethod(module, adapterClass, oAuthManagerClass, oAuthCapability);
        generateFetchCallbackParameters(module, oAuthCapability, oAuthManagerClass, adapterClass);
        generateSetOnNoTokenPolicy(oAuthManagerClass);
        generateRefreshAccessTokenOn(module, oAuthManagerClass);
    }

    /**
     * Given a module, returns the concrete object that represents the {@link org.mule.api.annotations.oauth.OAuth2} annotation
     */
    protected abstract OAuthCapability getOAuthCapability(Module module);

    /**
     * Wraps the connector to the custom strategy if applies. If not, it will return the same object instance
     * @param module
     * @param field
     * @param connector
     * @return
     */
    protected abstract GeneratedExpression wrapAccessorConnector(Module module,Field field, GeneratedExpression connector);

    /**
     * Returns the list of configurable fields that the generated manager must wrap when being filled from the definition parser.
     * Some of this fields might be just for the connector, whereas others might be also used for the OAuth2 dance (consumer key, consumer secret or the scope).
     * If that's the case, then, those configurables are going to be treated differently
     * @param module
     * @return
     */
    protected abstract List<Field> getConfigurableFields(Module module);

    /**
     * Returns a factory generator to fulfil the {@link BaseOAuthClientFactory} extends contract
     */
    protected OAuthClientFactoryGenerator getOAuthFactoryGenerator(){
        return new OAuthClientFactoryGenerator();
    }

    /**
     * For a given {@code oAuthManagerClass}, it will generate the complete list of possible @Configurable(s) from the concrete @Connector,
     * as well from the given @OAuth2 component.
     *
     * @param module
     * @param oAuthManagerClass
     * @param oAuthCapability
     * @param adapterClass
     */
    private void generateConfigurableFields(Module module, GeneratedClass oAuthManagerClass, OAuthCapability oAuthCapability, GeneratedClass adapterClass) {

        for (Field field : getConfigurableFields(module)) {
            if (oAuthCapability.getConsumerKeyField().equals(field)) {
                GeneratedMethod setter = oAuthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, NameUtils.buildSetter(field.getName()));
                setter.javadoc().add("Sets " + field.getName());
                setter.javadoc().addParam("key to set");
                GeneratedVariable value = setter.param(ref(String.class), "value");
                setter.body().add(ExpressionFactory._super().invoke("setConsumerKey").arg(value));
                continue;
            } else if (oAuthCapability.getConsumerSecretField().equals(field)) {
                GeneratedMethod setter = oAuthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, NameUtils.buildSetter(field.getName()));
                setter.javadoc().add("Sets " + field.getName());
                setter.javadoc().addParam("secret to set");
                GeneratedVariable value = setter.param(ref(String.class), "value");
                setter.body().add(ExpressionFactory._super().invoke("setConsumerSecret").arg(value));
                continue;
            } else if (oAuthCapability.getScopeField() != null && oAuthCapability.getScopeField().equals(field)) {
                GeneratedMethod setter = oAuthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, NameUtils.buildSetter(field.getName()));
                setter.javadoc().add("Sets " + field.getName());
                setter.javadoc().addParam("scope to set");
                GeneratedVariable value = setter.param(ref(String.class), "value");
                setter.body().add(ExpressionFactory._super().invoke("setScope").arg(value));
                continue;
            }

            GeneratedMethod setter = oAuthManagerClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, NameUtils.buildSetter(field.getName()));
            setter.javadoc().add("Sets " + field.getName());
            setter.javadoc().addParam("scope to set");
            GeneratedVariable value = setter.param(ref(field.asTypeMirror()), "value");
            GeneratedExpression setterConnector = setter.body().decl(adapterClass, "connector", ExpressionFactory.cast(adapterClass, ExpressionFactory._this().invoke("getDefaultUnauthorizedConnector")));
            setterConnector = wrapAccessorConnector(module, field, setterConnector);
            setter.body().add(setterConnector.invoke(setter.name()).arg(value));

            GeneratedMethod getter = oAuthManagerClass.method(Modifier.PUBLIC, ref(field.asTypeMirror()), NameUtils.buildGetter(field.getName()));
            getter.javadoc().add("Retrieves " + field.getName());
            GeneratedExpression getterConnector = getter.body().decl(adapterClass,"connector",ExpressionFactory.cast(adapterClass,ExpressionFactory._this().invoke("getDefaultUnauthorizedConnector")));
            getterConnector = wrapAccessorConnector(module, field, getterConnector);
            getter.body()._return(getterConnector.invoke(getter.name()));
        }
    }

    private void generateSetOnNoTokenPolicy(GeneratedClass oauthManagerClass) {
        /**
         * this.getDefaultUnauthorizedConnector().setOnNoTokenPolicy(policy);
         */
        GeneratedMethod setOnNoTokenPolicyMethod = oauthManagerClass.method(Modifier.PUBLIC,ctx().getCodeModel().VOID,"setOnNoToken");
        GeneratedVariable policy = setOnNoTokenPolicyMethod.param(ref(OnNoTokenPolicy.class),"policy");
        setOnNoTokenPolicyMethod.body().add(ExpressionFactory._this().invoke("getDefaultUnauthorizedConnector").invoke("setOnNoTokenPolicy").arg(policy));
    }

    private void generateRefreshAccessTokenOn(Module module, GeneratedClass oauthManagerClass) {
        List<GeneratedExpression> exceptionClasses = new ArrayList<GeneratedExpression>();

        for (javax.lang.model.element.AnnotationValue reconnectionException : module.reconnectOn()) {
            exceptionClasses.add(ref(reconnectionException.getValue().toString()).boxify().dotclass());
        }
        for (ProcessorMethod processor : module.getProcessorMethods()) {
            for (javax.lang.model.element.AnnotationValue annotationValue : processor.reconnectOn()) {
                exceptionClasses.add(ref(annotationValue.getValue().toString()).boxify().dotclass());
            }
        }
        if (exceptionClasses.isEmpty()){ //just to support the old behaviour
            for (ProcessorMethod processor : module.getProcessorMethods()) {
                if (processor.invalidateAccessTokenOn() != null) {
                        exceptionClasses.add(ref(processor.invalidateAccessTokenOn()).boxify().dotclass());
                }
            }
        }

        if (!exceptionClasses.isEmpty()) {
            GeneratedMethod method = oauthManagerClass.method(Modifier.PROTECTED, ref(Set.class).narrow(ref(Class.class).narrow(ref(Exception.class).wildcard())), "refreshAccessTokenOn");
            method.annotate(Override.class);

            GeneratedExpression set = method.body().decl(ref(Set.class).narrow(ref(Class.class).narrow(ref(Exception.class).wildcard())), "types", ExpressionFactory._new(ref(HashSet.class).narrow(ref(Class.class).narrow(ref(Exception.class).wildcard()))));

            for (GeneratedExpression type : exceptionClasses) {
                method.body().invoke(set, "add").arg(type);
            }
            method.body()._return(set);
        }
    }

    private void generateFetchCallbackParameters(Module module, OAuthCapability oAuthCapability, GeneratedClass oauthManagerClass, GeneratedClass adapterClass) {
        GeneratedMethod fetchCallbackParameters = oauthManagerClass.method(Modifier.PROTECTED, context.getCodeModel().VOID, "fetchCallbackParameters");
        GeneratedVariable adapter = fetchCallbackParameters.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable response = fetchCallbackParameters.param(ref(String.class), "response");

        GeneratedVariable connector = fetchCallbackParameters.body().decl(adapterClass,"connector", ExpressionFactory.cast(adapterClass,adapter));

        GeneratedBlock body = fetchCallbackParameters.body();
        GeneratedVariable expressionManager = body.decl(ref(ExpressionManager.class), "expressionManager", ExpressionFactory.direct("muleContext.getExpressionManager()"));

        GeneratedVariable muleMessage = body.decl(ref(MuleMessage.class), "muleMessage",
                ExpressionFactory._new(ref(DefaultMuleMessage.class)).arg(response).arg(ExpressionFactory.direct("muleContext")));

        for (OAuthCallbackParameterField field : oAuthCapability.getCallbackParameters()) {
            GeneratedExpression castedConnector = OAuth2StrategyUtilsResolver.getOAuthConcreteComponent(module, connector, ctx());

            GeneratedInvocation invocation = castedConnector.invoke(NameUtils.buildSetter(field.getName())).arg(
                    ExpressionFactory.cast(ref(field.asTypeMirror()), expressionManager.invoke("evaluate").arg(field.getExpression()).arg(muleMessage)));
            body.add(invocation);
        }
    }

    private void generateSetCustomPropertiesMethod(Module module, GeneratedClass adapterClass, GeneratedClass oauthManagerClass, OAuthCapability oAuthCapability) {
        GeneratedMethod setCustomPropertiesMethod = oauthManagerClass.method(Modifier.PROTECTED,ctx().getCodeModel().VOID,"setCustomProperties");
        setCustomPropertiesMethod.annotate(Override.class);
        GeneratedVariable adapter = setCustomPropertiesMethod.param(ref(OAuth2Adapter.class), "adapter");
        GeneratedVariable connector = setCustomPropertiesMethod.body().decl(adapterClass, "connector", ExpressionFactory.cast(adapterClass, adapter));

        for (Field field :getConfigurableFields(module)) {
            String fieldGetterMethod = getSetterFieldMethod(oAuthCapability, field);
            GeneratedExpression targetObjectToSetField = wrapAccessorConnector(module, field, connector);
            GeneratedInvocation setterField = targetObjectToSetField.invoke(NameUtils.buildSetter(field.getName()))
                    .arg(ExpressionFactory.invoke(fieldGetterMethod));
            setCustomPropertiesMethod.body().add(setterField);
        }

    }

    /**
     * Returns the setter field of a given @Configurable field resolving them accordingly where:
     * 1) if they are within the scope of OAuth2 adapter (those methods are hardcoded in mule's code, in {@link org.mule.security.oauth.BaseOAuth2Manager} class)
     * 2) if the are just configurable of the connector/strategy
     *
     * @param oAuthCapability
     * @param field
     * @return
     */
    private String getSetterFieldMethod(OAuthCapability oAuthCapability, Field field) {
        //base case
        String fieldGetterMethod = NameUtils.buildGetter(field.getName());
        //if the configurable is actually some of this specific cases, we need to target the hardcoded methods of mule
        if (oAuthCapability.getConsumerKeyField().equals(field)) {
            fieldGetterMethod = "getConsumerKey";
        } else if (oAuthCapability.getConsumerSecretField().equals(field)) {
            fieldGetterMethod = "getConsumerSecret";
        } else if (oAuthCapability.getScopeField() != null && oAuthCapability.getScopeField().equals(field)) {
            fieldGetterMethod = "getScope";
        }
        return fieldGetterMethod;
    }


    private void generateCreatePoolFactoryMethod(GeneratedClass oauthClientFactoryClass, GeneratedClass oauthManagerClass) {
        GeneratedMethod createPoolFactoryMethod = oauthManagerClass.method(Modifier.PROTECTED,ref(KeyedPoolableObjectFactory.class),"createPoolFactory");
        createPoolFactoryMethod.annotate(Override.class);
        GeneratedVariable oauthManager = createPoolFactoryMethod.param(ref(OAuth2Manager.class).narrow(OAuth2Adapter.class),"oauthManager");
        GeneratedVariable objectStore = createPoolFactoryMethod.param(ref(ObjectStore.class).narrow(Serializable.class),"objectStore");
        createPoolFactoryMethod.body()._return(ExpressionFactory._new(oauthClientFactoryClass).arg(oauthManager).arg(objectStore));
    }

    private void generateInstantiateMethod(Module module,GeneratedClass oauthManagerClass) {
        GeneratedMethod instantiateMethod = oauthManagerClass.method(Modifier.PROTECTED,ref(OAuth2Adapter.class),"instantiateAdapter");
        instantiateMethod.annotate(Override.class);
        GeneratedClass oauthAdapterClass;
        if (module.hasRestCalls()) {
            instantiateMethod.body().directStatement("return new " +module.getPackage().getName() + ".adapters."+ module.getClassName() + "RestClientAdapter(this);");
        } else {
            oauthAdapterClass = ctx().getProduct(Product.OAUTH_ADAPTER,module);
            instantiateMethod.body()._return(ExpressionFactory._new(oauthAdapterClass.topLevelClass()).arg(ExpressionFactory._this()));
        }

    }
 }