/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.rest;

import org.apache.commons.httpclient.HttpClient;
import org.mule.api.annotations.rest.HttpMethod;
import org.mule.api.annotations.rest.RestHeaderParam;
import org.mule.api.annotations.rest.RestPostParam;
import org.mule.api.annotations.rest.RestQueryParam;
import org.mule.api.annotations.rest.RestUriParam;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.ModuleAnnotationVerifier;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.rest.RestCall;
import org.mule.devkit.model.module.rest.RestExceptionOn;
import org.mule.devkit.model.module.rest.RestModule;

public class RestClientAnnotationVerifier implements ModuleAnnotationVerifier {

    @Override
    public boolean shouldVerify(Module module) {
        return (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) && module.hasRestCalls();
    }

    @Override
    public void verify(Module module) throws AnnotationVerificationException {

        for (RestCall method : module.getRestCalls()) {

            if (!method.isAbstract()) {
                throw new AnnotationVerificationException(method, "@RestCall can only be applied to abstract methods");
            }

            if (method.getThrownTypes().size() != 1) {
                throw new AnnotationVerificationException(method, "@RestCall abstract method must throw IOException");
            }

            if (method.getExceptions() != null) {
                for (RestExceptionOn restExceptionOn : method.getExceptions()) {
                    if (restExceptionOn.getExpression() == null || "".equals(restExceptionOn.getExpression())) {
                        throw new AnnotationVerificationException(method, "@RestExceptionOn requires a expression to be defined");
                    }
                }
            }

            int nonAnnotatedParameterCount = 0;

            for (Parameter parameter : method.getParameters()) {
                if (parameter.getAnnotation(RestUriParam.class) == null &&
                        parameter.getAnnotation(RestHeaderParam.class) == null &&
                        parameter.getAnnotation(RestQueryParam.class) == null &&
                        parameter.getAnnotation(RestPostParam.class) == null) {
                    nonAnnotatedParameterCount++;
                }
            }

            int postParameter = method.getPostParameters().size();

            if (nonAnnotatedParameterCount > 1) {
                throw new AnnotationVerificationException(method, "Only one parameter can be used as payload, everything else must be annotated with @RestUriParam, @RestPostParam, @RestQueryParam or @RestHeaderParam.");
            }

            if (nonAnnotatedParameterCount == 1 && postParameter > 0) {
                throw new AnnotationVerificationException(method, "Post method supports a non annotated parameter as payload or @RestPostParam parameters, not both.");
            }

            if (method.getRestNoun() != HttpMethod.POST && postParameter > 0) {
                throw new AnnotationVerificationException(method, "@RestPostParam can only be used on POST methods.");
            }
        }

        for (Field field : ((RestModule) module).getUriFields()) {
            if (!field.hasGetter()) {
                throw new AnnotationVerificationException(field, "Cannot find a getter method for " + field.getName() + " but its being marked with @RestUriParam.");
            }
        }

        for (Field field : ((RestModule) module).getQueryFields()) {
            if (!field.hasGetter()) {
                throw new AnnotationVerificationException(field, "Cannot find a getter method for " + field.getName() + " but its being marked with @RestQueryParam.");
            }
        }

        for (Field field : ((RestModule) module).getHeaderFields()) {
            if (!field.hasGetter()) {
                throw new AnnotationVerificationException(field, "Cannot find a getter method for " + field.getName() + " but its being marked with @RestHeaderParam.");
            }
        }

        for (Field field : ((RestModule) module).getPostFields()) {
            if (!field.hasGetter()) {
                throw new AnnotationVerificationException(field, "Cannot find a getter method for " + field.getName() + " but its being marked with @RestPostParam.");
            }
        }

        Field httpClientField = module.getRestHttpClientField();

        if (httpClientField != null && !httpClientField.asTypeMirror().toString().equals(HttpClient.class.getName())) {
            throw new AnnotationVerificationException(httpClientField, "A field annotated with @RestHttpClient must be of type " + HttpClient.class.getName());
        }
        if (httpClientField != null && !httpClientField.hasGetter()) {
            throw new AnnotationVerificationException(httpClientField, "Cannot find a getter method for " + httpClientField.getName() + " but its being marked with @RestHttpClient.");
        }
    }
}