/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.rest;

import org.apache.commons.httpclient.HttpClient;
import org.mule.api.annotations.rest.*;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.ModuleAnnotationVerifier;
import org.mule.devkit.generation.api.gatherer.*;
import org.mule.devkit.generation.api.gatherer.Message;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.rest.RestCall;
import org.mule.devkit.model.module.rest.RestExceptionOn;
import org.mule.devkit.model.module.rest.RestModule;

import java.io.InputStream;

public class RestClientAnnotationVerifier implements ModuleAnnotationVerifier {

    @Override
    public boolean shouldVerify(Module module) {
        return (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) && module.hasRestCalls();
    }

    @Override
    public void verify(Module module,NotificationGatherer gatherer) throws AnnotationVerificationException {

        for (RestCall method : module.getRestCalls()) {

            if (!method.isAbstract()) {
                gatherer.error(method, Message.DEVKIT_186);
            }

            if (method.getThrownTypes().size() != 1) {
                gatherer.error(method, Message.DEVKIT_187);
            }

            if (method.getExceptions() != null) {
                for (RestExceptionOn restExceptionOn : method.getExceptions()) {
                    if (restExceptionOn.getExpression() == null || "".equals(restExceptionOn.getExpression())) {
                        gatherer.error(method, Message.DEVKIT_188);
                    }
                }
            }

            int nonAnnotatedParameterCount = 0;
            int binaryAnnotatedParameterCount = 0;

            for (Parameter parameter : method.getParameters()) {
                if (!parameter.hasAnnotation(RestUriParam.class) &&
                        !parameter.hasAnnotation(RestHeaderParam.class)&&
                        !parameter.hasAnnotation(RestQueryParam.class) &&
                        !parameter.hasAnnotation(RestPostParam.class)) {
                    nonAnnotatedParameterCount++;
                }
                if (parameter.hasAnnotation(BinaryParam.class)){
                    binaryAnnotatedParameterCount ++;
                }
            }

            int postParameter = method.getPostParameters().size();

            if (nonAnnotatedParameterCount > 1) {
                gatherer.error(method, Message.DEVKIT_189);
            }

            if (nonAnnotatedParameterCount == 1 && postParameter > 0) {
                gatherer.error(method, Message.DEVKIT_190);
            }

            if (method.getRestNoun() != HttpMethod.POST && postParameter > 0) {
                gatherer.error(method, Message.DEVKIT_191);
            }

            if (binaryAnnotatedParameterCount > 0) {
                if (binaryAnnotatedParameterCount > 1) {
                    gatherer.error(method, Message.DEVKIT_192);
                }
                if (method.payloadIsBinary()){
                    Parameter payloadParameter = method.getPayloadParameter();
                    if (! payloadParameter.asType().inheritsFrom(InputStream.class)){
                        gatherer.error(method, Message.DEVKIT_193);
                    }
                    if (method.getRestNoun() != HttpMethod.POST) {
                        gatherer.error(method, Message.DEVKIT_194);
                    }
                } else {
                    gatherer.error(method, Message.DEVKIT_195);
                }
            }
        }

        for (Field field : ((RestModule) module).getUriFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.DEVKIT_196, field.getName());
            }
        }

        for (Field field : ((RestModule) module).getQueryFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.DEVKIT_197,field.getName());
            }
        }

        for (Field field : ((RestModule) module).getHeaderFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.DEVKIT_198,field.getName());
            }
        }

        for (Field field : ((RestModule) module).getPostFields()) {
            if (!field.hasGetter()) {
                gatherer.error(field, Message.DEVKIT_199, field.getName());
            }
        }

        Field httpClientField = module.getRestHttpClientField();

        if (httpClientField != null && !httpClientField.asTypeMirror().toString().equals(HttpClient.class.getName())) {
            gatherer.error(httpClientField, Message.DEVKIT_200,HttpClient.class.getName());
        }
        if (httpClientField != null && !httpClientField.hasGetter()) {
            gatherer.error(httpClientField, Message.DEVKIT_201,httpClientField.getName());
        }
    }
}