/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.rest;

import oauth.signpost.http.HttpRequest;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.httpclient.*;
import org.apache.commons.httpclient.cookie.CookiePolicy;
import org.apache.commons.httpclient.methods.*;
import org.apache.commons.httpclient.util.URIUtil;
import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;
import org.mule.api.annotations.rest.HttpMethod;
import org.mule.api.context.MuleContextAware;
import org.mule.api.expression.ExpressionManager;
import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.registry.RegistrationException;
import org.mule.api.registry.ResolverException;
import org.mule.api.registry.TransformerResolver;
import org.mule.api.transformer.DataType;
import org.mule.api.transformer.Transformer;
import org.mule.api.transformer.TransformerException;
import org.mule.config.i18n.CoreMessages;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.utils.OAuth2StrategyUtilsResolver;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.components.connection.HttpBasicAuthComponent;
import org.mule.devkit.model.module.oauth.OAuthCapability;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.model.module.rest.*;
import org.mule.devkit.utils.NameUtils;
import org.mule.registry.TypeBasedTransformerResolver;
import org.mule.security.oauth.OAuth1Manager;
import org.mule.security.oauth.OAuth2Adapter;
import org.mule.security.oauth.OAuth2Manager;
import org.mule.transformer.simple.ObjectToString;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transformer.types.MimeTypes;
import org.mule.transport.http.HttpMuleMessageFactory;

import javax.inject.Inject;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;
import java.util.*;

public class RestClientAdapterGenerator extends AbstractRestClientGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(new Product[]{Product.OAUTH_ADAPTER, Product.PROCESS_ADAPTER});
    private final static List<Product> PRODUCES = Arrays.asList(new Product[]{Product.REST_CLIENT_ADAPTER});

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) && module.hasRestCalls();
    }

    @Override
    public void generate(Module module) {
        OAuthCapability oAuthCapability = null;
        if(module instanceof OAuthModule
                || OAuth2StrategyUtilsResolver.hasOAuth2Component(module)) {
            oAuthCapability = OAuth2StrategyUtilsResolver.getOAuthCapability(module);
        }

        TypeReference previous = getPrevious(module, oAuthCapability);
        GeneratedClass restClientAdapterClass = getRestClientAdapterClass(module, previous);
        GeneratedField responseTimeout = restClientAdapterClass.field(Modifier.PRIVATE, ctx().getCodeModel().INT, "responseTimeout");
        GeneratedField muleContext = restClientAdapterClass.field(Modifier.PRIVATE, ref(MuleContext.class), "muleContext");

        if(oAuthCapability != null) {
            GeneratedMethod constructor = restClientAdapterClass.constructor(Modifier.PUBLIC);
            if (oAuthCapability.getOAuthVersion().equals(OAuthVersion.V2)) {
                constructor.param(ref(OAuth2Manager.class).narrow(ref(OAuth2Adapter.class)), "oauthManager");
            } else if (oAuthCapability.getOAuthVersion().equals(OAuthVersion.V10A)) {
                constructor.param(ref(OAuth1Manager.class), "oauthManager");
            }

            constructor.body().directStatement("super(oauthManager);");
        }


        GeneratedExpression httpClient;
        if (module.getRestHttpClientField() == null) {
            httpClient = restClientAdapterClass.field(Modifier.PRIVATE | Modifier.VOLATILE, ref(HttpClient.class), "httpClient");
        } else {
            httpClient = ExpressionFactory.invoke("get" + StringUtils.capitalize(module.getRestHttpClientField().getName()));
        }

        GeneratedField httpMuleMessageFactory = restClientAdapterClass.field(Modifier.PRIVATE, ref(HttpMuleMessageFactory.class), "httpMuleMessageFactory");
        generateGetMuleMessage(restClientAdapterClass, httpMuleMessageFactory);
        generateSetMuleContext(restClientAdapterClass, muleContext, previous, httpMuleMessageFactory);
        generateGetPayloadTransformerMethod(restClientAdapterClass, muleContext);
        generateInitialiseMethod(restClientAdapterClass, module, httpClient, responseTimeout, previous);
        restClientAdapterClass.setter(responseTimeout);
        GeneratedClass http3Request = null;
        if(oAuthCapability != null && oAuthCapability.getOAuthVersion() == OAuthVersion.V10A) {
            http3Request = generateHttp3RequestInnerClass(restClientAdapterClass);
        }
        generateRestCallImplementations((RestModule) module, httpClient, muleContext, restClientAdapterClass, http3Request, oAuthCapability);
        generateSetHttpBasicAuthHeader(module, restClientAdapterClass);
    }

    private void generateSetHttpBasicAuthHeader(Module module, GeneratedClass restClientAdapterClass) {
        if (module.manager().httpBasicAuthComponent().isPresent()){
            HttpBasicAuthComponent httpBasicAuthComponent = module.manager().httpBasicAuthComponent().get();
            GeneratedMethod setBasicAuthHeaderMethod = restClientAdapterClass.method(Modifier.PRIVATE, ctx().getCodeModel().VOID, "setHttpBasicAuthHeader");
            GeneratedVariable methodParam = setBasicAuthHeaderMethod.param(ref(org.apache.commons.httpclient.HttpMethod.class), "method");

            GeneratedInvocation getStrategyMethod = ExpressionFactory.invoke(NameUtils.buildGetter(module.getConnectionStrategy().get().getName()));
            Type strategyClassType = ref(httpBasicAuthComponent.asTypeMirror());
            GeneratedBlock thenBlock = setBasicAuthHeaderMethod.body()._if(Op._instanceof(getStrategyMethod, strategyClassType))._then();
            GeneratedVariable stg = thenBlock.decl(strategyClassType, "stg",ExpressionFactory.cast(strategyClassType, getStrategyMethod));

            GeneratedInvocation stringConcatenation = stg.invoke(httpBasicAuthComponent.username().getGetter().getName()).invoke("concat").arg(":");
            if (httpBasicAuthComponent.password().isPresent()){
                stringConcatenation = stringConcatenation.invoke("concat").arg(stg.invoke(httpBasicAuthComponent.password().get().getGetter().getName()));
            }

            GeneratedVariable notEncodedHeader = thenBlock.decl(ref(String.class), "notEncodedHeader", stringConcatenation);
            GeneratedVariable encodedHeader = thenBlock.decl(ref(String.class), "encodedHeader",
                    ExpressionFactory.lit(httpBasicAuthComponent.prefix()).invoke("concat").arg(
                    ExpressionFactory._new(ref(String.class)).arg(ref(Base64.class).staticInvoke("encodeBase64").arg(notEncodedHeader.invoke("getBytes")))));
            thenBlock.add(methodParam.invoke("addRequestHeader").arg(httpBasicAuthComponent.headerName()).arg(encodedHeader));
        }
    }

    private GeneratedClass generateHttp3RequestInnerClass(GeneratedClass restClientAdapterClass) {
        GeneratedClass http3Request = restClientAdapterClass._class("Http3Request")._implements(ref(HttpRequest.class));
        GeneratedField httpMethod = http3Request.field(Modifier.PRIVATE, ref(org.apache.commons.httpclient.HttpMethod.class), "httpMethod");

        // Add constructor
        GeneratedMethod constructor = http3Request.constructor(Modifier.PUBLIC);
        GeneratedVariable method = constructor.param(ref(org.apache.commons.httpclient.HttpMethod.class), "httpMethod");
        constructor.body().assign(ExpressionFactory._this().ref(httpMethod), method);
        // Add getMethod method
        GeneratedMethod getMethod = http3Request.method(Modifier.PUBLIC, ref(String.class), "getMethod");
        getMethod.body()._return(httpMethod.invoke("getName"));
        // Add getRequestUrl method
        GeneratedMethod getRequestUrl = http3Request.method(Modifier.PUBLIC, ref(String.class), "getRequestUrl");
        GeneratedTry getRequestUrlTry = getRequestUrl.body()._try();
        getRequestUrlTry.body()._return(httpMethod.invoke("getURI").invoke("toString"));
        getRequestUrlTry._catch(ref(URIException.class)).param("e");
        getRequestUrl.body()._return(ExpressionFactory._null());
        // Add setRequestUrl method
        GeneratedMethod setRequestUrl = http3Request.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setRequestUrl");
        GeneratedVariable url = setRequestUrl.param(ref(String.class), "url");
        GeneratedTry setRequestUrlTry = setRequestUrl.body()._try();
        setRequestUrlTry.body().add(httpMethod.invoke("setURI").arg(ExpressionFactory._new(ref(URI.class)).arg(url)));
        setRequestUrlTry._catch(ref(URIException.class)).param("e");
        // Add getHeader method
        GeneratedMethod getHeader = http3Request.method(Modifier.PUBLIC, ref(String.class), "getHeader");
        GeneratedVariable headerName = getHeader.param(ref(String.class), "name");
        GeneratedVariable header = getHeader.body().decl(ref(Header.class), "header", httpMethod.invoke("getRequestHeader").arg(headerName));
        GeneratedConditional ifHeader = getHeader.body()._if(header.isNotNull());
        ifHeader._then()._return(header.invoke("getValue"));
        getHeader.body()._return(ExpressionFactory._null());
        // Add setHeader method
        GeneratedMethod setHeader = http3Request.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setHeader");
        headerName = setHeader.param(ref(String.class), "name");
        GeneratedVariable headerValue = setHeader.param(ref(String.class), "value");
        setHeader.body().add(httpMethod.invoke("setRequestHeader").arg(headerName).arg(headerValue));
        // Add getAllHeaders method
        GeneratedMethod getAllHeaders = http3Request.method(Modifier.PUBLIC, ref(Map.class).narrow(ref(String.class), ref(String.class)), "getAllHeaders");
        GeneratedVariable origHeaders = getAllHeaders.body().decl(ref(Header.class).array(), "origHeaders", httpMethod.invoke("getRequestHeaders"));
        GeneratedVariable headers = getAllHeaders.body().decl(ref(Map.class).narrow(ref(String.class), ref(String.class)), "headers", ExpressionFactory._new(ref(HashMap.class).narrow(ref(String.class), ref(String.class))));
        GeneratedForEach forEach = getAllHeaders.body().forEach(ref(Header.class), "h", origHeaders);
        forEach.body().add(headers.invoke("put").arg(forEach.var().invoke("getName")).arg(forEach.var().invoke("getValue")));
        getAllHeaders.body()._return(headers);
        // Add getContentType method
        GeneratedMethod getContentType = http3Request.method(Modifier.PUBLIC, ref(String.class), "getContentType");
        GeneratedVariable type = getContentType.body().decl(ref(String.class), "type", ExpressionFactory._null());
        GeneratedConditional ifPostMethod = getContentType.body()._if(Op._instanceof(httpMethod, ref(PostMethod.class)));
        ifPostMethod._then().assign(type, ExpressionFactory.cast(ref(PostMethod.class), httpMethod).invoke("getRequestEntity").invoke("getContentType"));
        ifPostMethod._else().assign(type, ExpressionFactory.invoke("getHeader").arg(ExpressionFactory.lit("Content-Type")));
        getContentType.body()._return(type);
        // Add getMessagePayload method
        GeneratedMethod getMessagePayload = http3Request.method(Modifier.PUBLIC, ref(InputStream.class), "getMessagePayload");
        getMessagePayload._throws(ref(IOException.class));
        ifPostMethod = getMessagePayload.body()._if(Op._instanceof(httpMethod, ref(PostMethod.class)));
        GeneratedVariable out = ifPostMethod._then().decl(ref(ByteArrayOutputStream.class), "out", ExpressionFactory._new(ref(ByteArrayOutputStream.class)));
        ifPostMethod._then().add(ExpressionFactory.cast(ref(PostMethod.class), httpMethod).invoke("getRequestEntity").invoke("writeRequest").arg(out));
        ifPostMethod._then()._return(ExpressionFactory._new(ref(ByteArrayInputStream.class)).arg(out.invoke("toByteArray")));
        getMessagePayload.body()._return(ExpressionFactory._null());
        // Add unwrap method
        GeneratedMethod unwrap = http3Request.method(Modifier.PUBLIC, ref(Object.class), "unwrap");
        unwrap.body()._return(httpMethod);
        return http3Request;
    }

    private void generateInitialiseMethod(GeneratedClass restClientAdapterClass, Module module, GeneratedExpression httpClient, GeneratedField responseTimeout, TypeReference previous) {
        GeneratedMethod initialise = restClientAdapterClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "initialise");
        initialise.annotate(ref(Override.class));
        if (previous instanceof GeneratedClass && ((GeneratedClass) previous).implementsClass(Initialisable.class)) {
            initialise.body().add(ExpressionFactory._super().invoke("initialise"));
        }
        if (module.getRestHttpClientField() == null) {
            initialise.body().assign((GeneratedField) httpClient, ExpressionFactory._new(ref(HttpClient.class)));
        } else {
            GeneratedConditional ifNullHttpClient = initialise.body()._if(httpClient.isNull());
            ifNullHttpClient._then()._throw(ExpressionFactory._new(ref(InitialisationException.class)).arg(ref(CoreMessages.class).staticInvoke("createStaticMessage").arg("Http client cannot be null.")
            ).arg(ExpressionFactory._this()));
        }

        initialise.body().add(httpClient.invoke("getParams").invoke("setParameter").arg("http.protocol.version").arg(ref(HttpVersion.class).staticRef("HTTP_1_1")));
        initialise.body().add(httpClient.invoke("getParams").invoke("setParameter").arg("http.socket.timeout").arg(responseTimeout));
        initialise.body().add(httpClient.invoke("getParams").invoke("setParameter").arg("http.protocol.content-charset").arg("UTF-8"));
        initialise.body().add(httpClient.invoke("getParams").invoke("setCookiePolicy").arg(ref(CookiePolicy.class).staticRef("BROWSER_COMPATIBILITY")));
        initialise._throws(ref(InitialisationException.class));
    }

    private void generateGetMuleMessage(GeneratedClass restClientAdapterClass, GeneratedVariable httpMuleMessageFactory) {
        GeneratedMethod getMuleMessage = restClientAdapterClass.method(Modifier.PRIVATE, ref(MuleMessage.class), "getMuleMessage");
        GeneratedVariable method = getMuleMessage.param(ref(org.apache.commons.httpclient.HttpMethod.class), "method");
        GeneratedVariable encoding = getMuleMessage.param(ref(String.class), "encoding");
        GeneratedTry tryTransform = getMuleMessage.body()._try();

        GeneratedVariable muleMessage = tryTransform.body().decl(ref(MuleMessage.class), "muleMessage", httpMuleMessageFactory.invoke("create").arg(method).arg(encoding));
        tryTransform.body().invoke(muleMessage, "getPayloadAsString");
        tryTransform.body()._return(muleMessage);

        GeneratedCatchBlock catchCreateMuleMessage = tryTransform._catch(ref(Exception.class));
        GeneratedVariable payloadTransformerException = catchCreateMuleMessage.param("e");
        catchCreateMuleMessage.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Couldn't transform http response to MuleMessage").arg(payloadTransformerException));
    }

    private void generateGetPayloadTransformerMethod(GeneratedClass restClientAdapterClass, GeneratedField muleContext) {
        GeneratedMethod getPayloadTransformer = restClientAdapterClass.method(Modifier.PRIVATE, ref(Transformer.class), "getPayloadTransformer");
        GeneratedVariable inputDataType = getPayloadTransformer.param(ref(DataType.class), "inputDataType");
        GeneratedVariable outputDataType = getPayloadTransformer.param(ref(DataType.class), "outputDataType");

        GeneratedTry tryGetTransformer = getPayloadTransformer.body()._try();
        GeneratedVariable typeResolver = tryGetTransformer.body().decl(ref(TransformerResolver.class), "typeBasedResolver", muleContext.invoke("getRegistry").invoke("lookupObject").arg(ref(TypeBasedTransformerResolver.class).dotclass()));
        GeneratedVariable typeTransformer = tryGetTransformer.body().decl(ref(Transformer.class), "typeResolverTransformer", typeResolver.invoke("resolve").arg(inputDataType).arg(outputDataType));
        GeneratedConditional ifNoTypeTransformer = tryGetTransformer.body()._if(Op.cor(typeTransformer.isNull(), Op._instanceof(typeTransformer, ref(ObjectToString.class))));
        GeneratedVariable payloadTransformer = ifNoTypeTransformer._then().decl(ref(Transformer.class), "transformer", muleContext.invoke("getRegistry").invoke("lookupTransformer").arg(inputDataType).arg(outputDataType));
        GeneratedConditional ifPayloadTransformer = ifNoTypeTransformer._then()._if(payloadTransformer.isNotNull());
        ifPayloadTransformer._then()._return(payloadTransformer);
        tryGetTransformer.body()._return(typeTransformer);
        GeneratedCatchBlock catchResolverException = tryGetTransformer._catch(ref(ResolverException.class));
        GeneratedVariable resolverException = catchResolverException.param("rese");
        catchResolverException.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(resolverException.invoke("getMessage")).arg(resolverException));

        GeneratedCatchBlock catchRegistrationException = tryGetTransformer._catch(ref(RegistrationException.class));
        GeneratedVariable registrationException = catchRegistrationException.param("re");
        catchRegistrationException.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(registrationException.invoke("getMessage")).arg(registrationException));

        GeneratedCatchBlock catchTransformerException = tryGetTransformer._catch(ref(TransformerException.class));
        GeneratedVariable transformerException = catchTransformerException.param("te");
        catchTransformerException.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(transformerException.invoke("getMessage")).arg(transformerException));
    }

    private void generateSetMuleContext(GeneratedClass restClientAdapterClass, GeneratedField muleContext, TypeReference previous, GeneratedVariable httpMuleMessageFactory) {
        GeneratedMethod setMuleContext = restClientAdapterClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setMuleContext");
        GeneratedVariable context = setMuleContext.param(ref(MuleContext.class), "context");
        if (previous instanceof GeneratedClass && ((GeneratedClass) previous).implementsClass(MuleContextAware.class)) {
            setMuleContext.body().add(ExpressionFactory._super().invoke("setMuleContext").arg(context));
        }
        setMuleContext.body().assign(muleContext, context);
        setMuleContext.body().assign(httpMuleMessageFactory, ExpressionFactory._new(ref(HttpMuleMessageFactory.class)).arg(muleContext));
    }

    private void generateRestCallImplementations(RestModule module, GeneratedExpression httpClient, GeneratedVariable muleContext, GeneratedClass capabilitiesAdapter, GeneratedClass http3Request, OAuthCapability oAuthCapability) {
        Map<String, GeneratedVariable> variables = new HashMap<String, GeneratedVariable>();
        for (RestCall executableElement : module.getRestCalls()) {
            GeneratedMethod override = capabilitiesAdapter.method(Modifier.PUBLIC, ref(executableElement.getReturnType()), executableElement.getName());
            override._throws(ref(IOException.class));

            for (Parameter parameter : executableElement.getParameters()) {
                variables.put(
                        parameter.getName(),
                        override.param(ref(parameter.asTypeMirror()), parameter.getName())
                );
            }

            GeneratedVariable method = override.body().decl(ref(org.apache.commons.httpclient.HttpMethod.class), "method", ExpressionFactory._null());
            generateMethodAssignment(override, method, executableElement);
            generateParametersCode(module, variables, executableElement, override, method, oAuthCapability);

            generateHttpClientRequestEntity(variables, executableElement, override, method);
            signOauth1(module, executableElement, override, http3Request, method, oAuthCapability);

            generateHttpClientExecuteMethod(override, httpClient, method, executableElement);
            generateParseResponseCode(module, variables, executableElement, override, method, muleContext);
        }
    }

    private void generateHttpClientRequestEntity(Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method) {
        if (executableElement.getRestNoun() == HttpMethod.POST || executableElement.getRestNoun() == HttpMethod.PUT) {
            RestParameter payloadParameter = executableElement.getPayloadParameter();
            if (payloadParameter != null) {
                if (executableElement.payloadIsBinary()) {
                    if (payloadParameter.isContentChunked()){
                        override.body().add(ExpressionFactory.cast(ref(EntityEnclosingMethod.class), method).invoke("setContentChunked").arg(ExpressionFactory.TRUE));
                    }
                    GeneratedVariable payloadRequestEntity = override.body().decl(ref(RequestEntity.class), "payloadRequestEntity",
                            ExpressionFactory._new(ref(InputStreamRequestEntity.class)).arg(variables.get(payloadParameter.getName())). arg(executableElement.getContentType()));
                    override.body().add(ExpressionFactory.cast(ref(EntityEnclosingMethod.class), method).invoke("setRequestEntity").arg(payloadRequestEntity));
                }
                else {

                    if (!executableElement.getContentType().equals(MimeTypes.ANY)) {
                        GeneratedVariable inputDataType = override.body().decl(ref(DataType.class), "inputDataType", ref(DataTypeFactory.class).staticInvoke("createFromObject").arg(variables.get(payloadParameter.getName())));
                        GeneratedVariable outputDataType = override.body().decl(ref(DataType.class), "outputDataType", ref(DataTypeFactory.class).staticInvoke("create").arg(ref(String.class).dotclass()).arg(executableElement.getContentType()));

                        GeneratedVariable payloadTransformer = override.body().decl(ref(Transformer.class), "payloadTransformer", ExpressionFactory.invoke("getPayloadTransformer").arg(inputDataType).arg(outputDataType));
                        GeneratedTry tryToTransform = override.body()._try();
                        GeneratedVariable payloadRequestEntity = tryToTransform.body().decl(ref(RequestEntity.class), "payloadRequestEntity", ExpressionFactory._new(ref(StringRequestEntity.class)).arg(ExpressionFactory.cast(ref(String.class), payloadTransformer.invoke("transform").arg(
                                variables.get(payloadParameter.getName())
                        ))).arg(executableElement.getContentType()).arg(ExpressionFactory.lit("UTF-8")));
                        tryToTransform.body().add(ExpressionFactory.cast(ref(EntityEnclosingMethod.class), method).invoke("setRequestEntity").arg(payloadRequestEntity));

                        GeneratedCatchBlock catchTransformerException = tryToTransform._catch(ref(TransformerException.class));
                        GeneratedVariable transformerException = catchTransformerException.param("te");
                        catchTransformerException.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(transformerException.invoke("getMessage")).arg(transformerException));
                    } else {
                        GeneratedVariable payloadRequestEntity = override.body().decl(ref(RequestEntity.class), "payloadRequestEntity", ExpressionFactory._new(ref(StringRequestEntity.class)).arg(ref(String.class).staticInvoke("valueOf").arg(variables.get(payloadParameter.getName()))));
                        override.body().add(ExpressionFactory.cast(ref(EntityEnclosingMethod.class), method).invoke("setRequestEntity").arg(payloadRequestEntity));
                    }
                }
            }
            
        }
    }

    private void generateHttpClientExecuteMethod(GeneratedMethod override, GeneratedExpression httpClient, GeneratedVariable method, RestCall executableElement) {
        if(executableElement.getTimeout() == null) {
            override.body().add(httpClient.invoke("executeMethod").arg(method));
        } else {
            GeneratedTry tryExecuteMethod = override.body()._try();
            tryExecuteMethod.body().add(httpClient.invoke("getParams").invoke("setParameter").arg(ExpressionFactory.lit("http.socket.timeout")).arg(ExpressionFactory.lit(executableElement.getTimeout().getTimeout())));
            tryExecuteMethod.body().add(httpClient.invoke("executeMethod").arg(method));
            GeneratedCatchBlock catchExecuteMethod = tryExecuteMethod._catch(ref(SocketTimeoutException.class));
            GeneratedVariable socketTimeoutException = catchExecuteMethod.param("stoe");
            GeneratedInvocation exception;
            if(executableElement.getTimeout().getException() == null) {
                exception = ExpressionFactory._new(ref(RuntimeException.class));
            } else {
                exception = ExpressionFactory._new(ref(executableElement.getTimeout().getException().asTypeMirror()));
            }
            catchExecuteMethod.body()._throw(exception.arg(ExpressionFactory.lit("Read timed out")).arg(socketTimeoutException));

        }
    }

    private void generateParametersCode(RestModule module, Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method, OAuthCapability oAuthCapability) {
        GeneratedVariable uri = generateUriParametersCode(module, variables, executableElement, override, method);
        generateQueryParametersCode(module, variables, executableElement, override, method, uri, oAuthCapability);
        generateHeaderParametersCode(module, variables, executableElement, override, method);
        if (executableElement.getRestNoun() == HttpMethod.POST) {
            generatePostParametersCode(module, variables, executableElement, override, method);
        }
    }

    private GeneratedVariable generateUriParametersCode(RestModule module, Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method) {
        GeneratedVariable uri = override.body().decl(ref(String.class), "uri", ExpressionFactory.lit(executableElement.getUri()));
        for (RestParameter parameter : executableElement.getUriParameters()) {
            addUriParameter(executableElement,override.body(), uri, variables.get(parameter.getName()), parameter.ignoreIfEmpty(), parameter.getUriName());
        }
        for (RestField field : module.getUriFields()) {
            addUriParameter(executableElement,override.body(), uri, ExpressionFactory.invoke("get" + StringUtils.capitalize(field.getName())), field.ignoreIfEmpty(), field.getUriName());
        }
        override.body().add(method.invoke("setURI").arg(ExpressionFactory._new(ref(URI.class)).arg(uri).arg(ExpressionFactory.FALSE)));
        return uri;
    }

    private void generateQueryParametersCode(RestModule module, Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method, GeneratedVariable uri, OAuthCapability oAuthCapability) {
        GeneratedVariable queryString = override.body().decl(ref(StringBuilder.class), "queryString", ExpressionFactory._new(ref(StringBuilder.class)).arg(Op.cond(method.invoke("getQueryString").isNotNull(), method.invoke("getQueryString"), ExpressionFactory.lit(""))));
        for (RestParameter parameter : executableElement.getQueryParameters()) {
            addQueryParameter(override.body(), queryString, variables.get(parameter.getName()), parameter.ignoreIfEmpty(), parameter.getUriName(), parameter.getSeparatedBy());
        }
        for (RestField field : module.getQueryFields()) {
            addQueryParameter(override.body(), queryString, ExpressionFactory.invoke("get" + StringUtils.capitalize(field.getName())), field.ignoreIfEmpty(), field.getUriName(), field.getSeparatedBy());
        }
        addOauth2AccessToken(module, executableElement, queryString, override, oAuthCapability);
        GeneratedConditional ifQueryString = override.body()._if(Op.cand(queryString.invoke("length").gt(ExpressionFactory.lit(0)), queryString.invoke("charAt").arg(ExpressionFactory.lit(0)).eq(ExpressionFactory.lit('&'))));
        ifQueryString._then().add(queryString.invoke("deleteCharAt").arg(ExpressionFactory.lit(0)));
        override.body().add(method.invoke("setQueryString").arg(ref(URIUtil.class).staticInvoke("encodeQuery").arg(queryString.invoke("toString"))));
    }

    private void generateHeaderParametersCode(RestModule module, Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method) {
        for (RestParameter parameter : executableElement.getHeaderParameters()) {
            addHeaderParameter(override.body(), method, variables.get(parameter.getName()), parameter.ignoreIfEmpty(), parameter.getUriName(), parameter.getSeparatedBy());
        }
        for (RestField field : module.getHeaderFields()) {
            addHeaderParameter(override.body(), method, ExpressionFactory.invoke("get" + StringUtils.capitalize(field.getName())), field.ignoreIfEmpty(), field.getUriName(), field.getSeparatedBy());
        }
        //if using @HttpBasicAuth then we add the call to the custom header implementation
        if(module.manager().httpBasicAuthComponent().isPresent()){
            override.body().invoke("setHttpBasicAuthHeader").arg(method);
        }
    }

    private void addOauth2AccessToken(Module module, RestCall executableElement, GeneratedVariable queryString, GeneratedMethod override, OAuthCapability oAuthCapability) {
        if(oAuthCapability != null
                && executableElement.isOAuthProtected()) {
            if(oAuthCapability.getOAuthVersion() == OAuthVersion.V2) {
                Field accessToken = oAuthCapability.getAccessTokenField();
                override.body().add(queryString.invoke("append").arg("&").invoke("append").arg(ExpressionFactory.lit("access_token")).invoke("append").arg("=").invoke("append").arg(ExpressionFactory.invoke("get" + StringUtils.capitalize(accessToken.getName()))));
            }
        }
    }

    private void signOauth1(Module module, RestCall executableElement, GeneratedMethod override, GeneratedClass http3Request, GeneratedVariable method, OAuthCapability oAuthCapability) {
        if(oAuthCapability != null && executableElement.isOAuthProtected()) {
            if(oAuthCapability.getOAuthVersion() == OAuthVersion.V10A) {
                GeneratedTry oauthTry = override.body()._try();
                oauthTry.body().add(ExpressionFactory._super().invoke("getConsumer").invoke("sign").arg(ExpressionFactory._new(http3Request).arg(method)));
                GeneratedCatchBlock oauthCatch = oauthTry._catch(ref(Exception.class));
                GeneratedVariable oauthException = oauthCatch.param("oe");
                oauthCatch.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(ExpressionFactory.lit("Unable to add Oauth 1.0 information to request")).arg(oauthException));
            }
        }
    }

    private void generatePostParametersCode(RestModule module, Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method) {
        GeneratedVariable postParameters = override.body().decl(ref(List.class).narrow(ref(NameValuePair.class)), "postParameters", ExpressionFactory._new(ref(ArrayList.class).narrow(ref(NameValuePair.class))));
        for (RestParameter parameter : executableElement.getPostParameters()) {
            addPostParameter(override.body(), postParameters, variables.get(parameter.getName()), parameter.ignoreIfEmpty(), parameter.getUriName());
        }
        for (RestField field : module.getPostFields()) {
            addPostParameter(override.body(), postParameters, ExpressionFactory.invoke("get" + StringUtils.capitalize(field.getName())), field.ignoreIfEmpty(), field.getUriName());
        }
        override.body().add(ExpressionFactory.cast(ref(PostMethod.class), method).invoke("addParameters").arg(postParameters.invoke("toArray").arg(ExpressionFactory._new(ref(NameValuePair.class).array()))));
    }

    private void generateMethodAssignment(GeneratedMethod override, GeneratedVariable method, RestCall restCall) {
        if (restCall.getRestNoun() == HttpMethod.GET) {
            override.body().assign(method, ExpressionFactory._new(ref(GetMethod.class)));
        } else if (restCall.getRestNoun() == HttpMethod.PUT) {
            override.body().assign(method, ExpressionFactory._new(ref(PutMethod.class)));
        } else if (restCall.getRestNoun() == HttpMethod.DELETE) {
            override.body().assign(method, ExpressionFactory._new(ref(DeleteMethod.class)));
        } else if (restCall.getRestNoun() == HttpMethod.POST) {
            override.body().assign(method, ExpressionFactory._new(ref(PostMethod.class)));
        } else if (restCall.getRestNoun() == HttpMethod.TRACE) {
            override.body().assign(method, ExpressionFactory._new(ref(TraceMethod.class)));
        }
    }

    private void generateParseResponseCode(Module module,  Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method, GeneratedVariable muleContext) {
        GeneratedVariable muleMessage = override.body().decl(ref(MuleMessage.class), "muleMessage", ExpressionFactory.invoke("getMuleMessage").arg(method).arg(ExpressionFactory.lit("UTF-8")));
        generateCopyResponseHeadersToMuleEvent(variables, executableElement, override, method);
        generateExceptionOnBlock(executableElement, override, muleContext, muleMessage);
        if(ref(executableElement.getReturnType()) != ctx().getCodeModel().VOID) {
            generateTransformAndReturn(module, executableElement, override, muleMessage);
        }
    }

    /**
     * Adds the response HTTP headers to the MuleEvent only if the processor was annotated with {@link Inject}
     * and it receives a {@link MuleEvent} as parameter
     * */
    private void generateCopyResponseHeadersToMuleEvent(Map<String, GeneratedVariable> variables, RestCall executableElement, GeneratedMethod override, GeneratedVariable method) {
        if (executableElement.getAnnotation(Inject.class) != null){
            boolean hasInjectedMuleEvent = false;
            GeneratedVariable muleEventParameter = null;
            for (Parameter parameter : executableElement.getParameters()) {
                if (parameter.asTypeMirror().toString().startsWith(MuleEvent.class.getName())){
                    hasInjectedMuleEvent = true;
                    muleEventParameter = variables.get(parameter.getName());
                }
            }
            if (hasInjectedMuleEvent){
                GeneratedForEach generatedForEach = override.body().forEach(ref(Header.class), "header", ref(Arrays.class).staticInvoke("asList").arg(method.invoke("getResponseHeaders")));
                generatedForEach.body().invoke(muleEventParameter.invoke("getMessage"), "setOutboundProperty").arg(generatedForEach.var().invoke("getName")).arg(generatedForEach.var().invoke("getValue"));
            }
        }
    }

    private void generateTransformAndReturn(Module moduleClass, RestCall executableElement, GeneratedMethod methodBody, GeneratedVariable muleMessage) {
        GeneratedVariable output = methodBody.body().decl(ref(String.class), "output", ExpressionFactory.cast(ref(String.class), muleMessage.invoke("getPayload")));
        GeneratedConditional shouldTransform = methodBody.body()._if(Op.cand(
                output.isNotNull(), Op.not(ref(executableElement.getReturnType()).dotclass().invoke("isAssignableFrom").arg(ref(String.class).dotclass()))));

        GeneratedVariable outputDataType = shouldTransform._then().decl(ref(DataType.class), "payloadOutputDataType", ExpressionFactory._null());
        GeneratedTry tryToTransform = shouldTransform._then()._try();
        GeneratedInvocation getMethod = ref(moduleClass.asTypeMirror()).dotclass().invoke("getMethod").arg(executableElement.getName());
        for (Parameter parameter : executableElement.getParameters()) {
            getMethod.arg(ref(parameter.asTypeMirror()).dotclass());
        }
        GeneratedVariable method = tryToTransform.body().decl(ref(java.lang.reflect.Method.class), "reflectedMethod", getMethod);
        tryToTransform.body().assign(outputDataType,ref(DataTypeFactory.class).staticInvoke("createFromReturnType").arg(method));
        GeneratedVariable inputDataType = tryToTransform.body().decl(ref(DataType.class), "payloadInputDataType", ref(DataTypeFactory.class).staticInvoke("create").arg(ref(String.class).dotclass()).arg(ExpressionFactory.cast(ref(String.class), muleMessage.invoke("getOutboundProperty").arg(ExpressionFactory.lit("Content-Type")))));
        GeneratedVariable payloadTransformer = tryToTransform.body().decl(ref(Transformer.class), "transformer", ExpressionFactory.invoke("getPayloadTransformer").arg(inputDataType).arg(outputDataType));

        tryToTransform.body()._return(ExpressionFactory.cast(ref(executableElement.getReturnType()).boxify(), payloadTransformer.invoke("transform").arg(output)));

        GeneratedCatchBlock catchTransformerException = tryToTransform._catch(ref(TransformerException.class));
        GeneratedVariable transformerException = catchTransformerException.param("te");
        catchTransformerException.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(Op.plus(ExpressionFactory.lit("Unable to transform output from String to "), outputDataType.invoke("toString"))).arg(transformerException));
        GeneratedCatchBlock catchNoSuchMethodException = tryToTransform._catch(ref(NoSuchMethodException.class));
        GeneratedVariable noSuchMethodException = catchNoSuchMethodException.param("nsme");
        catchNoSuchMethodException.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(ExpressionFactory.lit("Unable to find method named " + executableElement.getName())).arg(noSuchMethodException));

        shouldTransform._else()._return(ExpressionFactory.cast(ref(executableElement.getReturnType()).boxify(), ExpressionFactory.cast(ref(Object.class), output)));
    }


    private void generateExceptionOnBlock(RestCall executableElement, GeneratedMethod methodBody, GeneratedVariable muleContext, GeneratedVariable muleMessage) {
        if(executableElement.getExceptions() != null) {
            List<RestExceptionOn> exceptions = executableElement.getExceptions();
            GeneratedVariable expressionManager = methodBody.body().decl(ref(ExpressionManager.class), "expressionManager", muleContext.invoke("getExpressionManager"));
            for(RestExceptionOn restExceptionOn : exceptions) {
                if(restExceptionOn.getExpression() != null && !"".equals(restExceptionOn.getExpression())) {
                    GeneratedConditional ifError = methodBody.body()._if(expressionManager.invoke("evaluateBoolean").arg(restExceptionOn.getExpression()).arg(muleMessage));
                    GeneratedInvocation exception;
                    if (restExceptionOn.getException() == null) {
                        exception = ExpressionFactory._new(ref(RuntimeException.class));
                    } else {
                        exception = ExpressionFactory._new(ref(restExceptionOn.getException().asTypeMirror()));
                    }
                    ifError._then()._throw(exception.arg(ExpressionFactory.cast(ref(String.class), muleMessage.invoke("getPayload"))));
                }
            }
        }
    }

    private void addQueryParameter(GeneratedBlock body, GeneratedVariable queryString, GeneratedExpression variable, boolean ignoreIfEmpty, String uriName, String token) {
        GeneratedBlock bodyHolder= retrieveBlockWithNullValidations(body, variable, ignoreIfEmpty);
        GeneratedExpression rvalue = ref(String.class).staticInvoke("valueOf").arg(variable);
        if (!StringUtils.isEmpty(token)) {
            rvalue = ref(StringUtils.class).staticInvoke("join").arg(variable.invoke("toArray")).arg(token);
        }
        bodyHolder.add(queryString.invoke("append").arg("&").invoke("append").arg(uriName).invoke("append").arg("=").invoke("append").arg(rvalue));
    }

    private void addHeaderParameter (GeneratedBlock body,  GeneratedVariable method, GeneratedExpression variable, boolean ignoreIfEmpty, String uriName, String token){
        GeneratedBlock bodyHolder= retrieveBlockWithNullValidations(body, variable, ignoreIfEmpty);
        GeneratedExpression rvalue = ref(String.class).staticInvoke("valueOf").arg(variable);
        if (!StringUtils.isEmpty(token)) {
            rvalue = ref(StringUtils.class).staticInvoke("join").arg(variable.invoke("toArray")).arg(token);
        }
        bodyHolder.add(method.invoke("addRequestHeader").arg(uriName).arg(rvalue));
    }

    private void addPostParameter(GeneratedBlock body, GeneratedVariable queryString, GeneratedExpression variable, boolean ignoreIfEmpty, String uriName) {
        GeneratedBlock bodyHolder= retrieveBlockWithNullValidations(body, variable, ignoreIfEmpty);
        GeneratedExpression rvalue = ref(String.class).staticInvoke("valueOf").arg(variable);
        bodyHolder.add(queryString.invoke("add").arg(ExpressionFactory._new(ref(NameValuePair.class)).arg(uriName).arg(rvalue)));
    }

    private void addUriParameter(RestCall executableElement, GeneratedBlock body, GeneratedVariable uri, GeneratedExpression variable, boolean ignoreIfEmpty, String uriName) {
        if (executableElement.getUri().contains("{" + uriName + "}")) {
            if (ignoreIfEmpty){
                GeneratedConditional ifTestingAgainstNull = retrieveIfWithNullValidations(body, variable);
                ifTestingAgainstNull._then().assign(uri, uri.invoke("replace").arg("{" + uriName + "}").arg(ref(String.class).staticInvoke("valueOf").arg(variable)));
                ifTestingAgainstNull._else().assign(uri, uri.invoke("replace").arg("{" + uriName + "}").arg(ExpressionFactory.lit("")));
            }else{
                body.assign(uri, uri.invoke("replace").arg("{" + uriName + "}").arg(ref(String.class).staticInvoke("valueOf").arg(variable)));
            }
        }
    }

    private GeneratedConditional retrieveIfWithNullValidations(GeneratedBlock body, GeneratedExpression variable){
        return body._if(Op.cand(variable.isNotNull(),Op.not(ref(StringUtils.class).staticInvoke("isEmpty").arg(ref(String.class).staticInvoke("valueOf").arg(variable)))));
    }

    private GeneratedBlock retrieveBlockWithNullValidations(GeneratedBlock body, GeneratedExpression variable, boolean ignoreIfEmpty) {
        GeneratedBlock bodyHolder = body;
        if (ignoreIfEmpty){
            bodyHolder = retrieveIfWithNullValidations(body, variable)._then();
        }
        return bodyHolder;
    }

    private GeneratedClass getRestClientAdapterClass(Module module, TypeReference previous) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + ".adapters");
        GeneratedClass clazz = pkg._class(module.getClassName() + "RestClientAdapter", previous);
        clazz._implements(ref(Initialisable.class));
        clazz._implements(ref(Disposable.class));
        clazz._implements(ref(MuleContextAware.class));
        ctx().registerProduct(Product.REST_CLIENT_ADAPTER, module, clazz);
        return clazz;
    }

    private TypeReference getPrevious(Module module, OAuthCapability oAuthCapability) {
        TypeReference previous;
        if(oAuthCapability != null) {
            previous = ctx().<GeneratedClass>getProduct(Product.OAUTH_ADAPTER, module).topLevelClass();
        } else {
            previous = ctx().<GeneratedClass>getProduct(Product.PROCESS_ADAPTER, module).topLevelClass();
        }

        if(previous == null) {
            previous = (TypeReference) ref(module.asTypeMirror());
        }
        return previous;
    }

}