package org.mule.devkit.generation.rest;

import org.mule.api.annotations.rest.BasicAuthPassword;
import org.mule.api.annotations.rest.BasicAuthUsername;
import org.mule.devkit.generation.api.AbstractBaseAnnotationVerifier;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.gatherer.Message;
import org.mule.devkit.generation.api.gatherer.NotificationGatherer;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.components.connection.ConnectionComponent;
import org.mule.devkit.model.module.components.connection.HttpBasicAuthComponent;
import org.mule.devkit.verifiers.strategies.ConnectionStrategiesVerifierUtils;

import java.lang.annotation.Annotation;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class HttpBasicAuthComponentVerifier extends AbstractBaseAnnotationVerifier {

    @Override
    public boolean shouldVerify(Module module) {
        return module.manager().httpBasicAuthComponent().isPresent();
    }

    @Override
    public void verify(Module module, NotificationGatherer gatherer) throws AnnotationVerificationException {
        HttpBasicAuthComponent httpBasicAuthComponent = module.manager().httpBasicAuthComponent().get();
        if (!module.hasRestCalls()) {
            gatherer.error(httpBasicAuthComponent, Message.HTTP_BASIC_AUTH_WITHOUT_RESTCALL, module.getName());
        }
        verifyProperUserOfAnnotation(module, gatherer, httpBasicAuthComponent, BasicAuthUsername.class, 1, 1);
        verifyProperUserOfAnnotation(module, gatherer, httpBasicAuthComponent, BasicAuthPassword.class, 1, 0);
        ConnectionStrategiesVerifierUtils.verifyNoConnectMethods(module, gatherer, httpBasicAuthComponent);

    }

    private void verifyProperUserOfAnnotation(Module module, NotificationGatherer gatherer, HttpBasicAuthComponent httpBasicAuthComponent,
                                              Class<? extends Annotation> annotationClass,
                                              int maxOccurs, int minOccurs) {
        List<Field> fieldsAnnotatedWith = httpBasicAuthComponent.getFieldsAnnotatedWith(annotationClass);
        if (fieldsAnnotatedWith.size() > maxOccurs ){
            gatherer.errorAll(fieldsAnnotatedWith, Message.HTTP_BASIC_AUTH_REPEATED_ANNOTATION, "@".concat(annotationClass.getSimpleName()));
        }else if(fieldsAnnotatedWith.size() < minOccurs){
            gatherer.error(httpBasicAuthComponent, Message.HTTP_BASIC_AUTH_REPEATED_ANNOTATION, "@".concat(annotationClass.getSimpleName()));
        }else{
            verifyHttpAnnotatedField(module, httpBasicAuthComponent, gatherer, fieldsAnnotatedWith.get(0), annotationClass);
        }
    }

    private void verifyHttpAnnotatedField(Module module, HttpBasicAuthComponent httpBasicAuthComponent, NotificationGatherer gatherer, Field field, Class<? extends Annotation> annotationClass) {
        if (!field.asTypeMirror().toString().equals("java.lang.String")){
            gatherer.error(field, Message.HTTP_BASIC_AUTH_REQUIRES_STRING, "@".concat(annotationClass.getSimpleName()));
        }

        verifyConfigurablesGettersAndSetters(httpBasicAuthComponent, gatherer);

        Set<String> configurableNames = new HashSet<String>();
        for (Field moduleField : module.getConfigurableFields()) {
            configurableNames.add(moduleField.getName());
        }
        if (configurableNames.contains(field.getName())) {
            gatherer.error(field,Message.CONFIGURABLE_FIELDS_CRASHES,field.getName(),httpBasicAuthComponent.getName(),module.getName());
        }
    }

    /** This Method is implemented in org.mule.devkit.verifiers.util.VerifierUtils
     TODO: Move VerifierUtils class to a commons module so we can call it from this class.
     */
    public static void verifyConfigurablesGettersAndSetters(ConnectionComponent httpBasicAuthComponent, NotificationGatherer gatherer){
        for(Field field : httpBasicAuthComponent.getConfigurableFields()) {
            if (!field.hasSetter()) {
                gatherer.error(field, Message.CONFIGURABLE_MISSING_SETTER, field.getName());
            }
            if (!field.hasGetter()) {
                gatherer.error(field, Message.CONFIGURABLE_MISSING_GETTER, field.getName());
            }
        }
    }
}
