/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.studio;

import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.generation.utils.VersionUtils;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.module.Module;
import org.mule.util.IOUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.List;

public class MuleStudioManifestGenerator extends AbstractMuleStudioNamespaceGenerator {

    private static final String MANIFEST_FILE_NAME = "META-INF/MANIFEST.MF";

    private static final List<Product> CONSUMES = Arrays.asList(Product.STUDIO_ACTIVATOR);
    private static final List<Product> PRODUCES = Arrays.asList(Product.STUDIO_MANIFEST);
    private static final String STUDIO_CONTRIBUTION = "org.mule.tooling.ui.contribution";

    private EclipsePlugin eclipsePlugin;

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        eclipsePlugin = new EclipsePlugin();

        PrintStream printStream = null;

        ModuleRelativePathBuilder relativePath = new ModuleRelativePathBuilder(MANIFEST_FILE_NAME);

        try {
            OutputStream outputStream = ctx().getCodeModel().getCodeWriter().openBinary(null, relativePath.build(module).getFullPath());
            printStream = new PrintStream(outputStream);
            printStream.append(getManifestContents(module));
            printStream.flush();
        } catch (IOException e) {
            throw new GenerationException("Could not create MANIFEST for Studio plugin: " + e.getMessage(), e);
        } finally {
            IOUtils.closeQuietly(printStream);
        }

        eclipsePlugin.setPath(relativePath);
        ctx().registerProduct(Product.STUDIO_MANIFEST, module, eclipsePlugin);
    }

    public class EclipsePlugin {
        private String symbolicName;
        private ModuleRelativePathBuilder path;

        public EclipsePlugin() {
        }

        public String getSymbolicName() {
            return symbolicName;
        }

        public ModuleRelativePathBuilder getPath() {
            return path;
        }

        public void setSymbolicName(String symbolicName) {
            this.symbolicName = symbolicName;
        }

        public void setPath(ModuleRelativePathBuilder path) {
            this.path = path;
        }
    }

    private String getManifestContents(Module module) {
        StringBuilder manfiestContentBuilder = new StringBuilder(100);
        manfiestContentBuilder.append("Manifest-Version: 1.0\n");
        manfiestContentBuilder.append("Bundle-ManifestVersion: 2\n");

        String bundleName = module.getFriendlyName();
        manfiestContentBuilder.append("Bundle-Name: ").append(bundleName).append("\n");
        String symbolicName = STUDIO_CONTRIBUTION + "." + module.getModuleName();
        manfiestContentBuilder.append("Bundle-SymbolicName: ").append(symbolicName).append(";singleton:=true\n");
        eclipsePlugin.setSymbolicName(symbolicName);

        manfiestContentBuilder.append("Bundle-Version: ").append(VersionUtils.buildVersion(module.getModuleSchemaVersion())).append("\n");
        manfiestContentBuilder.append("Bundle-Activator: ")
                .append(ctx().<GeneratedClass>getProduct(Product.STUDIO_ACTIVATOR, module).fullName())
                .append("\n");
        manfiestContentBuilder.append("Bundle-Vendor: ").append(module.getJavaDocTagContent("author")).append("\n");
        manfiestContentBuilder.append("Require-Bundle: org.eclipse.ui,\n");
        manfiestContentBuilder.append(" org.eclipse.core.runtime,\n");
        manfiestContentBuilder.append(" org.mule.tooling.core;bundle-version=\"1.0.0\"\n");
        manfiestContentBuilder.append("Bundle-RequiredExecutionEnvironment: JavaSE-1.6\n");
        manfiestContentBuilder.append("Bundle-ActivationPolicy: lazy\n");
        manfiestContentBuilder.append("Eclipse-BundleShape: dir\n");
        return manfiestContentBuilder.toString();
    }
}