/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.display.Placement;
import org.mule.api.annotations.param.Default;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.utils.NameUtils;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Identifiable;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Variable;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.schema.SchemaConstants;
import org.mule.devkit.model.studio.*;
import org.mule.util.StringUtils;

import javax.lang.model.type.TypeKind;
import javax.xml.bind.JAXBElement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public abstract class BaseStudioXmlBuilder {

    public static final String GENERAL_GROUP_NAME = "General";
    public static final String CONNECTION_GROUP_NAME = "Connection";
    public static final String CONNECTION_GROUP_LABEL = "Use these fields to override the credentials defined in the %s connector.";
    protected ObjectFactory objectFactory;
    protected MuleStudioUtils helper;
    protected Module module;
    protected Method method;
    protected String moduleName;
    protected Context context;


    protected BaseStudioXmlBuilder(Context context) {
        this.context = context;
        helper = new MuleStudioUtils();
        objectFactory = new ObjectFactory();
    }

    protected BaseStudioXmlBuilder(Context context, Module module) {
        this(context);
        this.module = module;
        moduleName = module.getModuleName();
    }

    protected Group createGroupWithModeSwitch(List<? extends Method> methods) {
        ModeType modeSwitch = new ModeType();
        modeSwitch.getMode().addAll(this.getModes(methods));
        modeSwitch.setCaption(helper.formatCaption("Operation"));
        modeSwitch.setName(StringUtils.capitalize(moduleName) + " operations to execute");
        modeSwitch.setDescription(helper.formatDescription("Operation"));
        modeSwitch.setAlwaysCombo(true);

        Group group = new Group();
        group.setId(module.getModuleName() + "ConnectorGeneric");
        group.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupModeSwitch(modeSwitch));
        group.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.GROUP_DEFAULT_CAPTION));
        return group;
    }

    protected List<ModeElementType> getModes(List<? extends Method> methods) {
        List<ModeElementType> modes = new ArrayList<ModeElementType>();
        for (Method method : methods) {
            ModeElementType mode = new ModeElementType();
            mode.setModeId(MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/' + helper.getLocalId(method));
            mode.setModeLabel(StringUtils.capitalize(helper.getFriendlyName(method)));
            modes.add(mode);
        }

        return modes;
    }

    protected BaseStudioXmlBuilder(Context context, Method method, Module module) {
        this(context, module);
        this.method = method;
    }

    protected List<AttributeCategory> processMethodParameters() {
        Map<String, AttributeCategory> attributeCategoriesByName = processVariableElements(getParametersSorted());

        if(method instanceof ProcessorMethod && ((ProcessorMethod) method).canBeUsedInOAuthManagement()
                && ((ProcessorMethod) method).getOAuthModule().getUserIdentifierMethod() != null
                && ((ProcessorMethod) method).isOAuthProtected()) {
            StringAttributeType accessTokenId = new StringAttributeType();
            accessTokenId.setName("accessTokenId");
            accessTokenId.setRequired(false);
            accessTokenId.setCaption(helper.formatCaption("Access Token Id"));
            accessTokenId.setDescription(helper.formatDescription("The id of the access token that will be used to authenticate the call"));
            accessTokenId.setJavaType("java.lang.String");

            AttributeCategory attributeCategory = null;
            if (attributeCategoriesByName.get(CONNECTION_GROUP_NAME) != null) {
                attributeCategory = attributeCategoriesByName.get(CONNECTION_GROUP_NAME);
            } else {
                attributeCategory = new AttributeCategory();
                attributeCategory.setCaption(helper.formatCaption(CONNECTION_GROUP_NAME));
                attributeCategory.setDescription(CONNECTION_GROUP_NAME);
                attributeCategoriesByName.put(CONNECTION_GROUP_NAME,attributeCategory);
            }

            Group group = getGroup(attributeCategory, CONNECTION_GROUP_NAME);
            if(group == null) {
                group = new Group();
                group.setCaption(helper.formatCaption(CONNECTION_GROUP_NAME));
                group.setId(StringUtils.uncapitalize(CONNECTION_GROUP_NAME));
                attributeCategory.getGroup().add(group);
            }
            group.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(accessTokenId));
        }

        return new ArrayList<AttributeCategory>(attributeCategoriesByName.values());
    }

    private Group getDefaultGroup(AttributeCategory attributeCategory) {
        return getGroup(attributeCategory, GENERAL_GROUP_NAME);
    }

    private Group getGroup(AttributeCategory attributeCategory, String groupName) {
        if(attributeCategory != null) {
            for(Group group : attributeCategory.getGroup()) {
                if(StringUtils.uncapitalize(groupName).equalsIgnoreCase(group.getId())) {
                    return group;
                }
            }
        }
        return null;
    }

    protected List<AttributeCategory> processConfigurableFields(Group defaultGroup) {
        Map<String, AttributeCategory> attributeCategoriesByName = processVariableElements(getConfigurableFieldsSorted());
        List<AttributeCategory> attributeCategories = new ArrayList<AttributeCategory>(attributeCategoriesByName.values());

        AttributeCategory attributeCategory = attributeCategoriesByName.get(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION);
        if (attributeCategory != null) {
            attributeCategory.setDescription(helper.formatDescription(module.getModuleName() + " configuration properties"));
            List<Group> groups = attributeCategory.getGroup();
            if (groups.isEmpty()) {
                groups.add(defaultGroup);
            } else {
                groups.add(0, defaultGroup);
            }
        } else {
            attributeCategory = new AttributeCategory();
            attributeCategory.setCaption(helper.getFormattedCaption(module));
            attributeCategory.setDescription(helper.formatDescription(module.getModuleName() + " configuration properties"));
            attributeCategory.getGroup().add(defaultGroup);
            attributeCategories.add(attributeCategory);
        }
        return attributeCategories;
    }

    private Map<String, AttributeCategory> processVariableElements(List<? extends Variable> variableElements) {

        Map<String, Group> groupsByName = new LinkedHashMap<String, Group>();
        Map<String, AttributeCategory> attributeCategoriesByName = new LinkedHashMap<String, AttributeCategory>();
        getOrCreateDefaultAttributeCategory(attributeCategoriesByName); // Create default category
        processConnectionAttributes(groupsByName, attributeCategoriesByName);
        createPoolingProfileAttributes(groupsByName, attributeCategoriesByName);
        createReconnectionAttributes(groupsByName, attributeCategoriesByName);
        createOAuthConfig(attributeCategoriesByName);
        createMetaDataAttributes(groupsByName, attributeCategoriesByName);

        for (Variable parameter : variableElements) {
            JAXBElement<? extends AttributeType> jaxbElement = createJaxbElement(parameter);
            AttributeCategory attributeCategory = getOrCreateAttributeCategory(attributeCategoriesByName, parameter.getAnnotation(Placement.class));
            Group group = getOrCreateGroup(groupsByName, parameter);
            group.getRegexpOrEncodingOrModeSwitch().add(jaxbElement);

            if (!attributeCategory.getGroup().contains(group)) {
                attributeCategory.getGroup().add(group);
            }
        }

        return attributeCategoriesByName;
    }

    protected void createMetaDataAttributes(Map<String,Group> groupsByName, Map<String,AttributeCategory> attributeCategoriesByName) {
       // override if necessary
    }

    protected void processConnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createReconnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createPoolingProfileAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createOAuthConfig(Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    private AttributeCategory getOrCreateDefaultAttributeCategory(Map<String, AttributeCategory> attributeCategoriesByName) {
        return getOrCreateAttributeCategory(attributeCategoriesByName, null);

    }

    private AttributeCategory getOrCreateAttributeCategory(Map<String, AttributeCategory> attributeCategoriesByName, Placement placement) {
        if (placement == null || StringUtils.isBlank(placement.tab())) {
            if (!attributeCategoriesByName.containsKey(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION)) {
                AttributeCategory attributeCategoryGeneral = new AttributeCategory();
                attributeCategoryGeneral.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION));
                attributeCategoryGeneral.setDescription(helper.formatDescription(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION));
                attributeCategoriesByName.put(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION, attributeCategoryGeneral);
            }
            return attributeCategoriesByName.get(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION);
        } else {
            String attributeCategoryName;
            if (StringUtils.isNotBlank(placement.tab())) {
                attributeCategoryName = placement.tab();
            } else {
                attributeCategoryName = MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION;
            }
            if (!attributeCategoriesByName.containsKey(attributeCategoryName)) {
                AttributeCategory attributeCategory = new AttributeCategory();
                attributeCategory.setCaption(helper.formatCaption(attributeCategoryName));
                attributeCategory.setDescription(helper.formatDescription(attributeCategoryName));
                attributeCategoriesByName.put(attributeCategoryName, attributeCategory);
            }
            return attributeCategoriesByName.get(attributeCategoryName);
        }
    }

    private Group getOrCreateGroup(Map<String, Group> groupsByName, Variable parameter) {
        Placement placement = parameter != null ? parameter.getAnnotation(Placement.class) : null;
        if (placement == null || StringUtils.isBlank(placement.group())) {
            if (!groupsByName.containsKey(GENERAL_GROUP_NAME)) {
                Group groupGeneral = new Group();
                groupGeneral.setCaption(helper.formatCaption(GENERAL_GROUP_NAME));
                groupGeneral.setId(StringUtils.uncapitalize(GENERAL_GROUP_NAME));
                groupsByName.put(GENERAL_GROUP_NAME, groupGeneral);
            }
            return groupsByName.get(GENERAL_GROUP_NAME);
        } else {
            String groupName = placement.group();
            if (!groupsByName.containsKey(groupName)) {
                Group group = new Group();
                group.setCaption(groupName);
                group.setId(StringUtils.uncapitalize(groupName));
                groupsByName.put(groupName, group);
            }
            return groupsByName.get(groupName);
        }
    }

    protected JAXBElement<? extends AttributeType> createJaxbElement(Variable<?> variable) {
        AttributeType attributeType;
       if (variable.isMetaDataKey()) {
            attributeType = createTypeChooserType(variable);
        } else if (variable.asType().isEnum()) {
            attributeType = createEnumType(variable);
        } else if (isComplexList(variable)) {
            attributeType = createComplexListType(variable);
        } else if (isListOfMaps(variable) || isSimpleList(variable) || isSimpleMap(variable) || isSimpleSet(variable)) {
            attributeType = handleCollectionVariable(variable);
        } else if (variable.asType().isComplexType() && !variable.isRefOnly()) {
            NestedElementReference complexTypeNestedElementReference = createNestedElementReference(variable);
            return objectFactory.createGroupChildElement(complexTypeNestedElementReference);
        } else {
            attributeType = createAttributeType(variable);
        }
        return helper.createJAXBElement(attributeType);
    }

    private AttributeType createComplexListType(Variable parameter) {
        ObjectListAttributeType objectListAttributeTyper = new ObjectListAttributeType();
        objectListAttributeTyper.setListName(NameUtils.uncamel(parameter.getName()));
        setCommonCollectionAttributes(parameter, objectListAttributeTyper);
        return objectListAttributeTyper;
    }

    private boolean isComplexList(Variable parameter) {
        return parameter.asType().isArrayOrList() && !parameter.getTypeArguments().isEmpty()
                && parameter.getTypeArguments().get(0).asTypeMirror().getKind().equals(TypeKind.DECLARED)
                && parameter.getTypeArguments().get(0).isComplexTypeWithGetterAndSetter(false);
    }

    private AttributeType createTypeChooserType(Variable<?> variable) {
        TypeChooserType type = new TypeChooserType();
        type.setSupportsExpressions(true);
        type.setAssociatedConfig("config-ref");
        helper.setAttributeTypeInfo(variable, type);
        return type;
    }

    private NestedElementReference createNestedElementReference(Variable parameter) {
        NestedElementReference childElement = new NestedElementReference();
        childElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + '/' + NameUtils.uncamel(parameter.getName()));
        childElement.setCaption(helper.getFormattedCaption(parameter));
        childElement.setAllowMultiple(false);
        childElement.setInplace(true);
        childElement.setJavaType(parameter.getJavaType());
        childElement.setRequired(!parameter.isOptional());
        return childElement;
    }

    private AttributeType handleCollectionVariable(Variable parameter) {
        AttributeType attributeType;
        if (isListOfMaps(parameter)) {
            ListOfMapAttributeType objectListAttributeType = new ListOfMapAttributeType();
            objectListAttributeType.setRequired(!parameter.isOptional());
            objectListAttributeType.setListName(NameUtils.uncamel(parameter.getName()));
            objectListAttributeType.setInnerName(NameUtils.uncamel(SchemaConstants.INNER_PREFIX + NameUtils.singularize(parameter.getName())));
            setCommonCollectionAttributes(parameter, objectListAttributeType);
            attributeType = objectListAttributeType;
        } else if (isSimpleList(parameter) || isSimpleSet(parameter)) {
            StringListAttributeType stringListAttributeTyper = new StringListAttributeType();

            stringListAttributeTyper.setListName(NameUtils.uncamel(parameter.getName()));
            setCommonCollectionAttributes(parameter, stringListAttributeTyper);
            attributeType = stringListAttributeTyper;
        } else {
            StringMapAttributeType stringMapAttributeType = new StringMapAttributeType();
            stringMapAttributeType.setMapName(NameUtils.uncamel(parameter.getName()));
            setCommonCollectionAttributes(parameter, stringMapAttributeType);
            attributeType = stringMapAttributeType;
        }
        // DEVKIT-275 Need required information in Studio
        attributeType.setRequired(!parameter.isOptional());
        return attributeType;
    }

    private void setCommonCollectionAttributes(Variable parameter, CollectionAttributeType collectionAttributeType) {
        collectionAttributeType.setItemName(NameUtils.uncamel(NameUtils.singularize(parameter.getName())));
        collectionAttributeType.setLocalName(helper.getLocalId(method, parameter));
        collectionAttributeType.setCaption(helper.getFormattedCaption(parameter));
        collectionAttributeType.setJavaType(parameter.getJavaType());
        Default annotation = parameter.getAnnotation(Default.class);
        if (annotation != null) {
            collectionAttributeType.setDefaultValue(annotation.value());
        }
        if (method != null) {
            collectionAttributeType.setDescription(helper.formatDescription(method.getJavaDocParameterSummary(parameter.getName())));
        } else {
            collectionAttributeType.setDescription(helper.formatDescription(parameter.getJavaDocSummary()));
        }
    }

    private boolean isListOfMaps(Variable parameter) {
        return parameter.asType().isArrayOrList() && !parameter.getTypeArguments().isEmpty() && parameter.getTypeArguments().get(0).isMap();
    }

    private boolean isSimpleMap(Variable parameter) {
        return parameter.asType().isMap() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(1).isCollection());
    }

    private boolean isSimpleList(Variable parameter) {
        return parameter.asType().isArrayOrList() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }

    private boolean isSimpleSet(Variable parameter) {
        return parameter.asType().isSet() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }

    private List<Parameter> getParametersSorted() {
        List<Parameter> parameters = new ArrayList<Parameter>(method.getParameters());
        Iterator<Parameter> iterator = parameters.iterator();
        while (iterator.hasNext()) {
            if (iterator.next().shouldBeIgnored()) {
                iterator.remove();
            }
        }

        Collections.sort(parameters, new VariableComparator());
        return parameters;
    }

    private AttributeType createAttributeType(Variable parameter) {
        AttributeType attributeType = helper.createAttributeTypeIgnoreEnumsAndCollections(parameter);
        if (attributeType != null) {
            helper.setAttributeTypeInfo(parameter, attributeType);
        }
        return attributeType;
    }

    protected List<AttributeType> getConnectionAttributes(ManagedConnectionModule module) {
        List<AttributeType> parameters = new ArrayList<AttributeType>();
        for (Parameter connectAttributeType : module.getConnectMethod().getParameters()) {
            AttributeType parameter = helper.createAttributeTypeIgnoreEnumsAndCollections(connectAttributeType);
            helper.setAttributeTypeInfo(connectAttributeType, parameter);
            parameter.setRequired(false);
            parameters.add(parameter);
        }
        return parameters;
    }

    private EnumType createEnumType(Variable<?> variable) {
        EnumType enumType = new EnumType();
        enumType.setSupportsExpressions(true);
        enumType.setAllowsCustom(true);
        helper.setAttributeTypeInfo(variable, enumType);
        for (Identifiable<?> enumMember : ((org.mule.devkit.model.EnumType) variable.asType()).getEnumConstants()) {
            String enumConstant = enumMember.getName();
            EnumElement enumElement = new EnumElement();
            enumElement.setValue(enumConstant);
            enumType.getOption().add(enumElement);
        }
        Collections.sort(enumType.getOption(), new EnumElementComparator());
        return enumType;
    }

    private List<Field> getConfigurableFieldsSorted() {
        List<Field> configurableFields = module.getConfigurableFields();
        Collections.sort(configurableFields, new VariableComparator());
        return configurableFields;
    }

    protected String buildVersionsString() {
        /*
         * DEVKIT-368: Studio needs the supported ESB versions in order to support
         * multiple cloud connectors.
         * TODO: Fix once Studio provides a better way to indicate that this is unbounded (a.k.a. remove 8.0.0!!!)
         */
        return "[" + module.getMinMuleVersion().getMajor() + "." + module.getMinMuleVersion().getMinor() + "." + (module.getMinMuleVersion().getRevision() >=0 ? module.getMinMuleVersion().getRevision() : 0)  + ",8.0.0]";

    }
}