/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor;

import org.mule.devkit.generation.api.Context;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthAuthorizationParameter;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.model.studio.AttributeCategory;
import org.mule.devkit.model.studio.AttributeType;
import org.mule.devkit.model.studio.Group;
import org.mule.devkit.model.studio.PatternType;
import org.mule.devkit.model.studio.StringAttributeType;
import org.mule.util.StringUtils;

import javax.xml.bind.JAXBElement;

/**
 * This pattern module builder generates the metadata for the authorize
 * method
 *
 * @author mariano.gonzalez@mulesoft.com
 */
public class OAuthPatternTypeBuilder extends PatternTypeBuilder {
    private String methodName = null;

    public OAuthPatternTypeBuilder(Context context, String methodName, Module module) {
        super(context, null, module);
        this.methodName = methodName;
    }

    @Override
    public PatternType build() {
        return this.createPatternType();
    }

    @Override
    protected PatternType createPatternType() {
        PatternType patternType = new PatternType();

        if("authorize".equalsIgnoreCase(this.methodName)) {
            buildAuthorizePatternType(patternType);
        } else if("unauthorize".equalsIgnoreCase(this.methodName))    {
            buildUnauthorizePatternType(patternType);
        }

        return patternType;
    }

    private void buildAuthorizePatternType(PatternType patternType) {
        patternType.setLocalId(this.methodName);
        patternType.setCaption(helper.formatCaption(this.methodName));
        patternType.setAbstract(true);
        patternType.setExtends(MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/' + helper.getGlobalRefId(module.getModuleName()));
        patternType.setIcon(getIcon());
        patternType.setImage(getImage());

        OAuthModule oAuthModule = (OAuthModule) module;
        boolean hasAdditionalParameters = oAuthModule.getAuthorizationParameters() != null && oAuthModule.getAuthorizationParameters().size() > 0;
        AttributeCategory attributeCategoryGeneral = new AttributeCategory();
        attributeCategoryGeneral.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION));
        attributeCategoryGeneral.setDescription(helper.formatDescription(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION));

        Group generalGroup = new Group();
        generalGroup.setCaption(helper.formatCaption(GENERAL_GROUP_NAME));
        generalGroup.setId(StringUtils.uncapitalize(GENERAL_GROUP_NAME));
        attributeCategoryGeneral.getGroup().add(generalGroup);

        Group authParametersGroup = null;
        if(hasAdditionalParameters) {
            authParametersGroup = new Group();
            authParametersGroup.setCaption(helper.formatCaption("Additional authorization parameters"));
            authParametersGroup.setId("additionalAuthorizationParameters");
            attributeCategoryGeneral.getGroup().add(authParametersGroup);
        }

        StringAttributeType attributeType = new StringAttributeType();
        attributeType.setName("state");
        attributeType.setRequired(false);
        attributeType.setDescription(helper.formatDescription("Any value that you wish to be sent with the callback"));
        attributeType.setCaption(helper.formatCaption("State"));
        generalGroup.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(attributeType));

        attributeType = new StringAttributeType();
        attributeType.setName("accessTokenUrl");
        attributeType.setRequired(false);
        attributeType.setDescription(helper.formatDescription("The URL defined by the Service Provider to obtain an access token"));
        attributeType.setCaption(helper.formatCaption("Access Token URL"));
        generalGroup.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(attributeType));

        attributeType = new StringAttributeType();
        attributeType.setName("authorizationUrl");
        attributeType.setRequired(false);
        attributeType.setDescription(helper.formatDescription("The URL defined by the Service Provider where the resource owner will be redirected to grant authorization to the connector"));
        attributeType.setCaption(helper.formatCaption("Authorization URL"));
        generalGroup.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(attributeType));

        if (hasAdditionalParameters) {
            for (OAuthAuthorizationParameter parameter : oAuthModule.getAuthorizationParameters()) {
                JAXBElement<? extends AttributeType> jaxbElement = createJaxbElement(new ParameterAdapter(parameter));
                authParametersGroup.getRegexpOrEncodingOrModeSwitch().add(jaxbElement);
            }
        }

        patternType.getAttributeCategoryOrRequiredSetAlternativesOrFixedAttribute().add(attributeCategoryGeneral);

        // Add accessTokenId to OAuth2 authorize processor
        if(oAuthModule.getOAuthVersion().equals(OAuthVersion.V2)
                && module.getMinMuleVersion().atLeast("3.5")) {
            StringAttributeType accessTokenId = new StringAttributeType();
            accessTokenId.setName("accessTokenId");
            accessTokenId.setRequired(false);
            accessTokenId.setCaption(helper.formatCaption("Access Token Id"));
            accessTokenId.setDescription(helper.formatDescription("The id of the access token that will be used to authenticate the call"));
            accessTokenId.setJavaType("java.lang.String");

            AttributeCategory attributeCategoryConnection = new AttributeCategory();
            attributeCategoryConnection.setCaption(helper.formatCaption(CONNECTION_GROUP_NAME));
            attributeCategoryConnection.setDescription(CONNECTION_GROUP_NAME);

            patternType.getAttributeCategoryOrRequiredSetAlternativesOrFixedAttribute().add(attributeCategoryConnection);

            Group group = new Group();
            group.setCaption(helper.formatCaption(CONNECTION_GROUP_NAME));
            group.setId(StringUtils.uncapitalize(CONNECTION_GROUP_NAME));
            attributeCategoryConnection.getGroup().add(group);

            group.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(accessTokenId));
        }

    }

    private void buildUnauthorizePatternType(PatternType patternType) {
        patternType.setLocalId(this.methodName);
        patternType.setCaption(helper.formatCaption(this.methodName));
        patternType.setAbstract(true);
        patternType.setExtends(MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/' + helper.getGlobalRefId(module.getModuleName()));
        patternType.setIcon(getIcon());
        patternType.setImage(getImage());

        OAuthModule oAuthModule = (OAuthModule) module;
        AttributeCategory attributeCategoryGeneral = new AttributeCategory();
        attributeCategoryGeneral.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION));
        attributeCategoryGeneral.setDescription(helper.formatDescription(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION));

        Group generalGroup = new Group();
        generalGroup.setCaption(helper.formatCaption(GENERAL_GROUP_NAME));
        generalGroup.setId(StringUtils.uncapitalize(GENERAL_GROUP_NAME));
        attributeCategoryGeneral.getGroup().add(generalGroup);

        StringAttributeType attributeType = new StringAttributeType();
        attributeType.setName("accessTokenId");
        attributeType.setRequired(false);
        attributeType.setDescription(helper.formatDescription("The id of the access token that will be used to authenticate the call"));
        attributeType.setCaption(helper.formatCaption("Access Token Id"));
        generalGroup.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(attributeType));

        patternType.getAttributeCategoryOrRequiredSetAlternativesOrFixedAttribute().add(attributeCategoryGeneral);
    }

    @Override
    protected String getImage() {
        return helper.getConnectorImage(module);
    }

    @Override
    protected String getIcon() {
        return helper.getConnectorIcon(module);
    }
}
