/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.Processor;
import org.mule.api.annotations.Source;
import org.mule.api.annotations.display.FriendlyName;
import org.mule.api.annotations.display.Icons;
import org.mule.api.annotations.display.Password;
import org.mule.api.annotations.display.Path;
import org.mule.api.annotations.display.Summary;
import org.mule.api.annotations.param.Default;
import org.mule.devkit.generation.utils.NameUtils;
import org.mule.devkit.model.Identifiable;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Variable;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.schema.SchemaConstants;
import org.mule.devkit.model.schema.SchemaTypeConversion;
import org.mule.devkit.model.studio.AttributeType;
import org.mule.devkit.model.studio.Booleantype;
import org.mule.devkit.model.studio.ClassType;
import org.mule.devkit.model.studio.EncodingType;
import org.mule.devkit.model.studio.EnumType;
import org.mule.devkit.model.studio.FlowRefType;
import org.mule.devkit.model.studio.Group;
import org.mule.devkit.model.studio.IntegerType;
import org.mule.devkit.model.studio.ListOfMapAttributeType;
import org.mule.devkit.model.studio.MetaDataType;
import org.mule.devkit.model.studio.NestedElementReference;
import org.mule.devkit.model.studio.ObjectFactory;
import org.mule.devkit.model.studio.ObjectListAttributeType;
import org.mule.devkit.model.studio.PasswordType;
import org.mule.devkit.model.studio.PathType;
import org.mule.devkit.model.studio.QueryType;
import org.mule.devkit.model.studio.StringAttributeType;
import org.mule.devkit.model.studio.StringListAttributeType;
import org.mule.devkit.model.studio.StringMapAttributeType;
import org.mule.devkit.model.studio.TextType;
import org.mule.devkit.model.studio.TypeChooserType;
import org.mule.devkit.model.studio.UrlType;
import org.mule.util.StringUtils;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.apache.commons.lang.WordUtils;

public class MuleStudioUtils {

    private static final String IMAGE_PREFIX = "icons/large/";
    private static final String ICON_PREFIX = "icons/small/";

    public String formatCaption(String caption) {
        return WordUtils.capitalizeFully(caption);
    }

    public String formatDescription(String description) {
        if(!StringUtils.isEmpty(description)) {
            if (Character.isLowerCase(description.charAt(0))) {
                description = StringUtils.capitalize(description);
            }
            if (!description.endsWith(".")) {
                description += '.';
            }
        }   else {
            description = "";
        }
        return description.replaceAll("\\<.*?\\>", "");
    }

    public String getConnectorImage(Module module) {
        Icons icons = module.getAnnotation(Icons.class);
        String image;
        if (icons != null) {
            image = icons.connectorLarge();
        } else {
            image = String.format(Icons.GENERIC_CLOUD_CONNECTOR_LARGE, module.getModuleName());
        }
        if (image.contains("/")) {
            image = image.substring(image.lastIndexOf("/") + 1);
        }
        return IMAGE_PREFIX + image;
    }

    public String getConnectorIcon(Module module) {
        Icons icons = module.getAnnotation(Icons.class);
        String icon;
        if (icons != null) {
            icon = icons.connectorSmall();
        } else {
            icon = String.format(Icons.GENERIC_CLOUD_CONNECTOR_SMALL, module.getModuleName());
        }
        if (icon.contains("/")) {
            icon = icon.substring(icon.lastIndexOf("/") + 1);
        }
        return ICON_PREFIX + icon;
    }

    public String getEndpointImage(Module module) {
        Icons icons = module.getAnnotation(Icons.class);
        String image;
        if (icons != null) {
            image = icons.endpointLarge();
        } else {
            image = String.format(Icons.GENERIC_ENDPOINT_LARGE, module.getModuleName());
        }
        if (image.contains("/")) {
            image = image.substring(image.lastIndexOf("/") + 1);
        }
        return IMAGE_PREFIX + image;
    }

    public String getEndpointIcon(Module module) {
        Icons icons = module.getAnnotation(Icons.class);
        String icon;
        if (icons != null) {
            icon = icons.endpointSmall();
        } else {
            icon = String.format(Icons.GENERIC_ENDPOINT_SMALL, module.getModuleName());
        }
        if (icon.contains("/")) {
            icon = icon.substring(icon.lastIndexOf("/") + 1);
        }
        return ICON_PREFIX + icon;
    }

    public String getTransformerImage(Module module) {
        Icons icons = module.getAnnotation(Icons.class);
        String image;
        if (icons != null) {
            image = icons.transformerLarge();
        } else {
            image = String.format(Icons.GENERIC_TRANSFORMER_LARGE, module.getModuleName());
        }
        if (image.contains("/")) {
            image = image.substring(image.lastIndexOf("/") + 1);
        }
        return IMAGE_PREFIX + image;
    }

    public String getTransformerIcon(Module module) {
        Icons icons = module.getAnnotation(Icons.class);
        String icon;
        if (icons != null) {
            icon = icons.transformerSmall();
        } else {
            icon = String.format(Icons.GENERIC_TRANSFORMER_SMALL, module.getModuleName());
        }
        if (icon.contains("/")) {
            icon = icon.substring(icon.lastIndexOf("/") + 1);
        }
        return ICON_PREFIX + icon;
    }

    public String getGlobalRefId(String moduleName) {
        return "abstract" + StringUtils.capitalize(moduleName) + "ConnectorGeneric";
    }

    public List<JAXBElement<? extends AttributeType>> createJAXBElements(List<AttributeType> attributeTypes) {
        List<JAXBElement<? extends AttributeType>> jaxbElements = new ArrayList<JAXBElement<? extends AttributeType>>();
        for (AttributeType attributeType : attributeTypes) {
            JAXBElement<? extends AttributeType> jaxbElement = createJAXBElement(attributeType);
            if (jaxbElement != null) {
                jaxbElements.add(jaxbElement);
            }
        }
        return jaxbElements;
    }

    public JAXBElement<? extends AttributeType> createJAXBElement(AttributeType attributeType) {
        ObjectFactory objectFactory = new ObjectFactory();
        if (attributeType instanceof PasswordType) {
            return objectFactory.createGroupPassword((PasswordType) attributeType);
        }
        if (attributeType instanceof PathType) {
            return objectFactory.createGroupFile(attributeType);
        }
        if (attributeType instanceof UrlType) {
            return objectFactory.createGroupUrl((UrlType) attributeType);
        }
        if (attributeType instanceof StringAttributeType) {
            return objectFactory.createGroupString((StringAttributeType) attributeType);
        }
        if (attributeType instanceof IntegerType) { // TODO: Studio has a problem with LongType, until that's resolved map longs to integer
            return objectFactory.createGroupInteger((IntegerType) attributeType);
        }
        if (attributeType instanceof EnumType) {
            return objectFactory.createGroupEnum((EnumType) attributeType);
        }
        if (attributeType instanceof Booleantype) {
            return objectFactory.createGroupBoolean((Booleantype) attributeType);
        }
        if (attributeType instanceof TextType) {
            return objectFactory.createGroupText((TextType) attributeType);
        }
        if (attributeType instanceof FlowRefType) {
            return objectFactory.createGroupFlowRef((FlowRefType) attributeType);
        }
        if (attributeType instanceof EncodingType) {
            return objectFactory.createGroupEncoding((EncodingType) attributeType);
        }
        if (attributeType instanceof NestedElementReference) {
            return objectFactory.createNestedElementTypeChildElement((NestedElementReference) attributeType);
        }
        if (attributeType instanceof ListOfMapAttributeType) {
            return objectFactory.createGroupObjectList((ListOfMapAttributeType) attributeType);
        }
        if (attributeType instanceof ObjectListAttributeType) {
            return objectFactory.createGroupObjectList((ObjectListAttributeType) attributeType);
        }
        if (attributeType instanceof StringMapAttributeType) {
            return objectFactory.createGroupObjectList((StringMapAttributeType) attributeType);
        }
        if (attributeType instanceof StringListAttributeType) {
            return objectFactory.createGroupObjectList((StringListAttributeType) attributeType);
        }
        if (attributeType instanceof TypeChooserType) {
            return objectFactory.createTypeChooser((TypeChooserType) attributeType);
        }
        if (attributeType instanceof MetaDataType) {
            return objectFactory.createMetaDataType((MetaDataType) attributeType);
        }
        if (attributeType instanceof QueryType) {
            return objectFactory.createQueryType((QueryType) attributeType);
        }

        return null;
    }

    public AttributeType createAttributeTypeIgnoreEnumsAndCollections(Variable element) {
        if (skipAttributeTypeGeneration(element)) {
            return null;
        } else if (SchemaTypeConversion.isSupported(element.getJavaType())) {
            return createAttributeTypeOfSupportedType(element);
        } else if (element.asType().isHttpCallback()) {
            FlowRefType flowRefType = new FlowRefType();
            flowRefType.setSupportFlow(true);
            flowRefType.setSupportSubflow(true);
            return flowRefType;
        } else {
            return new StringAttributeType();
        }
    }

    private boolean skipAttributeTypeGeneration(Identifiable element) {
        return element.asType().isCollection() || element.asType().isEnum() || ((element instanceof Parameter) && ((Parameter) element).shouldBeIgnored());
    }

    private AttributeType createAttributeTypeOfSupportedType(Variable element) {
        if (element.getAnnotation(Password.class) != null) {
            return new PasswordType();
        }
        if (element.getAnnotation(Path.class) != null) {
            return new PathType();
        }
        if (element.asType().isString() || element.asType().isDate() || element.asType().isChar() ||
                element.asType().isFloat() || element.asType().isDouble() || element.asType().isCalendar()) {
            return new StringAttributeType();
        } else if (element.asType().isBoolean()) {
            Booleantype booleantype = new Booleantype();
            booleantype.setSupportsExpressions(true);
            return booleantype;
        } else if(element.getJavaType().equals("java.lang.Class") || element.getJavaType().startsWith("java.lang.Class<")) {
            return new ClassType();
        } else if (element.asType().isInteger() || element.asType().isLong() || element.asType().isBigDecimal() || element.asType().isBigInteger()) {
            IntegerType integerType = new IntegerType();
            integerType.setMin(0);
            integerType.setStep(1);
            return integerType;
        } else if (element.asType().isURL()) {
            return new UrlType();
        } else {
            throw new RuntimeException("Failed to create Studio XML, type not recognized: type=" + element.asTypeMirror().toString() + " name=" + element.getName());
        }
    }

    public void setAttributeTypeInfo(Variable variable, AttributeType attributeType) {
        String parameterName = variable.getName();
        attributeType.setCaption(getFormattedCaption(variable));
        attributeType.setDescription(getFormattedDescription(variable));
        if (attributeType instanceof StringAttributeType && !SchemaTypeConversion.isSupported(variable.getJavaType())) {
            attributeType.setName(parameterName + SchemaConstants.REF_SUFFIX);
        } else if (attributeType instanceof FlowRefType) {
            attributeType.setName(NameUtils.uncamel(parameterName) + SchemaConstants.FLOW_REF_SUFFIX);
        } else {
            attributeType.setName(parameterName);
        }
        attributeType.setRequired(!variable.isOptional());
        attributeType.setJavaType(variable.getJavaType());
        setDefaultValueIfAvailable(variable, attributeType);
    }

    public void setDefaultValueIfAvailable(Variable variable, AttributeType parameter) {
        Default annotation = variable.getAnnotation(Default.class);
        if (annotation != null) {
            if (parameter instanceof Booleantype) {
                ((Booleantype) parameter).setDefaultValue(Boolean.valueOf(annotation.value()));
            } else if (parameter instanceof IntegerType) {
                ((IntegerType) parameter).setDefaultValue(Integer.valueOf(annotation.value()));
            } else if (parameter instanceof StringAttributeType) {
                ((StringAttributeType) parameter).setDefaultValue(annotation.value());
            } else if (parameter instanceof EnumType) {
                ((EnumType) parameter).setDefaultValue(annotation.value());
            }
        }
    }

    public String getLocalId(Method executableElement, Variable variable) {
        if (executableElement != null) {
            return NameUtils.uncamel(executableElement.getName()) + '-' + NameUtils.uncamel(variable.getName());
        } else {
            return "configurable-" + NameUtils.uncamel(variable.getName());
        }
    }

    public String getLocalId(Method executableElement) {
        String localId;
        Processor processor = executableElement.getAnnotation(Processor.class);
        if (processor != null && StringUtils.isNotBlank(processor.name())) {
            localId = processor.name();
        } else {
            localId = executableElement.getName();
        }
        return NameUtils.uncamel(localId);
    }

    public String getFormattedDescription(Variable element) {
        Summary description = element.getAnnotation(Summary.class);
        if (description != null && StringUtils.isNotBlank(description.value())) {
            return formatDescription(description.value());
        }
        if (element instanceof Parameter) {
            return formatDescription(element.parent().getJavaDocParameterSummary(element.getName()));
        }
        return formatDescription(element.getJavaDocSummary());
    }

    public String getFormattedDescription(Module module) {
        if (StringUtils.isNotBlank(module.getDescription())) {
            return module.getDescription();
        }
        return formatDescription(module.getJavaDocSummary());
    }

    public String getFormattedCaption(Module module) {
        if (StringUtils.isNotBlank(module.getFriendlyName())) {
            return module.getFriendlyName();
        }
        return formatCaption(module.getModuleName().replaceAll("-", " "));
    }

    public String getFormattedCaption(Method element) {
        return formatCaption(getFriendlyName(element));
    }

    public String getFormattedCaption(Variable element) {
        FriendlyName caption = element.getAnnotation(FriendlyName.class);
        if (caption != null && StringUtils.isNotBlank(caption.value())) {
            return caption.value();
        }
        String friendlyName = NameUtils.friendlyNameFromCamelCase(element.getName());
        if (element.asType().isHttpCallback()) {
            return formatCaption(friendlyName + " Flow");
        }
        if (!isKnownType(element)) {
            return formatCaption(friendlyName + " Reference");
        }
        return formatCaption(friendlyName);
    }

    public String getFriendlyName(Method element) {
        Processor processor = element.getAnnotation(Processor.class);
        if (processor != null && StringUtils.isNotBlank(processor.friendlyName())) {
            return processor.friendlyName();
        }
        Source source = element.getAnnotation(Source.class);
        if (source != null && StringUtils.isNotBlank(source.friendlyName())) {
            return source.friendlyName();
        }
        return NameUtils.friendlyNameFromCamelCase(element.getName());
    }

    public boolean isKnownType(Variable variable) {
        return variable.asType().isString() ||
                variable.asType().isChar() ||
                variable.asType().isDate() ||
                variable.asType().isDouble() ||
                variable.asType().isFloat() ||
                variable.asType().isLong() ||
                variable.asType().isHttpCallback() ||
                variable.asType().isInteger() ||
                variable.asType().isBigDecimal() ||
                variable.asType().isBigInteger() ||
                variable.asType().isBoolean() ||
                variable.asType().isEnum() ||
                variable.asType().isCollection() ||
                variable.asType().isURL();
    }

    public String getUrl(Module module) {
        return MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/';
    }

    public StringAttributeType createStringAttributeType(String caption, String description, String name) {
        StringAttributeType stringAttributeType = new StringAttributeType();
        stringAttributeType.setCaption(formatCaption(caption));
        stringAttributeType.setDescription(formatDescription(description));
        stringAttributeType.setName(name);
        return stringAttributeType;
    }

    public Group createGroup(String id, String caption) {
        Group group = new Group();
        group.setId(id);
        group.setCaption(formatCaption(caption));
        return group;
    }

    public String getPagedReturnType(Method method) {
        String returnType = List.class.getCanonicalName();
        if (!method.getReturnGenericType().getGenericTypeArguments().isEmpty()) {
            returnType += "<";
            returnType += method.getReturnGenericType().getGenericTypeArguments().get(0).getType().toString();
            returnType += ">";
        }
        return returnType;
    }
}