package org.mule.devkit.generation.studio;

import org.mule.api.annotations.display.Icons;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.MultiModuleAnnotationVerifier;
import org.mule.devkit.generation.api.gatherer.*;
import org.mule.devkit.generation.api.gatherer.Message;
import org.mule.devkit.generation.studio.utils.MuleStudioPathResolver;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;

import java.io.File;
import java.util.*;


public class MultiModuleStudioPackageVerifier implements MultiModuleAnnotationVerifier {

    @Override
    public boolean shouldVerify(List<Module> modules) {
        return true;
    }

    @Override
    public List<Module> processableModules(List<Module> modules) {
        List<Module> specificModules= new ArrayList<Module>();
        for(Module module: modules){
            if(!module.getKind().equals(ModuleKind.METADATA_CATEGORY)){
                specificModules.add(module);
            }
        }
        return specificModules;
    }

    @Override
    public void verify(List<Module> modules, NotificationGatherer gatherer) throws AnnotationVerificationException {
        /**
         * this condition ensures that if an {@link org.mule.api.annotations.ExpressionLanguage} is used, it won't
         * generate the update site for Studio if there is not a @Connector or @Module in the same project
         * */
        if (!hasConnectorOrModule(modules)){
            gatherer.error(modules.get(0), Message.DEVKIT_PACKAGE_PROPERTY_REQUIRES_AT_LEAST_ONE_MODULE);
        }else {
            //Module validation
            verifyModules(modules, gatherer);
        }
    }

    private void verifyModules(List<Module> modules, NotificationGatherer gatherer) {
        List<Module> modulesToBeValidated=  getModuleByKind(modules, Arrays.asList(ModuleKind.CONNECTOR, ModuleKind.GENERIC));
        for (Module module: modulesToBeValidated){
            verifyIcon(module, gatherer);
        }
    }

    private void verifyIcon(Module module, NotificationGatherer gatherer) {
        Icons icons = module.getAnnotation(Icons.class);

        String connectorSmall = String.format(Icons.GENERIC_CLOUD_CONNECTOR_SMALL, module.getModuleName());
        String connectorLarge = String.format(Icons.GENERIC_CLOUD_CONNECTOR_LARGE, module.getModuleName());

        String transformerSmall = String.format(Icons.GENERIC_TRANSFORMER_SMALL, module.getModuleName());
        String transformerLarge = String.format(Icons.GENERIC_TRANSFORMER_LARGE, module.getModuleName());

        String endpointSmall = String.format(Icons.GENERIC_ENDPOINT_SMALL, module.getModuleName());
        String endpointLarge = String.format(Icons.GENERIC_ENDPOINT_LARGE, module.getModuleName());

        if (icons != null) {
            connectorSmall = icons.connectorSmall();
            connectorLarge = icons.connectorLarge();

            transformerSmall = icons.transformerSmall();
            transformerLarge = icons.transformerLarge();

            endpointSmall = icons.endpointSmall();
            endpointLarge = icons.endpointLarge();
        }

        checkFile(connectorSmall, module, gatherer);
        checkFile(connectorLarge, module, gatherer);
        if (module.hasTransformers()) {
            checkFile(transformerSmall, module, gatherer);
            checkFile(transformerLarge, module, gatherer);
        }
        if (module.hasSources()) {
            checkFile(endpointSmall, module, gatherer);
            checkFile(endpointLarge, module, gatherer);
        }
    }
    private void checkFile(String fileName, Module module, NotificationGatherer gatherer)  {
        File fileToCopy = MuleStudioPathResolver.getFileResolvingPackages(module, fileName);
        if (!fileToCopy.exists()) {
            gatherer.error(module, Message.MISSING_ICON,module.getModuleName(),fileToCopy.getAbsolutePath());
        }
    }

    private List<Module> getModuleByKind(List<Module> modules, List<ModuleKind> kinds){
        List<Module> specificModules= new ArrayList<Module>();
        for(Module module: modules){
            if(kinds.contains(module.getKind())){
                specificModules.add(module);
            }
        }
        return specificModules;
    }


    private boolean hasConnectorOrModule(List<Module> modules) {
        boolean hasConnectorOrModule = false;
        for (Module module : modules) {
            if (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC){
                hasConnectorOrModule = true;
            }
        }
        return hasConnectorOrModule;
    }
}
