/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.Category;
import org.mule.api.annotations.Paged;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.Transformer;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.connectivity.ConnectionManagementCapability;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.studio.AttributeCategory;
import org.mule.devkit.model.studio.AttributeType;
import org.mule.devkit.model.studio.Group;
import org.mule.devkit.model.studio.PatternType;
import org.mule.util.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;

public class PatternTypeBuilder extends BaseStudioXmlBuilder {

    public PatternTypeBuilder(Context context, Method method, Module module) {
        super(context, method, module);
    }

    public PatternType build() {
        PatternType patternType = createPatternType();
        if (method.getAnnotation(Processor.class) != null) {
            Collection<AttributeCategory> attributeCategories = processMethodParameters();
            patternType.getAttributeCategoryOrRequiredSetAlternativesOrFixedAttribute().addAll(attributeCategories);
        }
        return patternType;
    }

    protected PatternType createPatternType() {
        PatternType patternType = new PatternType();
        patternType.setLocalId(helper.getLocalId(method));
        patternType.setCaption(helper.getFormattedCaption(method));
        patternType.setAbstract(true);
        if (method.getAnnotation(Processor.class) != null) {
            patternType.setExtends(helper.getUrl(module) + helper.getGlobalRefId(module.getModuleName()));
            String returnType = method.getReturnType().toString();

            if (method.getAnnotation(Paged.class) != null) {
                returnType = helper.getPagedReturnType(method);
            }

            patternType.setReturnType(returnType);
        } else if (method.getAnnotation(Transformer.class) != null) {
            patternType.setExtends(helper.getUrl(module) + AbstractTransformerBuilder.ABSTRACT_TRANSFORMER_LOCAL_ID);
            patternType.setDescription(helper.formatDescription(method.getJavaDocSummary()));
        }

        Category category = module.getAnnotation(Category.class);
        if (category != null && StringUtils.isNotBlank(category.name())) {
            patternType.setCategory(category.name());
        }

        patternType.setIcon(getIcon());
        patternType.setImage(getImage());
        return patternType;
    }

    protected String getImage() {
        if (method.getAnnotation(Transformer.class) != null) {
            return helper.getTransformerImage(module);
        } else {
            return helper.getConnectorImage(module);
        }
    }

    protected String getIcon() {
        if (method.getAnnotation(Transformer.class) != null) {
            return helper.getTransformerIcon(module);
        } else {
            return helper.getConnectorIcon(module);
        }
    }

    /**
     * Used when describing a processor, if the current module has connection management or if it has an strategy
     * with @ConnectionManagement that also has been annotated with @OverrideAtProcessors, then the processor must have
     * the parameters of @Connect as attributes as well.
     *
     * @param groupsByName
     * @param attributeCategoriesByName
     */
    @Override
    protected void processConnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        ConnectionManagementCapability connectionManagementCapability = null;
        if (module.manager().connectionManagementOverride().isPresent()){
            connectionManagementCapability = module.manager().connectionManagementOverride().get();
        } else if(module instanceof ManagedConnectionModule) {
            connectionManagementCapability = (ConnectionManagementCapability) module;
        }

        if (connectionManagementCapability != null ) {
            Group connectionAttributesGroup = new Group();
            connectionAttributesGroup.setCaption(helper.formatCaption(ADVANCED_CONNECTION_GROUP_NAME));
            connectionAttributesGroup.setId(StringUtils.uncapitalize(ADVANCED_CONNECTION_GROUP_NAME));

            groupsByName.put(ADVANCED_CONNECTION_GROUP_NAME, connectionAttributesGroup);

            List<AttributeType> connectionAttributes = getConnectionAttributes(connectionManagementCapability);
            connectionAttributesGroup.getRegexpOrEncodingOrModeSwitch().addAll(helper.createJAXBElements(connectionAttributes));

            AttributeCategory connectionAttributeCategory = new AttributeCategory();
            connectionAttributeCategory.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.ADVANCED_ATTRIBUTE_CATEGORY_CAPTION));
            connectionAttributeCategory.setDescription(helper.formatDescription(MuleStudioEditorXmlGenerator.ADVANCED_ATTRIBUTE_CATEGORY_CAPTION));
            attributeCategoriesByName.put(MuleStudioEditorXmlGenerator.ADVANCED_ATTRIBUTE_CATEGORY_CAPTION, connectionAttributeCategory);
            connectionAttributeCategory.getGroup().add(connectionAttributesGroup);
        }
    }
}