/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.generation.studio;

import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.editor.MuleStudioUtils;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.verifiers.util.MulePathResolver;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.util.IOUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class MuleStudioIconsGenerator extends AbstractMuleStudioNamespaceGenerator {


    private static List<Product> PRODUCES = Arrays.asList(Product.STUDIO_ICONS);

    private ArrayList<ModuleRelativePathBuilder> iconsFileName;
    private static final String ICONS_SMALL = "icons/small";
    private static final String ICONS_LARGE = "icons/large";

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        iconsFileName = new ArrayList<>();

        createCloudConnectorIcon(module);

        if (module.hasTransformers()) {
            createTransformerIcon(module);
        }

        if (module.hasSources()) {
            createSourceIcon(module);
        }

        if (hasContainer(module)){
            createContainerIcon(module);
        }

        if (hasContainerList(module)){
            createFlowIcon(module);
        }

        ctx().registerProduct(Product.STUDIO_ICONS, module, iconsFileName);
    }

    private boolean hasContainer(Module module)
    {
        for (ProcessorMethod processorMethod : module.getProcessorMethods())
        {
            if (processorMethod.isContainer()){
                return true;
            }
        }
        return false;
    }

    private boolean hasContainerList(Module module)
    {
        for (ProcessorMethod processorMethod : module.getProcessorMethods())
        {
            if (processorMethod.isContainerList()){
                return true;
            }
        }
        return false;
    }

    private void createSourceIcon(Module module) throws GenerationException {
        copyFile(String.format(MuleStudioUtils.GENERIC_ENDPOINT_SMALL, module.getModuleName()), ICONS_SMALL, module, MuleStudioUtils.SUFIX_ENDPOINT_24X16_PNG);
        copyFile(String.format(MuleStudioUtils.GENERIC_ENDPOINT_LARGE, module.getModuleName()), ICONS_LARGE, module, MuleStudioUtils.SUFIX_ENDPOINT_48X32_PNG);
    }

    private void createTransformerIcon(Module module) throws GenerationException {
        copyFile(String.format(MuleStudioUtils.GENERIC_TRANSFORMER_SMALL, module.getModuleName()), ICONS_SMALL, module, MuleStudioUtils.SUFIX_TRANSFORMER_24X16_PNG);
        copyFile(String.format(MuleStudioUtils.GENERIC_TRANSFORMER_LARGE, module.getModuleName()), ICONS_LARGE, module, MuleStudioUtils.SUFIX_TRANSFORMER_48X32_PNG);
    }

    private void createCloudConnectorIcon(Module module) throws GenerationException {
        copyFile(String.format(MuleStudioUtils.GENERIC_CLOUD_CONNECTOR_SMALL, module.getModuleName()),ICONS_SMALL , module, MuleStudioUtils.SUFIX_CONNECTOR_24X16_PNG);
        copyFile(String.format(MuleStudioUtils.GENERIC_CLOUD_CONNECTOR_LARGE, module.getModuleName()), ICONS_LARGE, module, MuleStudioUtils.SUFIX_CONNECTOR_48X32_PNG);
    }

    private void createContainerIcon(Module module) throws GenerationException {
        copyFile(String.format(MuleStudioUtils.GENERIC_CONTAINER_SMALL, module.getModuleName()),ICONS_SMALL , module, MuleStudioUtils.SUFIX_CONTAINER_24X16_PNG);
        copyFile(String.format(MuleStudioUtils.GENERIC_CONTAINER_LARGE, module.getModuleName()), ICONS_LARGE, module, MuleStudioUtils.SUFIX_CONTAINER_48X32_PNG);
    }

    private void createFlowIcon(Module module) throws GenerationException {
        copyFile(String.format(MuleStudioUtils.GENERIC_FLOW_SMALL, module.getModuleName()),ICONS_SMALL , module, MuleStudioUtils.SUFIX_FLOW_24X16_PNG);
        copyFile(String.format(MuleStudioUtils.GENERIC_FLOW_LARGE, module.getModuleName()), ICONS_LARGE, module, MuleStudioUtils.SUFIX_FLOW_48X32_PNG);
    }

    private void copyFile(String fileName, String folder, Module module, String sufix) throws GenerationException {
        InputStream iconStream = null;
        OutputStream outputStream = null;
        try {
            outputStream = createFile(module, folder, fileName);
            File fileToCopy = MulePathResolver.getFileResolvingPackages(module, fileName);
            if (fileToCopy.exists()) {
                //taking the icons from the connectors/icons folder
                iconStream = new FileInputStream(fileToCopy);
            }else{
                //using the default ones
                String fileIconPath = String.format(sufix, "generic");
                iconStream = MuleStudioIconsGenerator.class.getClassLoader().getResourceAsStream(fileIconPath);
            }
            IOUtils.copy(iconStream, outputStream);
        } catch (IOException e) {
            throw new GenerationException("Error copying icons to output folder: " + e.getMessage(), e);
        } finally {
            IOUtils.closeQuietly(outputStream);
            IOUtils.closeQuietly(iconStream);
        }
    }

    private OutputStream createFile(Module module, String folder, String fileName) throws GenerationException {
        if (fileName.contains("/")) {
            fileName = fileName.substring(fileName.lastIndexOf("/") + 1);
        }
        try {
            ModuleRelativePathBuilder pathBuilder = new ModuleRelativePathBuilder(folder + "/" + fileName);
            String fullIconPath = pathBuilder.build(module).getFullPath();
            iconsFileName.add(pathBuilder);
            return ctx().getCodeModel().getCodeWriter().openBinary(null, fullIconPath);
        } catch (IOException e) {
            throw new GenerationException("Could not create file or folder " + fileName + ": " + e.getMessage(), e);
        }
    }
}