/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.generation.studio.editor;

import com.google.common.base.Optional;
import org.apache.commons.lang.WordUtils;
import org.mule.api.annotations.display.FriendlyName;
import org.mule.api.annotations.display.Password;
import org.mule.api.annotations.display.Path;
import org.mule.api.annotations.display.Summary;
import org.mule.api.annotations.param.Default;
import org.mule.devkit.model.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.schema.SchemaConstants;
import org.mule.devkit.model.schema.SchemaTypeConversion;
import org.mule.devkit.model.studio.*;
import org.mule.devkit.model.studio.ClassType;
import org.mule.devkit.model.studio.EnumType;
import org.mule.devkit.model.studio.collection.*;
import org.mule.devkit.model.studio.metadata.MultiTypeChooserType;
import org.mule.devkit.model.studio.metadata.TypeChooserType;
import org.mule.devkit.utils.NameUtils;
import org.mule.module.http.api.requester.HttpRequesterConfig;
import org.mule.util.StringUtils;

import javax.xml.bind.JAXBElement;
import java.util.ArrayList;
import java.util.List;

public class MuleStudioUtils {

    private static final String IMAGE_PREFIX = "icons/large/";
    private static final String ICON_PREFIX = "icons/small/";
    public static final String ICONS_FOLDEER = "../../../icons/";

    public static final String SUFIX_CONNECTOR_24X16_PNG = "%s-connector-24x16.png";
    public static final String SUFIX_CONNECTOR_48X32_PNG = "%s-connector-48x32.png";
    public static final String SUFIX_TRANSFORMER_24X16_PNG = "%s-transformer-24x16.png";
    public static final String SUFIX_TRANSFORMER_48X32_PNG = "%s-transformer-48x32.png";
    public static final String SUFIX_ENDPOINT_24X16_PNG = "%s-endpoint-24x16.png";
    public static final String SUFIX_ENDPOINT_48X32_PNG = "%s-endpoint-48x32.png";
    public static final String SUFIX_CONTAINER_24X16_PNG = "%s-container-24x16.png";
    public static final String SUFIX_CONTAINER_48X32_PNG = "%s-container-48x32.png";
    public static final String SUFIX_FLOW_24X16_PNG = "%s-flow-24x16.png";
    public static final String SUFIX_FLOW_48X32_PNG = "%s-flow-48x32.png";


    public static final String GENERIC_CLOUD_CONNECTOR_SMALL = ICONS_FOLDEER + SUFIX_CONNECTOR_24X16_PNG;
    public static final String GENERIC_CLOUD_CONNECTOR_LARGE = ICONS_FOLDEER + SUFIX_CONNECTOR_48X32_PNG;
    public static final String GENERIC_TRANSFORMER_SMALL = ICONS_FOLDEER + SUFIX_TRANSFORMER_24X16_PNG;
    public static final String GENERIC_TRANSFORMER_LARGE = ICONS_FOLDEER + SUFIX_TRANSFORMER_48X32_PNG;
    public static final String GENERIC_ENDPOINT_SMALL = ICONS_FOLDEER + SUFIX_ENDPOINT_24X16_PNG;
    public static final String GENERIC_ENDPOINT_LARGE = ICONS_FOLDEER + SUFIX_ENDPOINT_48X32_PNG;
    public static final String GENERIC_CONTAINER_SMALL = ICONS_FOLDEER + SUFIX_CONTAINER_24X16_PNG;
    public static final String GENERIC_CONTAINER_LARGE = ICONS_FOLDEER + SUFIX_CONTAINER_48X32_PNG;
    public static final String GENERIC_FLOW_SMALL = ICONS_FOLDEER + SUFIX_FLOW_24X16_PNG;
    public static final String GENERIC_FLOW_LARGE = ICONS_FOLDEER + SUFIX_FLOW_48X32_PNG;

    public String formatCaption(String caption) {
        return WordUtils.capitalizeFully(caption);
    }

    public String formatDescription(String description) {
        if(!StringUtils.isEmpty(description)) {
            if (Character.isLowerCase(description.charAt(0))) {
                description = StringUtils.capitalize(description);
            }
            if (!description.endsWith(".")) {
                description += '.';
            }
        }   else {
            description = "";
        }
        return description.replaceAll("\\<.*?\\>", "");
    }

    public String getConnectorImage(Module module) {
       return getImage(module,GENERIC_CLOUD_CONNECTOR_LARGE,IMAGE_PREFIX);
    }

    public String getConnectorIcon(Module module) {
        return getImage(module,GENERIC_CLOUD_CONNECTOR_SMALL,ICON_PREFIX);
    }

    public String getEndpointImage(Module module) {
        return getImage(module, GENERIC_ENDPOINT_LARGE, IMAGE_PREFIX);
    }

    public String getEndpointIcon(Module module) {
        return getImage(module, GENERIC_ENDPOINT_SMALL, ICON_PREFIX);
    }

    public String getTransformerImage(Module module) {
        return getImage(module, GENERIC_TRANSFORMER_LARGE, IMAGE_PREFIX);
    }

    public String getTransformerIcon(Module module) {
        return getImage(module, GENERIC_TRANSFORMER_SMALL, ICON_PREFIX);
    }

    public String getContainerImage(Module module) {
        return getImage(module, GENERIC_CONTAINER_LARGE, IMAGE_PREFIX);
    }

    public String getContainerIcon(Module module) {
        return getImage(module, GENERIC_CONTAINER_SMALL, ICON_PREFIX);
    }

    public String getFlowImage(Module module) {
        return getImage(module, GENERIC_FLOW_LARGE, IMAGE_PREFIX);
    }

    public String getFlowIcon(Module module) {
        return getImage(module, GENERIC_FLOW_SMALL, ICON_PREFIX);
    }

    public String getImage(Module module, String path, String prefix){
        String image = String.format(path, module.getModuleName());
        if (image.contains("/")) {
            image = image.substring(image.lastIndexOf("/") + 1);
        }
        return prefix + image;
    }

    public String getGlobalRefId(String moduleName) {
        return "abstract" + StringUtils.capitalize(moduleName) + "ConnectorGeneric";
    }

    public List<JAXBElement<? extends AttributeType>> createJAXBElements(List<AttributeType> attributeTypes) {
        List<JAXBElement<? extends AttributeType>> jaxbElements = new ArrayList<JAXBElement<? extends AttributeType>>();
        for (AttributeType attributeType : attributeTypes) {
            JAXBElement<? extends AttributeType> jaxbElement = createJAXBElement(attributeType);
            if (jaxbElement != null) {
                jaxbElements.add(jaxbElement);
            }
        }
        return jaxbElements;
    }

    public JAXBElement<? extends AttributeType> createJAXBElement(AttributeType attributeType) {
        ObjectFactory objectFactory = new ObjectFactory();
        if (attributeType instanceof PasswordType) {
            return objectFactory.createGroupPassword((PasswordType) attributeType);
        }
        if (attributeType instanceof PathType) {
            return objectFactory.createGroupFile(attributeType);
        }
        if (attributeType instanceof UrlType) {
            return objectFactory.createGroupUrl((UrlType) attributeType);
        }
        if (attributeType instanceof StringAttributeType) {
            return objectFactory.createGroupString((StringAttributeType) attributeType);
        }
        if (attributeType instanceof IntegerType) { // TODO: Studio has a problem with LongType, until that's resolved map longs to integer
            return objectFactory.createGroupInteger((IntegerType) attributeType);
        }
        if (attributeType instanceof EnumType) {
            return objectFactory.createGroupEnum((EnumType) attributeType);
        }
        if (attributeType instanceof Booleantype) {
            return objectFactory.createGroupBoolean((Booleantype) attributeType);
        }
        if (attributeType instanceof TextType) {
            return objectFactory.createGroupText((TextType) attributeType);
        }
        if (attributeType instanceof FlowRefType) {
            return objectFactory.createGroupFlowRef((FlowRefType) attributeType);
        }
        if (attributeType instanceof EncodingType) {
            return objectFactory.createGroupEncoding((EncodingType) attributeType);
        }
        if (attributeType instanceof NestedElementReference) {
            return objectFactory.createNestedElementTypeChildElement((NestedElementReference) attributeType);
        }
        if (attributeType instanceof ListOfMapNoExpressionType) {
            return objectFactory.createGroupObjectList((ListOfMapNoExpressionType) attributeType);
        }
        if (attributeType instanceof ObjectListNoExpressionType) {
            return objectFactory.createGroupObjectList((ObjectListNoExpressionType) attributeType);
        }
        if (attributeType instanceof StringMapNoExpressionType) {
            return objectFactory.createGroupObjectList((StringMapNoExpressionType) attributeType);
        }
        if (attributeType instanceof StringListNoExpressionType) {
            return objectFactory.createGroupObjectList((StringListNoExpressionType) attributeType);
        }
        if (attributeType instanceof ListOfMapAttributeType) {
            return objectFactory.createGroupObjectList((ListOfMapAttributeType) attributeType);
        }
        if (attributeType instanceof ObjectListAttributeType) {
            return objectFactory.createGroupObjectList((ObjectListAttributeType) attributeType);
        }
        if (attributeType instanceof StringMapAttributeType) {
            return objectFactory.createGroupObjectList((StringMapAttributeType) attributeType);
        }
        if (attributeType instanceof StringListAttributeType) {
            return objectFactory.createGroupObjectList((StringListAttributeType) attributeType);
        }
        if (attributeType instanceof MultiTypeChooserType) {
            return objectFactory.createMultiTypeChooser((MultiTypeChooserType) attributeType);
        }
        if (attributeType instanceof TypeChooserType) {
            return objectFactory.createTypeChooser((TypeChooserType) attributeType);
        }
        if (attributeType instanceof MetaDataType) {
            return objectFactory.createMetaDataType((MetaDataType) attributeType);
        }
        if (attributeType instanceof QueryType) {
            return objectFactory.createQueryType((QueryType) attributeType);
        }

        return null;
    }

    public AttributeType createAttributeTypeIgnoreEnumsAndCollections(Variable element) {
        if (skipAttributeTypeGeneration(element)) {
            return null;
        }
        if (SchemaTypeConversion.isSupported(element.getJavaType())) {
            return createAttributeTypeOfSupportedType(element);
        }
        if (element.asType().isHttpCallback()) {
            FlowRefType flowRefType = new FlowRefType();
            flowRefType.setSupportFlow(true);
            flowRefType.setSupportSubflow(true);
            return flowRefType;
        }

        return new StringAttributeType();

    }

    private boolean skipAttributeTypeGeneration(Identifiable element) {
        return (element.asType().isCollection() ||
                element.asType().isEnum() ||
                ((element instanceof Parameter) && ((Parameter) element).shouldBeIgnored())) && !((Parameter) element).isRefOnly();
    }

    private AttributeType createAttributeTypeOfSupportedType(Variable element) {
        if (element.getAnnotation(Password.class) != null) {
            return new PasswordType();
        }
        if (element.getAnnotation(Path.class) != null) {
            return new PathType();
        }

        Type elementType = element.asType();
        if (elementType.isString() || elementType.isDate() || elementType.isChar() || elementType.isFloat() || elementType.isDouble() || elementType.isCalendar()) {
            return new StringAttributeType();
        }
        if (elementType.isBoolean()) {
            Booleantype booleantype = new Booleantype();
            booleantype.setSupportsExpressions(true);
            return booleantype;
        }
        if(element.getJavaType().equals("java.lang.Class") || element.getJavaType().startsWith("java.lang.Class<")) {
            return new ClassType();
        }
        if (elementType.isInteger() || elementType.isLong() ||
            elementType.isBigDecimal() || elementType.isBigInteger() ||
            elementType.isShort() || elementType.isByte())
        {
            IntegerType integerType = new IntegerType();
            integerType.setMin(0);
            integerType.setStep(1);
            return integerType;
        }
        if (elementType.isURL()) {
            return new UrlType();
        }

        throw new RuntimeException("Failed to create Studio XML, type not recognized: type=" + element.asTypeMirror().toString() + " name=" + element.getName());

    }

    public void setAttributeTypeInfo(Variable variable, AttributeType attributeType) {
        String parameterName = variable.getName();
        attributeType.setCaption(getFormattedCaption(variable));
        attributeType.setDescription(getFormattedDescription(variable));
        if (attributeType instanceof StringAttributeType && !SchemaTypeConversion.isSupported(variable.getJavaType())) {
            attributeType.setName(parameterName + SchemaConstants.REF_SUFFIX);
        } else if (attributeType instanceof FlowRefType) {
            attributeType.setName(NameUtils.uncamel(parameterName) + SchemaConstants.FLOW_REF_SUFFIX);
        } else {
            attributeType.setName(parameterName);
        }
        attributeType.setRequired(!variable.isOptional());
        attributeType.setJavaType(variable.getJavaType());
        setDefaultValueIfAvailable(variable, attributeType);
    }

    public void setDefaultValueIfAvailable(Variable variable, AttributeType parameter) {
        Default annotation = variable.getAnnotation(Default.class);
        if (annotation != null) {
            if (parameter instanceof Booleantype) {
                ((Booleantype) parameter).setDefaultValue(Boolean.valueOf(annotation.value()));
            } else if (parameter instanceof IntegerType) {
                ((IntegerType) parameter).setDefaultValue(Integer.valueOf(annotation.value()));
            } else if (parameter instanceof StringAttributeType) {
                ((StringAttributeType) parameter).setDefaultValue(annotation.value());
            } else if (parameter instanceof EnumType) {
                ((EnumType) parameter).setDefaultValue(annotation.value());
            }
        }
    }

    public String getLocalId(Method executableElement, Variable variable) {
        if (executableElement != null) {
            return NameUtils.uncamel(executableElement.getName()) + '-' + NameUtils.uncamel(variable.getName());
        } else {
            return "configurable-" + NameUtils.uncamel(variable.getName());
        }
    }

    public String getLocalId(Method executableElement) {
        return executableElement.xsdName();
    }

    public String getFormattedDescription(Variable element) {
        Summary description = element.getAnnotation(Summary.class);
        if (description != null && StringUtils.isNotBlank(description.value())) {
            return formatDescription(description.value());
        }
        if (element instanceof Parameter) {
            return formatDescription(element.parent().getJavaDocParameterSummary(element.getName()));
        }
        return formatDescription(element.getJavaDocSummary());
    }

    public String getFormattedDescription(Module module) {
        if (StringUtils.isNotBlank(module.getDescription())) {
            return module.getDescription();
        }
        return formatDescription(module.getJavaDocSummary());
    }

    public String getFormattedCaption(Module module) {
        if (StringUtils.isNotBlank(module.getFriendlyName())) {
            return module.getFriendlyName();
        }
        return formatCaption(module.getModuleName().replaceAll("-", " "));
    }

    public String getFormattedCaption(Method element) {
        return formatCaption(getFriendlyName(element));
    }

    public String getFormattedCaption(Variable element) {
        FriendlyName caption = element.getAnnotation(FriendlyName.class);
        if (caption != null && StringUtils.isNotBlank(caption.value())) {
            return caption.value();
        }
        String friendlyName = NameUtils.friendlyNameFromCamelCase(element.getName());
        if (element.asType().isHttpCallback()) {
            return formatCaption(friendlyName + " Flow");
        }
        if (!isKnownType(element) || element.isRefOnly()) {
            return formatCaption(friendlyName + " Reference");
        }
        return formatCaption(friendlyName);
    }

    public String getFriendlyName(Method element) {
        return element.friendlyName();
    }

    public boolean isKnownType(Variable variable) {
        return variable.asType().isString() ||
                variable.asType().isChar() ||
                variable.asType().isDate() ||
                variable.asType().isDouble() ||
                variable.asType().isFloat() ||
                variable.asType().isLong() ||
                variable.asType().isHttpCallback() ||
                variable.asType().isInteger() ||
                variable.asType().isBigDecimal() ||
                variable.asType().isBigInteger() ||
                variable.asType().isBoolean() ||
                variable.asType().isEnum() ||
                variable.asType().isCollection() ||
                variable.asType().isURL() ||
                variable.asType().inheritsFrom(HttpRequesterConfig.class);
    }

    public String getUrl(Module module) {
        return MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/';
    }

    public StringAttributeType createStringAttributeType(String caption, String description, String name) {
        StringAttributeType stringAttributeType = new StringAttributeType();
        stringAttributeType.setCaption(formatCaption(caption));
        stringAttributeType.setDescription(formatDescription(description));
        stringAttributeType.setName(name);
        return stringAttributeType;
    }

    public Group createGroup(String id, String caption) {
        Group group = new Group();
        group.setId(id);
        group.setCaption(formatCaption(caption));
        return group;
    }

    public String getPagedReturnType(Method method) {
        String returnType = List.class.getCanonicalName();
        if (!method.getReturnGenericType().getGenericTypeArguments().isEmpty()) {
            returnType += "<";
            returnType += method.getReturnGenericType().getGenericTypeArguments().get(0).getType().toString();
            returnType += ">";
        }
        return returnType;
    }

    public Optional<ProcessorMethod> getProcessorMethod(Module module, Method method) {
        for (ProcessorMethod processorMethod : module.getProcessorMethods()) {
            if (StringUtils.equals(processorMethod.getName(), method.getName())){
                return Optional.of(processorMethod);
            }
        }

        return Optional.absent();
    }

    public boolean isCollection(Variable parameter) {
        return isListOfMaps(parameter) || isSimpleList(parameter) || isSimpleMap(parameter) || isSimpleSet(parameter);
    }

    public boolean isListOfMaps(Variable parameter) {
        return parameter.asType().isArrayOrList() && !parameter.getTypeArguments().isEmpty() && parameter.getTypeArguments().get(0).isMap();
    }

    public boolean isSimpleMap(Variable parameter) {
        return parameter.asType().isMap() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(1).isCollection());
    }

    public boolean isSimpleList(Variable parameter) {
        return parameter.asType().isArrayOrList() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }

    public boolean isSimpleSet(Variable parameter) {
        return parameter.asType().isSet() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }
}