/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.devkit.generation.studio;

import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.License;
import org.mule.devkit.generation.api.MultiModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.packaging.FutureFile;
import org.mule.devkit.generation.studio.packaging.JarArchiverWrapper;
import org.mule.devkit.generation.studio.packaging.Placeholder;
import org.mule.devkit.generation.studio.utils.ModuleComparator;
import org.mule.devkit.generation.utils.VersionUtils;
import org.mule.devkit.model.module.Module;

import java.util.*;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.mule.devkit.model.module.ModuleKind;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * <p>Studio Feature builder. This is needed to publish the plugin in an Update Site.</p>
 * <p/>
 * <p>This class will generate a <plugin-name>-feature_<version>.qualifier.jar file in the target/update-site/features</p>
 */
public class MuleStudioFeatureGenerator extends AbstractMuleStudioGenerator implements MultiModuleGenerator {

    public static final String FEATURE_XML_FILENAME = "feature.xml";

    public static final String LABEL_VERSION_FORMAT = "(Mule %s+)";

    private static final List<Product> CONSUMES = Arrays.asList(Product.STUDIO_PLUGINS, Product.STUDIO_SITE_XML, Product.STUDIO_MANIFEST, Product.STUDIO_FEATURE_NAME);
    private static final List<Product> PRODUCES = Arrays.asList(Product.STUDIO_FEATURE);

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(List<Module> modules) {
        for (Module module : modules) {
            if (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) {
                return true;
            }
        }

        return false;
    }

    @Override
    public List<Module> processableModules(List<Module> modules) {
        List<Module> processableModules = new ArrayList<Module>();
        for (Module module : modules) {
            if (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) {
                processableModules.add(module);
            }
        }
        return processableModules;
    }

    /**
     * See {@link AbstractMuleStudioNamespaceGenerator#createListOfNamespaces(java.util.List)}. This code is duplicated because of DEVKIT-548
     * @param modules
     * @return
     */
    private Collection<Module> createListOfNamespaces(List<Module> modules) {
        Map<String, Module> namespaces = new HashMap<String, Module>();

        /**
         * sorting the modules because of DEVKIT-548
         */
        ArrayList<Module> sortedModules = new ArrayList<Module>(modules);
        Collections.sort(sortedModules, new ModuleComparator());

        for (Module module : sortedModules) {
            if ( (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC)
                && (!namespaces.containsKey(module.getModuleName()))) {
                namespaces.put(module.getModuleName(), module);
            }
        }

        return namespaces.values();
    }

    @Override
    public void generate(List<Module> modules) throws GenerationException {
        Collection<Module> modulesPerNamespaces = createListOfNamespaces(modules);

        /** Checks if it the artifactId start's with mule-module-XXX and in that case
         * it removes it.
         */
        String featureId = getFeatureId(modulesPerNamespaces);

        try {
            DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
            Document document = documentBuilder.newDocument();

            Element feature = document.createElement("feature");


            feature.setAttribute("id", featureId);

            feature.setAttribute("label", getLabel(modules, ctx().getMavenInformation().getName()));

            feature.setAttribute("version", VersionUtils.buildVersion(ctx().getMavenInformation().getVersion()));
            feature.setAttribute("provider-name", "Mulesoft, Inc.");
            document.appendChild(feature);

            Element licenseElement = document.createElement("license");
            populateLicense(licenseElement);
            feature.appendChild(licenseElement);

            for (Module module : modulesPerNamespaces) {
                MuleStudioManifestGenerator.EclipsePlugin product = ctx().getProduct(Product.STUDIO_MANIFEST, module);
                if (product != null) {
                    Element plugin = document.createElement("plugin");
                    plugin.setAttribute("id", product.getSymbolicName());
                    plugin.setAttribute("download-size", "0");
                    plugin.setAttribute("install-size", "0");
                    plugin.setAttribute("version", VersionUtils.buildVersion(ctx().getMavenInformation().getVersion()));
                    plugin.setAttribute("unpack", "true");
                    feature.appendChild(plugin);
                }
            }

            TransformerFactory transformerFactory = TransformerFactory.newInstance();
            Transformer transformer = transformerFactory.newTransformer();

            DOMSource source = new DOMSource(document);

            StreamResult result = new StreamResult(ctx().getCodeModel().getCodeWriter().openBinary(null, FEATURE_XML_FILENAME));
            transformer.transform(source, result);

        } catch (Exception e) {
            throw new GenerationException("Error generating Mule Studio feature.xml", e);
        }

        JarArchiverWrapper archiver = new JarArchiverWrapper();
        archiver.addFile(new FutureFile(Placeholder.CLASSES_DIRECTORY, FEATURE_XML_FILENAME), FEATURE_XML_FILENAME);

        for (Module module : modulesPerNamespaces) {
            archiver.addFile(ctx().<JarArchiverWrapper>getProduct(Product.STUDIO_PLUGINS, module));
        }

        archiver.setDestFile(new FutureFile(Placeholder.FEATURES_DIRECTORY, featureId + "_" + VersionUtils.buildVersion(ctx().getMavenInformation().getVersion()) + ".jar"));

        ctx().registerProduct(Product.STUDIO_FEATURE, archiver);
    }

    private void populateLicense(Element licenseElement) {
        License license = ctx().getMavenInformation().getLicense();

        if (license != null) {

            String licenseUrl = license.getUrl();

            if (licenseUrl != null) {
                licenseElement.setAttribute("url", licenseUrl);
            }

            licenseElement.setTextContent(license.getLicenseBody());
            licenseElement.setNodeValue(license.getLicenseBody());

        }
    }

}
