/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.devkit.generation.studio;

import org.codehaus.plexus.archiver.ArchiverException;
import org.mule.devkit.generation.api.ConnectorCategory;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.packaging.*;
import org.mule.devkit.generation.utils.VersionUtils;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.module.Module;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * <p>Studio plugin jar file builder</p>
 * <p/>
 * <p>This class generates a <plugin-name>_version.qualifier.jar in the target/update-site/plugins folder</p>
 */
public class MuleStudioPluginGenerator extends AbstractMuleStudioNamespaceGenerator {

    private static final List<Product> CONSUMES = Arrays.asList(Product.STUDIO_ACTIVATOR, Product.STUDIO_MANIFEST,
            Product.STUDIO_PLUGIN_XML, Product.STUDIO_EDITOR_XML, Product.STUDIO_ICONS, Product.STUDIO_CONTEXTUAL_HELP);

    private static final List<Product> PRODUCES = Arrays.asList(Product.STUDIO_PLUGINS);


    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    private boolean isManifest(Path fileName) {
        return fileName.getFullPath().endsWith("MF");
    }

    private void archiveFile(JarArchiverWrapper archiver, String fileName) throws RuntimeException, ArchiverException {
        FutureFile muleZipFile = new FutureFile(Placeholder.ROOT_DIRECTORY, fileName);

        archiver.addFile(muleZipFile, fileName);
    }

    @Override
    public void generate(Module module) throws GenerationException {

        FutureFile studioPlugin = null;
        JarArchiverWrapper archiver = new JarArchiverWrapper();

        String muleAppName = ctx().getMavenInformation().getArtifactId() + "-" + ctx().getMavenInformation().getVersion();

        try {
            archiveFile(archiver, muleAppName + ".zip");
            archiveFile(archiver, muleAppName + ".jar");
            if (ConnectorCategory.COMMUNITY.name().equalsIgnoreCase(getCategory())) archiveFile(archiver, muleAppName + "-sources.jar");
        } catch (ArchiverException e) {
            throw new RuntimeException("Error while packaging Studio plugin", e);
        }

        for (Product product : CONSUMES) {
            Object oneOrMoreFileNames = ctx().getProduct(product, module);
            List<Path> fileNames = new ArrayList<Path>();

            if (oneOrMoreFileNames instanceof List && ((List) oneOrMoreFileNames).get(0) instanceof ModuleRelativePathBuilder) {
                for (ModuleRelativePathBuilder relativePath : (Collection<? extends ModuleRelativePathBuilder>) oneOrMoreFileNames) {
                    fileNames.add(relativePath.build(module));
                }
            } else if (oneOrMoreFileNames instanceof PathBuilder) {
                fileNames.add(((PathBuilder) oneOrMoreFileNames).build(module));
            } else if (oneOrMoreFileNames instanceof GeneratedClass) {
                fileNames.add(new SimplePath((GeneratedClass) oneOrMoreFileNames));
            } else if (oneOrMoreFileNames instanceof MuleStudioManifestGenerator.EclipsePlugin) {
                MuleStudioManifestGenerator.EclipsePlugin eclipsePlugin = (MuleStudioManifestGenerator.EclipsePlugin) oneOrMoreFileNames;
                studioPlugin = new FutureFile(Placeholder.PLUGINS_DIRECTORY, eclipsePlugin.getSymbolicName() + "_" + VersionUtils.buildVersion(ctx().getMavenInformation().getVersion()) + ".jar");
                fileNames.add(eclipsePlugin.getPath().build(module));
            } else {
                throw new RuntimeException("Unsupported product instance: " + oneOrMoreFileNames + ", while looking up " +product.toString());
            }

            for (Path fileName : fileNames) {
                FutureFile file = new FutureFile(Placeholder.CLASSES_DIRECTORY, fileName.getFullPath());

                if (isManifest(fileName)) {
                    archiver.setManifest(file);
                } else {
                    archiver.addFile(file, fileName.getRelativeToModulePath());
                }
            }
        }

        if (studioPlugin == null) {
            throw new RuntimeException("Studio Plugin path was not set by the ManifestGenerator");
        }
        archiver.setDestFile(studioPlugin);

        ctx().registerProduct(Product.STUDIO_PLUGINS, module, archiver);
    }

    private String getCategory() {
        return ctx().getMavenInformation().getCategory();
    }


}

