/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.ConnectStrategy;
import org.mule.api.annotations.MetaDataScope;
import org.mule.api.annotations.display.UserDefinedMetaData;
import org.mule.api.callback.HttpCallback;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.api.MultiModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.spring.global.factory.AbstractGlobalElementResolver;
import org.mule.devkit.generation.studio.AbstractMuleStudioGenerator;
import org.mule.devkit.generation.studio.editor.callback.HttpCallbackNestedElementBuilder;
import org.mule.devkit.generation.studio.editor.callback.OAuthConfigNestedElementsBuilder;
import org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorConfigurationBuilder;
import org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorConnectionManagementBuilder;
import org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorHttpBasicAuthBuilder;
import org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorModuleBuilder;
import org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorOAuthBuilder;
import org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorWsdlProviderBuilder;
import org.mule.devkit.generation.studio.editor.ws.InvokeWsdlPatternTypeBuilder;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.generation.studio.utils.ModuleComparator;
import org.mule.devkit.generation.utils.OAuth2StrategyUtilsResolver;
import org.mule.devkit.generation.utils.global.element.GlobalElementFactory;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.SourceMethod;
import org.mule.devkit.model.module.components.connection.ConfigurationComponent;
import org.mule.devkit.model.module.components.connection.ConnectionManagementComponent;
import org.mule.devkit.model.module.components.connection.WsdlProviderComponent;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.studio.AbstractElementType;
import org.mule.devkit.model.studio.ContainerType;
import org.mule.devkit.model.studio.EndpointType;
import org.mule.devkit.model.studio.GlobalType;
import org.mule.devkit.model.studio.NamespaceType;
import org.mule.devkit.model.studio.NestedElementType;
import org.mule.devkit.model.studio.ObjectFactory;
import org.mule.devkit.model.studio.PatternType;
import org.mule.devkit.model.studio.StudioModel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.lang.model.type.DeclaredType;
import javax.xml.bind.JAXBElement;

import org.apache.commons.lang.StringUtils;

public class MuleStudioEditorXmlGenerator extends AbstractMuleStudioGenerator implements MultiModuleGenerator {

    public static final String URI_PREFIX = "http://www.mulesoft.org/schema/mule/";
    public static final String ATTRIBUTE_CATEGORY_DEFAULT_CAPTION = "General";
    public static final String ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION = "General";
    public static final String ADVANCED_ATTRIBUTE_CATEGORY_CAPTION = "Advanced";
    public static final String GROUP_DEFAULT_CAPTION = "Basic Settings";
    private ObjectFactory objectFactory = new ObjectFactory();
    private final static List<Product> PRODUCES = Arrays.asList(Product.STUDIO_EDITOR_XML);
    private static final String EDITOR_XML_FILE_NAME = "editors.xml";

    @Override
    public List<Product> consumes() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(List<Module> modules) {
        for (Module module : modules) {
            if (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) {
                return true;
            }
        }
        return false;
    }

    @Override
    public List<Module> processableModules(List<Module> modules) {
        List<Module> specificModules = new ArrayList<Module>();
        for (Module module : modules) {
            if (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) {
                specificModules.add(module);
            }
        }
        return specificModules;
    }

    @Override
    public void generate(List<Module> modules) {
        List<Module> connectorOrModule = getSortedConnectorAndOauthModules(modules);
        for (Module module : connectorOrModule) {
            generateModule(module);
        }
    }

    /**
     * Before processing the modules we need to make sure that (for multi-module scenarios, connection management + oauth),
     * the first modules to work with are the connection management ones.
     * <p>If the Oauth are the first ones to be parsed, the generation of the editors.xml will be messy, and have lots
     * of inconsistent information regarding the connector</p>
     */
    private List<Module> getSortedConnectorAndOauthModules(List<Module> modules) {
        List<Module> connectorOrModule = new ArrayList<Module>(modules);
        Collections.sort(connectorOrModule, new ModuleComparator());
        return connectorOrModule;
    }

    public static class PatternTypeOperationsChooser implements StudioModel.BuilderWithArgs<Boolean, JAXBElement<PatternType>> {

        private Context ctx;
        private Module module;

        public PatternTypeOperationsChooser(Context ctx, Module module) {
            this.ctx = ctx;
            this.module = module;
        }

        @Override
        public JAXBElement<PatternType> build(Boolean isOAuth) {
            if (isOAuth) {
                return new OAuthPatternTypeOperationsBuilder(ctx, module, PatternTypes.CLOUD_CONNECTOR).build();
            } else {
                return new PatternTypeOperationsBuilder(ctx, module, PatternTypes.CLOUD_CONNECTOR).build();
            }
        }

    }

    public static class ProcessorMethodsChooser implements StudioModel.BuilderWithArgs<Boolean, List<JAXBElement<? extends AbstractElementType>>> {

        private Module module;
        private ObjectFactory objectFactory;
        private Context ctx;

        public ProcessorMethodsChooser(Context ctx, Module module, ObjectFactory objectFactory) {
            this.module = module;
            this.ctx = ctx;
            this.objectFactory = objectFactory;
        }

        @Override
        public List<JAXBElement<? extends AbstractElementType>> build(Boolean isOAuth) {
            List<JAXBElement<? extends AbstractElementType>> list = new ArrayList<JAXBElement<? extends AbstractElementType>>();

            for (ProcessorMethod processorMethod : module.getProcessorMethods()) {
                PatternType cloudConnector = new PatternTypeBuilder(ctx, processorMethod, module).build();
                fillPatternType(processorMethod, cloudConnector);

                if (processorMethod.isContainer()){
                    list.add(objectFactory.createNamespaceTypeContainer((ContainerType) cloudConnector));
                }else if (processorMethod.isContainerList()){
                    list.add(objectFactory.createNamespaceTypeFlow(cloudConnector));
                }else{
                    list.add(objectFactory.createNamespaceTypeCloudConnector(cloudConnector));
                }
            }

            if (isOAuth) {
                PatternType authorize = new OAuthPatternTypeBuilder(ctx, "authorize", module).build();
                list.add(objectFactory.createNamespaceTypeCloudConnector(authorize));

                PatternType unAuthorize = new OAuthPatternTypeBuilder(ctx, "unauthorize", module).build();
                list.add(objectFactory.createNamespaceTypeCloudConnector(unAuthorize));

            }

            if (!module.manager().wsdlProviderComponent().isEmpty()){
                PatternType invoke = new InvokeWsdlPatternTypeBuilder(ctx, module).build();
                list.add(objectFactory.createNamespaceTypeCloudConnector(invoke));
            }

            return list;
        }

        private void fillPatternType(ProcessorMethod processorMethod, PatternType cloudConnector)
        {
            if (processorMethod.hasInputOrOutputDynamicMetaData() || processorMethod.hasQuery() || processorMethod.hasStaticKeyMetaData()
                    || processorMethod.hasMetaDataScope()) {
                cloudConnector.setMetaData("dynamic");
            } else if (processorMethod.hasDynamicMetaData()) {
                cloudConnector.setMetaData("static");
            }
            if (processorMethod.hasStaticKeyOutputMetaData()) {
                cloudConnector.setMetaDataStaticKey(processorMethod.getStaticKeyOutputMetaData().type());
            }
            List<String> categories = new ArrayList<String>();
            List<DeclaredType> annotationValuesCategories = new ArrayList<DeclaredType>();
            if (processorMethod.hasMetaDataScope()) {
                annotationValuesCategories.add(processorMethod.metaDataScope());
            } else {
                if (module.getAnnotation(MetaDataScope.class) != null) {
                    annotationValuesCategories.add(module.metaDataScope());
                }
            }
            for (DeclaredType declaredType : annotationValuesCategories) {
                String fullQualifiedName = declaredType.toString();
                categories.add(fullQualifiedName.substring(fullQualifiedName.lastIndexOf(".") + 1));
            }
            if (!categories.isEmpty()) {
                cloudConnector.setCategories(StringUtils.join(categories, ","));
            }
            if (processorMethod.hasAnnotation(UserDefinedMetaData.class)){
                //We set the value to true in processors iff it's a TransformingValue and it has been annotated
                cloudConnector.setSupportsUserDefinedMetaData(true);
            }
        }

        private boolean hasListNestedProcessor(ProcessorMethod processorMethod)
        {
            for (Parameter methodParameter : processorMethod.getParameters())
            {
                if(methodParameter.asType().isArrayOrList() &&
                   methodParameter.getTypeArguments().size() > 0 &&
                   methodParameter.getTypeArguments().get(0).isNestedProcessor()){
                    return true;
                }
            }
            return false;
        }

        private boolean hasNestedProcessor(ProcessorMethod processorMethod)
        {
            for (Parameter methodParameter : processorMethod.getParameters())
            {
                if(methodParameter.asType().isNestedProcessor()){
                    return true;
                }
            }
            return false;
        }

    }

    private void executeOncePerNamespace(NamespaceType namespace, Module module) {
        String moduleName = module.getModuleName();

        namespace.setPrefix(moduleName);
        namespace.setUrl(URI_PREFIX + moduleName);

        ctx().getStudioModel().addPatternTypeOperation(moduleName, new PatternTypeOperationsChooser(ctx(), module));

        if(needsGlobalCloudConnectorElement(module)) {
            GlobalType globalCloudConnector = new ParentCloudConnectorTypeBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalCloudConnector(globalCloudConnector));

            if (module instanceof ManagedConnectionModule || !module.manager().connectionManagementComponents().isEmpty() || module.manager().hasConnectionManagedWsdlProvider()) {
                if (hasStrategy(module, ConnectStrategy.SINGLE_INSTANCE)) {
                    NestedElementType cacheConfigNestedElementType = new CacheConfigNestedElementBuilder(ctx(), module).build();
                    namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(cacheConfigNestedElementType));
                }
                if (hasStrategy(module, ConnectStrategy.MULTIPLE_INSTANCES)) {
                    NestedElementType poolingProfileNestedElementType = new PoolingProfileNestedElementBuilder(ctx(), module).build();
                    namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(poolingProfileNestedElementType));
                }

                NestedElementType reconnectionNestedElement = new ReconnectionNestedElementBuilder(ctx(), module).build();
                namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(reconnectionNestedElement));
            }
        }

        processTransformerMethods(module, namespace);
        processSourceMethods(module, namespace);
    }

    /**
     * TODO:refactor needed, this method is copied and pasted in {@link org.mule.devkit.generation.connectivity.PoolGenerator} and {@link BaseStudioXmlBuilder}
     *
     * @param module   module to test
     * @param strategy specific strategy to test against
     * @return true if the connector, or if any of its strategies with @ConnectionManagement, has @Connect method using the same
     * strategy as the parametrized. False otherwise
     */
    private boolean hasStrategy(Module module, ConnectStrategy strategy) {
        if (module instanceof ManagedConnectionModule) {
            if (((ManagedConnectionModule) module).getConnectMethod().getStrategy().equals(strategy)) {
                return true;
            }
        } else {
            for (ConnectionManagementComponent connectionManagementComponent : module.manager().connectionManagementComponents()) {
                if (connectionManagementComponent.getConnectMethod().getStrategy().equals(strategy)) {
                    return true;
                }
            }

            for (WsdlProviderComponent wsdlComponent : module.manager().wsdlProviderComponent()) {
                if (wsdlComponent.hasConnectionManagement() && wsdlComponent.getConnectMethod().getStrategy().equals(strategy)) {
                    return true;
                }
            }
        }

        return false;
    }

    private void generateModule(Module module) {
        String moduleName = module.getModuleName();
        boolean isOAuth = module instanceof OAuthModule || module.manager().oauth2Component().isPresent();
        Context ctx = ctx();
        StudioModel studioModel = ctx.getStudioModel();
        NamespaceType namespace = studioModel.getOrCreateNamespace(module.getModuleName());

        if (isOAuth) {
            studioModel.addIsOAuth(moduleName, isOAuth);
            studioModel.addNestedElements(moduleName, new OAuthConfigNestedElementsBuilder(ctx, module));
        }

        if (module.hasProcessorMethodWithParameter(HttpCallback.class)) {
            studioModel.addNestedElements(moduleName, new HttpCallbackNestedElementBuilder(ctx, module));
        }

        studioModel.addProcessorMethods(moduleName, new ProcessorMethodsChooser(ctx, module, objectFactory));
        studioModel.addNestedElements(moduleName, new NestedsBuilder(ctx, module));

        if (!moduleName.equals(namespace.getPrefix())) {
            executeOncePerNamespace(namespace, module);
        }

        if (needsGlobalCloudConnectorElement(module)) {
            StudioModel.ConfigRefBuilder<JAXBElement<? extends AbstractElementType>> simpleConfigRefBuilder = studioModel.getConfigBuilderRef(moduleName);
            if (studioModel.getConfigBuilderRef(moduleName) == null && (needsGlobalCloudConnectorElement(module))) {
                simpleConfigRefBuilder = new SimpleConfigRefBuilder(ctx(), module);
                studioModel.addConfigBuilderRef(module.getModuleName(), simpleConfigRefBuilder);
            }

            generateGlobalCloudConnector(module, namespace, simpleConfigRefBuilder);
        }


        ModuleRelativePathBuilder editorXMLPath = new ModuleRelativePathBuilder(EDITOR_XML_FILE_NAME);
        studioModel.addNamespaceType(moduleName, editorXMLPath.build(module).getFullPath());
        ctx.registerProduct(Product.STUDIO_EDITOR_XML, module, editorXMLPath);
    }

    private boolean needsGlobalCloudConnectorElement(Module module) {
        return  (hasProcessorsWithoutContainers(module))
                || module.hasSources()
                || module.hasFilters()
                || !module.manager().wsdlProviderComponent().isEmpty();
    }

    /**
     * @return true if has at least one processor and it's not a container type, false otherwise
     */
    private boolean hasProcessorsWithoutContainers(Module module)
    {
        for (ProcessorMethod processorMethod : module.getProcessorMethods())
        {
            if (!(processorMethod.isContainer() || processorMethod.isContainerList())){
                return true;
            }
        }
        return false;
    }

    /**
     * This method will detect if the Module has several strategies or just one (the old behaviour), and accordingly to it,
     * it will add several global-cloud-connector elements or just one (respectively).
     * For each global cloud connector, it will also add a reference in the requiredTypes to let Studio render them in a fashionable way :)
     *
     * @param module
     * @param namespace
     * @param simpleConfigRefBuilder
     */
    private void generateGlobalCloudConnector(Module module, NamespaceType namespace, StudioModel.ConfigRefBuilder<JAXBElement<? extends AbstractElementType>> simpleConfigRefBuilder) {
        List<GlobalType> globalCloudConnectors = getGlobalTypes(module);
        for (GlobalType globalCloudConnector : globalCloudConnectors) {
            //adding the global cloud connector to the editors
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalCloudConnector(globalCloudConnector));
            //making that connector visible to Studio, in this case getLocalId works as the "config" name (aka: config, config-with-oauth, etc.)
            simpleConfigRefBuilder.addRequiredType(globalCloudConnector.getLocalId());
        }
    }

    /**
     * Returns the resolvers for the different types of global elements that the current {@code module} supports.
     *
     * @param module
     * @return
     */
    private List<GlobalType> getGlobalTypes(final Module module) {

        return GlobalElementFactory.getGlobalElementBeanDefinitionGenerator(module, new AbstractGlobalElementResolver<GlobalType>() {
            /**
             * we don't support the OAuthModule case in this resolver, just the strategy
             * @param module
             * @return
             */
            @Override
            public boolean supportsOAuth(Module module) {
                return OAuth2StrategyUtilsResolver.hasOAuth2Component(module);
            }

            /**
             * we don't support the ManagedConnectionModule case in this resolver, just the strategy
             * @param module
             * @return
             */
            @Override
            public boolean supportsConnectionManagement(Module module) {
                return !module.manager().connectionManagementComponents().isEmpty();
            }

            @Override
            public GlobalType genericGlobalElement() {
                return new GlobalCloudConnectorModuleBuilder(ctx(), module, false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build();
            }

            @Override
            public List<? extends GlobalType> connectionManagementGlobalElement() {
                List<GlobalType> globalCloudConnectors = new ArrayList<GlobalType>();
                for (ConnectionManagementComponent connectionManagementComponent : module.manager().connectionManagementComponents()) {
                    globalCloudConnectors.add(new GlobalCloudConnectorConnectionManagementBuilder(ctx(), module, connectionManagementComponent, false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build());
                }
                return globalCloudConnectors;
            }

            @Override
            public GlobalType oauthGlobalElement() {
                return new GlobalCloudConnectorOAuthBuilder(ctx(), module, module.manager().oauth2Component().get(), false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build();
            }

            @Override
            public GlobalType httpBasicAuthGlobalElement() {
                return new GlobalCloudConnectorHttpBasicAuthBuilder(ctx(), module, module.manager().httpBasicAuthComponent().get(), false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build();
            }

            @Override
            public List<? extends GlobalType> basicGlobalElement() {
                List<GlobalType> globalCloudConnectors = new ArrayList<GlobalType>();
                for (ConfigurationComponent configurationComponent : module.manager().configurationComponents()) {
                    globalCloudConnectors.add(new GlobalCloudConnectorConfigurationBuilder(ctx(), module, configurationComponent, false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build());
                }
                return globalCloudConnectors;
            }

            @Override
            public List<GlobalType> wsdlProviderGlobalElement() {
                List<GlobalType> globalCloudConnectors = new ArrayList<GlobalType>();
                for (WsdlProviderComponent wsdlProviderComponent : module.manager().wsdlProviderComponent()) {
                    globalCloudConnectors.add(new GlobalCloudConnectorWsdlProviderBuilder(ctx(), module, wsdlProviderComponent, false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build());
                }
                return globalCloudConnectors;
            }
        });
    }

    private void processTransformerMethods(Module module, NamespaceType namespace) {
        if (module.hasTransformers()) {
            namespace.getConnectorOrEndpointOrGlobal().add(new PatternTypeOperationsBuilder(ctx(), module, PatternTypes.TRANSFORMER).build());
            namespace.getConnectorOrEndpointOrGlobal().add(new AbstractTransformerBuilder(ctx(), module).build());
            GlobalType globalTransformer = new GlobalTransformerTypeOperationsBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalTransformer(globalTransformer));
        }
        for (Method transformerMethod : module.getTransformerMethods()) {
            PatternType transformer = new PatternTypeBuilder(ctx(), transformerMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeTransformer(transformer));
            GlobalType globalTransformer = new GlobalTransformerTypeBuilder(ctx(), transformerMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalTransformer(globalTransformer));
        }
    }

    private void processSourceMethods(Module module, NamespaceType namespace) {
        List<SourceMethod> sourceMethods = module.getSourceMethods();
        if (!sourceMethods.isEmpty()) {
            EndpointType endpointTypeListingOps = new EndpointTypeOperationsBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createCloudConnectorEndpoint(endpointTypeListingOps));
        }
        for (Method sourceMethod : sourceMethods) {
            EndpointType endpoint = new EndpointTypeBuilder(ctx(), sourceMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createCloudConnectorEndpoint(endpoint));
        }
    }
}