/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.display.FriendlyName;
import org.mule.api.annotations.display.Placement;
import org.mule.devkit.model.Variable;

import java.util.Comparator;

import javax.lang.model.element.VariableElement;

import org.apache.commons.lang.StringUtils;

public class VariableComparator implements Comparator<Variable> {

    private static final int VARIABLE1_FIRST = -1;
    private static final int VARIABLE2_FIRST = 1;

    /**
     * Compares two {@link VariableElement} using the following order:
     * 1) strings
     * 2) integers
     * 3) booleans
     * 4) enums
     * 5) collections and maps
     * For same types or types not listed here, order alphabetically.
     */
    @Override
    public int compare(Variable variable1, Variable variable2) {

        Placement placementVar1 = variable1.getAnnotation(Placement.class);
        Placement placementVar2 = variable2.getAnnotation(Placement.class);

        if (!sameGroup(placementVar1, placementVar2)) {
            return 0;
        }

        if (placementVar1 != null && placementVar2 != null) {
            return new Integer(placementVar1.order()).compareTo(placementVar2.order());
        } else if (placementVar1 != null) {
            return new Integer(placementVar1.order()).compareTo(Placement.DEFAULT_ORDER);
        } else if (placementVar2 != null) {
            return new Integer(Placement.DEFAULT_ORDER).compareTo(placementVar2.order());
        }

        if (bothOfSameType(variable1, variable2)) {
            return compareByName(variable1, variable2);
        }

        if (variable1.asType().isCollection()) {
            return VARIABLE2_FIRST;
        }
        if (variable2.asType().isCollection()) {
            return VARIABLE1_FIRST;
        }

        if (variable1.asType().isBoolean()) {
            return VARIABLE2_FIRST;
        }
        if (variable2.asType().isBoolean()) {
            return VARIABLE1_FIRST;
        }

        if (variable1.asType().isEnum()) {
            return VARIABLE2_FIRST;
        }
        if (variable2.asType().isEnum()) {
            return VARIABLE1_FIRST;
        }

        return compareByName(variable1, variable2);
    }

    private boolean sameGroup(Placement placementVar1, Placement placementVar2) {
        String group1 = extractGroup(placementVar1);
        String group2 = extractGroup(placementVar2);
        return group1.equals(group2);
    }

    private String extractGroup(Placement placement) {
        if (placement != null && StringUtils.isNotBlank(placement.group())) {
            return placement.group();
        }
        return BaseStudioXmlBuilder.GENERAL_GROUP_NAME;
    }

    private boolean bothOfSameType(Variable variable1, Variable variable2) {
        return variable1.asType().isString() && variable2.asType().isString() ||
                variable1.asType().isInteger() && variable2.asType().isInteger() ||
                variable1.asType().isEnum() && variable2.asType().isEnum() ||
                variable1.asType().isBoolean() && variable2.asType().isBoolean() ||
                variable1.asType().isCollection() && variable2.asType().isCollection();
    }

    private int compareByName(Variable variable1, Variable variable2) {
        String name1 = extractName(variable1);
        String name2 = extractName(variable2);
        return name1.compareTo(name2);
    }

    private String extractName(Variable variable) {
        if (variable.getAnnotation(FriendlyName.class) != null) {
            return variable.getAnnotation(FriendlyName.class).value();
        } else {
            return variable.getName();
        }
    }
}