/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.lic.model;

import com.google.common.base.Optional;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.mule.devkit.internal.lic.SecurityUtils;
import org.mule.devkit.internal.lic.exception.InvalidKeyException;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.Key;
import java.security.PublicKey;
import java.util.Properties;

/**
 * @author MuleSoft, Inc
 */
public class LicenseProviderData {

    public static final String EMAIL_KEY = "contact.email";
    public static final String CONTACT_MESSAGE_KEY = "contact.message";

    private static final Logger logger = Logger.getLogger(LicenseProviderData.class);

    private final String name;
    private final PublicKey key;
    private final Properties metadata;
    private final String pubKeyFile;
    private final String bundleFile;
    private String infoFile;

    public LicenseProviderData(String vendorName, Key muleDecryptionKey) throws InvalidKeyException {
        if (StringUtils.isBlank(vendorName)) {
            throw new IllegalArgumentException("Vendor name cannot be blank");
        }

        name = vendorName;
        bundleFile = name.concat(".key");
        pubKeyFile = name.concat(".pub");
        infoFile = name.concat(".info");

        ZippedBundle keyBundle = unzipBundle(bundleFile);

        key = loadPublicKey(muleDecryptionKey, keyBundle);
        metadata = loadMetaData(muleDecryptionKey, keyBundle);
    }

    public PublicKey getKey() {
        return this.key;
    }

    public String getName() {
        return name;
    }

    public Optional<String> getProperty(String propertyName) {
        return Optional.fromNullable(metadata.getProperty(propertyName));
    }

    private Properties loadMetaData(Key muleDecryptionKey, ZippedBundle bundleFiles) throws InvalidKeyException {

        Optional<byte[]> info = bundleFiles.get(infoFile);
        if (!info.isPresent()) {
            throw new InvalidKeyException("Failed to decrypt " + pubKeyFile + ". No metadata file named  " + infoFile + "  found in " + bundleFile);
        }

        try {
            Properties metadata = new Properties();
            metadata.load(new ByteArrayInputStream(SecurityUtils.decrypt(info.get(), muleDecryptionKey)));
            return metadata;
        } catch (Exception e) {
            throw new InvalidKeyException("Failed to load " + infoFile + " found in " + bundleFile, e);
        }
    }

    private PublicKey loadPublicKey(Key muleDecryptionKey, ZippedBundle bundleFiles) throws InvalidKeyException {

        Optional<byte[]> pub = bundleFiles.get(pubKeyFile);
        if (!pub.isPresent()) {
            throw new InvalidKeyException("Failed to decrypt " + pubKeyFile + ". No public key name " + pubKeyFile + " found in bundle " + bundleFile);
        }

        try {
            return SecurityUtils.loadPublic(SecurityUtils.decrypt(pub.get(), muleDecryptionKey));
        } catch (Exception e) {
            throw new InvalidKeyException("Failed to decrypt " + pubKeyFile + " found in " + bundleFile, e);
        }
    }

    private ZippedBundle unzipBundle(String file) {
        InputStream bundleIS = Thread.currentThread().getContextClassLoader().getResourceAsStream(file);
        if (bundleIS == null) {
            bundleIS = Thread.currentThread().getContextClassLoader().getResourceAsStream("license/" + file);
            if (bundleIS == null) {
                throw new IllegalArgumentException("Vendor key bundle for name [" + file + "] not found as resource");
            }
        }

        return new ZippedBundle(bundleIS);
    }
}
