/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.internal.connection.management;

import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;
import org.mule.api.devkit.ProcessInterceptor;
import org.mule.api.processor.MessageProcessor;
import org.mule.api.routing.filter.Filter;
import org.mule.devkit.processor.ExpressionEvaluatorSupport;
import org.mule.security.oauth.callback.ProcessCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

public class ConnectionManagementProcessInterceptor<P, Adapter extends ConnectionManagementConnectorAdapter, Key extends ConnectionManagementConnectionKey, Strategy>
        extends ExpressionEvaluatorSupport implements ProcessInterceptor<P, Adapter> {

    private static Logger logger = LoggerFactory.getLogger(ConnectionManagementProcessInterceptor.class);
    private final ConnectionManagementConnectionManager<Key, Adapter, Strategy> connManagementBasicConnectionManager;
    private final MuleContext muleContext;
    private final ProcessInterceptor<P, Adapter> next;

    public ConnectionManagementProcessInterceptor(ProcessInterceptor<P, Adapter> next,
            ConnectionManagementConnectionManager<Key, Adapter, Strategy> connManagementBasicConnectionManager, MuleContext muleContext) {
        this.next = next;
        this.connManagementBasicConnectionManager = connManagementBasicConnectionManager;
        this.muleContext = muleContext;
    }

    @Override
    public P execute(ProcessCallback<P, Adapter> processCallback, Adapter object, MessageProcessor messageProcessor, MuleEvent event) throws Exception {
        Adapter adapter = null;
        Key key = null;

        key = connManagementBasicConnectionManager.getConnectionKey(messageProcessor, event);
        try {
            if (logger.isDebugEnabled()) {
                logger.debug(("Attempting to acquire connection using " + key.toString()));
            }
            adapter = connManagementBasicConnectionManager.acquireConnection(key);
            if (adapter == null) {
                throw new UnableToAcquireConnectionException();
            } else {
                if (logger.isDebugEnabled()) {
                    ConnectionManagementConnectionAdapter connection = connManagementBasicConnectionManager.getConnectionAdapter(adapter);
                    logger.debug((("Connection has been acquired with [id=" + connection.connectionId()) + "]"));
                }
            }
            return next.execute(processCallback, adapter, messageProcessor, event);
        } catch (Exception e) {
            if (processCallback.getManagedExceptions() != null) {
                for (Class exceptionClass : ((List<Class<? extends Exception>>) processCallback.getManagedExceptions())) {
                    if (exceptionClass.isInstance(e)) {
                        if (logger.isDebugEnabled()) {
                            logger.debug((((("An exception ( " + exceptionClass.getName()) + ") has been thrown. Destroying the connection with [id=")
                                    + connManagementBasicConnectionManager.getConnectionAdapter(adapter).connectionId() + "]")));
                        }
                        try {
                            if (adapter != null) {
                                connManagementBasicConnectionManager.destroyConnection(key, adapter);
                                adapter = null;
                            }
                        } catch (Exception innerException) {
                            logger.error(innerException.getMessage(), innerException);
                        }
                    }
                }
            }
            throw e;
        } finally {
            try {
                if (adapter != null) {
                    if (logger.isDebugEnabled()) {
                        logger.debug(
                                (("Releasing the connection back into the pool [id=" + connManagementBasicConnectionManager.getConnectionAdapter(adapter).connectionId()) + "]"));
                    }
                    connManagementBasicConnectionManager.releaseConnection(key, adapter);
                }
            } catch (Exception e) {
                throw new UnableToReleaseConnectionException(e);
            }
        }
    }

    public P execute(ProcessCallback<P, Adapter> processCallback, Adapter object, Filter filter, MuleMessage message) throws Exception {
        throw new UnsupportedOperationException();
    }

}
