/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.api.ws.definition;

import com.google.common.base.Optional;
import org.mule.devkit.internal.ws.common.ServiceDefinitionInitializationException;
import org.mule.util.IOUtils;
import org.mule.util.StringUtils;

import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DefaultServiceDefinition implements ServiceDefinition {

    private String id;
    private String displayName;
    private URL wsdlUrl;
    private Optional<String> service;
    private Optional<String> port;

    private List<String> excludedOperations = new ArrayList<String>();
    private Map<String, Object> properties = new HashMap<String, Object>();

    public DefaultServiceDefinition(String id, String displayName, String wsdlLocation) {
        this(id, displayName, wsdlLocation, null, null);
    }

    public DefaultServiceDefinition(String id, String displayName, String wsdlLocation, String service, String port) {
        URL wsdlUrl = resolveWsdlUrl(id, wsdlLocation);
        initialize(id, displayName, wsdlUrl, service, port);
    }

    private void initialize(String id, String displayName, URL wsdlUrl, String service, String port) {

        //check all params before assigning them
        verifyParameters(id, wsdlUrl);

        setId(id);
        setDisplayName(displayName);
        setWsdlUrl(wsdlUrl);
        setService(service);
        setPort(port);

    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public URL getWsdlUrl() {
        return wsdlUrl;
    }

    @Override
    public Optional<String> getService() {
        return service;
    }

    @Override
    public Optional<String> getPort() {
        return port;
    }

    @Override
    public String getDisplayName() {
        return displayName;
    }

    @Override
    public Map<String, Object> getProperties() {
        return properties;
    }

    @Override
    public List<String> getExcludedOperations() {
        return excludedOperations;
    }

    private URL resolveWsdlUrl(String id, String wsdlLocation) {
        if (StringUtils.isBlank(wsdlLocation)) {
            throw new ServiceDefinitionInitializationException(String.format("The given 'wsdlLocation' for [%s] ServiceDefinition must not be null (nor empty).", id));
        }
        return IOUtils.getResourceAsUrl(wsdlLocation, getClass());
    }

    private void verifyParameters(String id, URL wsdlUrl) {
        if (StringUtils.isBlank(id)){
            throw new ServiceDefinitionInitializationException(String.format("The given 'id' for a ServiceDefinition must not be null (nor empty). Received [%s]", id));
        }
        if (wsdlUrl == null) {
            throw new ServiceDefinitionInitializationException(
                    String.format("The given 'wsdlUrl' for [%s] ServiceDefinition must not be null (the provided wsdl file is probably unreachable).", id));
        }
    }

    public void setId(String id) {
        this.id = id;
    }

    public void setDisplayName(String displayName) {
        this.displayName = displayName;
    }

    public void setWsdlUrl(URL wsdlUrl) {
        this.wsdlUrl = wsdlUrl;
    }

    public void setService(String service) {
        this.service = Optional.fromNullable(service);
    }

    public void setPort(String port) {
        this.port = Optional.fromNullable(port);
    }

    public void setProperties(Map<String, Object> properties) { this.properties = properties; }

    public void addProperty(String key, Object value) { this.properties.put(key, value); }

    public void setExcludedOperations(List<String> excludedOperations) {
        this.excludedOperations = excludedOperations;
    }

    public void excludeOperation(String operationName) {
        if (StringUtils.isBlank(operationName)){
            throw new IllegalArgumentException("Operation name cannot be blank");
        }

        this.excludedOperations.add(operationName);
    }
}
