/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.ws.model;

import org.mule.devkit.api.ws.definition.ServiceDefinition;
import org.mule.devkit.api.ws.exception.InvalidWsdlFileException;
import org.mule.devkit.internal.ws.common.WsdlUtils;

import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.Service;

public class WsdlIntrospecterUtils {

    public static Service resolveService(ServiceDefinition serviceDefinition, Definition definition) {
        return serviceDefinition.getService().isPresent() ? WsdlUtils.getService(definition, serviceDefinition.getService().get()) : lookUpServiceOn(serviceDefinition, definition);
    }

    public static Port resolvePort(ServiceDefinition serviceDefinition, Service service, Definition definition) {
        return serviceDefinition.getPort().isPresent() ? WsdlUtils.getPort(service, serviceDefinition.getPort().get()) : lookUpPortOn(serviceDefinition, service);
    }

    public static String resolveServiceName(ServiceDefinition serviceDefinition, Definition definition) {
        return lookUpServiceNameOn(serviceDefinition, definition);
    }

    public static String resolvePortName(ServiceDefinition serviceDefinition, Service service) {
        return lookUpPortNameOn(serviceDefinition, service);
    }

    private static String lookUpServiceNameOn(ServiceDefinition serviceDefinition, Definition definition) {
        String[] serviceNames = WsdlUtils.getServiceNames(definition);

        if (serviceNames.length > 1) {
            String errorMessage = String
                    .format("WSDL file [%s] has [%d] services in it. When providing more than 1 service, the connector should specify which one is the intended to be used.",
                            serviceDefinition.getId(), serviceNames.length);
            throw new InvalidWsdlFileException(errorMessage);
        } else if (serviceNames.length == 0) {
            String errorMessage = String.format("WSDL file [%s] does not have any services at all. Check if the WSDL file was properly generated.", serviceDefinition.getId());
            throw new InvalidWsdlFileException(errorMessage);
        }

        return serviceNames[0];
    }

    private static Service lookUpServiceOn(ServiceDefinition serviceDefinition, Definition definition) {
        return WsdlUtils.getService(definition, lookUpServiceNameOn(serviceDefinition, definition));
    }

    private static String lookUpPortNameOn(ServiceDefinition serviceDefinition, Service service) {
        String[] portNames = WsdlUtils.getPortNames(service);

        if (portNames.length > 1) {
            String errorMessage = String
                    .format("WSDL file [%s] has [%d] ports in it. When providing more than 1 port, the connector should specify which one is the intended to be used.",
                            serviceDefinition.getId(), portNames.length);
            throw new InvalidWsdlFileException(errorMessage);
        } else if (portNames.length == 0) {
            String errorMessage = String.format("WSDL file [%s] does not have any port at all. Check if the WSDL file was properly generated.", serviceDefinition.getId());
            throw new InvalidWsdlFileException(errorMessage);
        }

        return portNames[0];
    }

    private static Port lookUpPortOn(ServiceDefinition serviceDefinition, Service service) {
        return WsdlUtils.getPort(service, lookUpPortNameOn(serviceDefinition, service));
    }
}
