/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.api.metadata;

import org.mule.devkit.api.metadata.exception.InvalidKeyBuildStateException;
import org.mule.devkit.api.metadata.exception.InvalidKeyException;
import org.mule.devkit.internal.metadata.DefaultComposedMetaDataKey;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * @author Mulesoft, Inc
 */
public class ComposedMetaDataKeyBuilder {

    protected List<ComposedMetaDataKey> keys = null;
    protected String customSeparator = "";

    protected ComposedMetaDataKeyBuilder() {
        this.keys = new LinkedList<ComposedMetaDataKey>();
    }

    public static ComposedMetaDataKeyBuilder getInstance() {
        return new ComposedMetaDataKeyBuilder();
    }

    public CombinationBuilder newKeyCombination() {
        return new CombinationBuilder(this, customSeparator);
    }

    public ComposedMetaDataKeyBuilder endKeyCombination(List<ComposedMetaDataKey> keys) {
        this.keys.addAll(keys);
        return this;
    }

    public List<ComposedMetaDataKey> build() {
        List<ComposedMetaDataKey> result = this.keys;
        resetBuild();
        return result;
    }

    private void resetBuild() {
        this.keys = new LinkedList<ComposedMetaDataKey>();
    }

    public static class CombinationBuilder {

        protected final ComposedMetaDataKeyBuilder parentBuilder;
        private final String keySeparator;
        protected List<MetaDataKeyLevel> levels = new LinkedList<MetaDataKeyLevel>();
        private boolean combinationEnded = false;

        protected CombinationBuilder(ComposedMetaDataKeyBuilder parentBuilder, String keySeparator) {
            this.parentBuilder = parentBuilder;
            this.keySeparator = keySeparator;
        }

        public LevelBuilder newLevel() {
            if (combinationEnded) {
                throw new InvalidKeyBuildStateException("Cannot add a new level to an already ended keyCombination");
            }
            return new LevelBuilder(this);
        }

        public CombinationBuilder endLevel(MetaDataKeyLevel currentLevel) {
            this.levels.add(currentLevel);
            return this;
        }

        public ComposedMetaDataKeyBuilder endKeyCombination() {
            if (levels.isEmpty()) {
                throw new InvalidKeyException("Key can not be empty, at least one MetaDataKeyLevel is required");
            }

            combinationEnded = true;
            return parentBuilder.endKeyCombination(buildKeyCombinationForLevels());
        }

        private List<ComposedMetaDataKey> buildKeyCombinationForLevels() {
            Iterator<MetaDataKeyLevel> levelsIt = levels.iterator();
            MetaDataKeyLevel level = levelsIt.next();

            List<ComposedMetaDataKey> composedKeys = initializeKeys(level);
            while (levelsIt.hasNext()) {
                level = levelsIt.next();
                composedKeys = expandKeysWithLevel(level, composedKeys);
            }

            levels = new LinkedList<MetaDataKeyLevel>();
            return composedKeys;
        }

        private List<ComposedMetaDataKey> initializeKeys(MetaDataKeyLevel level) {

            List<ComposedMetaDataKey> composedKeys = new LinkedList<ComposedMetaDataKey>();
            for (Map.Entry<String, String> entry : level.getIds()) {
                ComposedMetaDataKey key = keySeparator.isEmpty() ? newKey() : newKey(keySeparator);
                key.addLevel(entry.getKey(), entry.getValue());
                composedKeys.add(key);
            }
            return composedKeys;
        }

        private List<ComposedMetaDataKey> expandKeysWithLevel(MetaDataKeyLevel level, List<ComposedMetaDataKey> composedKeys) {
            if (level.getIds().isEmpty()) {
                return composedKeys;
            }

            List<ComposedMetaDataKey> expandedKeys = new LinkedList<ComposedMetaDataKey>();
            for (ComposedMetaDataKey key : composedKeys) {
                for (Map.Entry<String, String> entry : level.getIds()) {
                    ComposedMetaDataKey expandedKey = new DefaultComposedMetaDataKey(key);
                    expandedKey.addLevel(entry.getKey(), entry.getValue());
                    expandedKeys.add(expandedKey);
                }
            }
            return expandedKeys;
        }

        private ComposedMetaDataKey newKey() {
            return new DefaultComposedMetaDataKey();
        }

        private ComposedMetaDataKey newKey(String separator) {
            return new DefaultComposedMetaDataKey(separator);
        }

    }

    public static class LevelBuilder {

        protected final CombinationBuilder parentBuilder;
        protected MetaDataKeyLevel currentLevel = new DefaultMetaDataKeyLevel();

        private boolean levelEnded = false;

        protected LevelBuilder(CombinationBuilder parentBuilder) {
            this.parentBuilder = parentBuilder;
        }

        public LevelBuilder addId(String id, String label) {
            if (levelEnded) {
                throw new InvalidKeyBuildStateException("Cannot add a new id to an already ended level");
            }
            this.currentLevel.addId(id, label);
            return this;
        }

        public LevelBuilder addIds(MetaDataKeyLevel level) {
            if (level != null && !level.getIds().isEmpty()) {
                if (this.currentLevel != null) {
                    for (Map.Entry<String, String> levelElement : level.getIds()) {
                        this.currentLevel.addId(levelElement.getKey(), levelElement.getValue());
                    }
                } else {
                    this.currentLevel = level;
                }
                return this;
            } else {
                throw new InvalidKeyException("Levels for key cannot be null nor empty");
            }
        }

        public CombinationBuilder endLevel() {
            if (currentLevel.getIds().isEmpty()) {
                throw new InvalidKeyBuildStateException("Cannot create an empty Level");
            }

            levelEnded = true;
            return this.parentBuilder.endLevel(currentLevel);
        }
    }
}
