/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
/**
 * (c) 2003-2014 MuleSoft, Inc. The software in this package is published under the terms of the CPAL v1.0 license,
 * a copy of which has been included with this distribution in the LICENSE.md file.
 */

package org.mule.devkit.internal.ws.metadata;

import com.google.common.base.Optional;
import org.apache.commons.lang.StringUtils;
import org.mule.common.DefaultResult;
import org.mule.common.Result;
import org.mule.common.metadata.*;
import org.mule.common.metadata.builder.DefaultMetaDataBuilder;
import org.mule.common.metadata.builder.XmlMetaDataBuilder;
import org.mule.common.metadata.datatype.DataType;
import org.mule.common.metadata.key.property.TypeDescribingProperty;
import org.mule.devkit.api.metadata.ComposedMetaDataKeyBuilder;
import org.mule.devkit.api.metadata.DefaultMetaDataKeyLevel;
import org.mule.devkit.api.metadata.MetaDataKeyLevel;
import org.mule.devkit.api.ws.definition.ServiceDefinition;
import org.mule.devkit.internal.metadata.InternalComposedMetaDataKeyBuilder;
import org.mule.devkit.internal.metadata.fixes.STUDIO7157;
import org.mule.devkit.internal.ws.common.EnhancedServiceDefinition;
import org.mule.devkit.internal.ws.common.WsdlAdapter;
import org.mule.devkit.internal.ws.common.WsdlSplitKey;
import org.mule.devkit.internal.ws.common.WsdlUtils;
import org.mule.devkit.internal.ws.metadata.utils.InvokeWsdlResolver;
import org.mule.devkit.internal.ws.model.InvokeSoapMessageProcessor;
import org.mule.devkit.internal.ws.model.WsdlIntrospecterUtils;

import javax.wsdl.*;
import javax.wsdl.extensions.soap.SOAPHeader;
import javax.xml.namespace.QName;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class WsdlMetaDataDescriptor {

    public static final String SOAP_PREFIX = "soap.";

    /**
     * Used to follow the contract of {@link org.mule.common.metadata.ConnectorMetaDataEnabled#getMetaDataKeys()} when
     * retrieving keys to Studio for the set of WSDL files
     * @param wsdlAdapter
     * @return
     */
    public Result<List<MetaDataKey>> getMetaDataKeys(WsdlAdapter wsdlAdapter) {
        try {
            List<MetaDataKey> gatheredMetaDataKeys = new ArrayList<MetaDataKey>();
            for (ServiceDefinition serviceDefinition : wsdlAdapter.serviceDefinitions()) {
                gatheredMetaDataKeys.addAll(getKeysFor(wsdlAdapter, serviceDefinition));
            }
            return new DefaultResult<List<MetaDataKey>>(gatheredMetaDataKeys, (Result.Status.SUCCESS));
        } catch (Exception e) {
            return new DefaultResult<List<MetaDataKey>>(null, (Result.Status.FAILURE),
                    "There was an error retrieving the metadata keys from service provider after acquiring connection, for more detailed information please read the provided stacktrace",
                    MetaDataFailureType.ERROR_METADATA_KEYS_RETRIEVER, e);
        }
    }

    /**
     * Used to follow the contract of either:
     * <ol>
     *     <li>{@link org.mule.common.metadata.ConnectorMetaDataEnabled#getMetaData(org.mule.common.metadata.MetaDataKey)} when describing types generically (e.g: @Query scenarios)</li>
     *     <li>{@link org.mule.common.metadata.OperationMetaDataEnabled#getInputMetaData()} when describing the input metadata of the invoke operation (see {@link InvokeSoapMessageProcessor})</li>
     *     <li>{@link org.mule.common.metadata.OperationMetaDataEnabled#getOutputMetaData(org.mule.common.metadata.MetaData)} when describing the output of the invoke operation (see {@link InvokeSoapMessageProcessor})</li>
     * </ol>
     * @param metaDataKey
     * @param wsdlAdapter
     * @return
     */
    public Result<MetaData> getMetaData(MetaDataKey metaDataKey, WsdlAdapter wsdlAdapter) {
        try {
            MetaData metaData;
            String name = metaDataKey.getId();
            boolean generatedStructure = false;
            EnhancedServiceDefinition enhancedServiceDefinition = resolveEnhancedServiceDefinition(metaDataKey, wsdlAdapter);

            if (isInputMetaData(metaDataKey)) {
                metaData = loadInputMetadata(enhancedServiceDefinition);
            } else {
                name += " Result";
                generatedStructure = true;
                metaData = loadOutputMetadata(enhancedServiceDefinition);
            }
            metaData.getPayload().addProperty(STUDIO7157.getStructureIdentifierMetaDataModelProperty(new DefaultMetaDataKey(name, null), false, generatedStructure));
            return new DefaultResult<MetaData>(metaData);
        } catch (Exception e) {
            return new DefaultResult<MetaData>(null, (Result.Status.FAILURE), getMetaDataException(metaDataKey), MetaDataFailureType.ERROR_METADATA_RETRIEVER, e);
        }
    }

    private boolean isInputMetaData(MetaDataKey metaDataKey) {
        return TypeDescribingProperty.TypeScope.INPUT.equals(metaDataKey.getProperty(TypeDescribingProperty.class).getTypeScope());
    }

    private String getMetaDataException(MetaDataKey key) {
        if (!StringUtils.isBlank(key.getId())) {
            return "There was an error retrieving metadata from key: " + (key.getId()
                    + " after acquiring the connection, for more detailed information please read the provided stacktrace");
        } else {
            return "There was an error retrieving metadata after acquiring the connection, MetaDataKey is null or its id is null, for more detailed information please read the provided stacktrace";
        }
    }

    private List<MetaDataKey> getKeysFor(WsdlAdapter wsdlAdapter, ServiceDefinition serviceDefinition) throws Exception {
        String keySeparator = wsdlAdapter.wsdlSeparator();
        Definition definition = WsdlUtils.parseWSDL(serviceDefinition.getWsdlUrl().toString());

        Service service = WsdlIntrospecterUtils.resolveService(serviceDefinition, definition);
        Port port = WsdlIntrospecterUtils.resolvePort(serviceDefinition, service, definition);

        MetaDataKeyLevel operations = new DefaultMetaDataKeyLevel();
        List<String> excludedOperations = serviceDefinition.getExcludedOperations();
        for (String operation: WsdlUtils.getOperationNames(port)){
            if (!excludedOperations.contains(operation)){
                operations.addId(operation, operation);
            }
        }

        ComposedMetaDataKeyBuilder.CombinationBuilder builder = InternalComposedMetaDataKeyBuilder.getInstance().withSeparator(keySeparator).newKeyCombination();

        if (!wsdlAdapter.singleServiceDefinitionId().isPresent()) {
            builder.newLevel().addId(serviceDefinition.getId(), serviceDefinition.getDisplayName()).endLevel();
        }

        return InternalComposedMetaDataKeyBuilder.toSimpleKey(builder.newLevel().addIds(operations).endLevel().endKeyCombination().build(), keySeparator);
    }

    private EnhancedServiceDefinition resolveEnhancedServiceDefinition(MetaDataKey key, WsdlAdapter wsdlAdapter) throws Exception {
        WsdlSplitKey splitKey = new WsdlSplitKey(key.getId(), wsdlAdapter);
        return wsdlAdapter.wsResolver().enhancedServiceDefinition(splitKey.id(), wsdlAdapter, splitKey.operation());
    }

    private MetaData loadInputMetadata(EnhancedServiceDefinition enhancedServiceDefinition) throws Exception {
        return loadMetadata(enhancedServiceDefinition, InvokeWsdlResolver.OperationMode.INPUT);
    }

    private MetaData loadOutputMetadata(EnhancedServiceDefinition enhancedServiceDefinition) throws Exception {
        return loadMetadata(enhancedServiceDefinition, InvokeWsdlResolver.OperationMode.OUTPUT);
    }

    private MetaData loadMetadata(EnhancedServiceDefinition enhancedServiceDefinition, InvokeWsdlResolver.OperationMode operationMode) throws Exception {
        MetaDataPropertyScope scope = operationMode == InvokeWsdlResolver.OperationMode.INPUT ? MetaDataPropertyScope.OUTBOUND : MetaDataPropertyScope.INBOUND;
        InvokeWsdlResolver invokeWsdlResolver = new InvokeWsdlResolver(operationMode, enhancedServiceDefinition.getWsdlUrl().toString(), enhancedServiceDefinition.getService(),
                enhancedServiceDefinition.getPort(), enhancedServiceDefinition.getOperation());
        MetaData outputMetadata = createMetaData(invokeWsdlResolver.getSchemas(), invokeWsdlResolver.getMessagePart(), enhancedServiceDefinition.getWsdlUrl());
        if (invokeWsdlResolver.getMessagePart().isPresent()){
            addProperties(outputMetadata, invokeWsdlResolver, scope);
        }
        return outputMetadata;
    }

    private MetaData createMetaData(List<String> schemas, Optional<Part> partOptional, URL url) {
        if (partOptional.isPresent()) {
            Part part = partOptional.get();
            if (part.getElementName() != null) {
                final QName elementName = part.getElementName();
                final XmlMetaDataBuilder<?> createXmlObject = new DefaultMetaDataBuilder().createXmlObject(elementName);
                for (String schema : schemas) {
                    createXmlObject.addSchemaStringList(schema);
                }
                createXmlObject.setEncoding(charset());
                createXmlObject.setSourceUri(url);
                return new DefaultMetaData(createXmlObject.build());
            } else if (part.getTypeName() != null) {
                DataType dataType = getDataTypeFromTypeName(part);
                DefaultSimpleMetaDataModel defaultSimpleMetaDataModel = new DefaultSimpleMetaDataModel(dataType);
                return new DefaultMetaData(defaultSimpleMetaDataModel);
            }
        }
        return new DefaultMetaData(new DefaultUnknownMetaDataModel());
    }

    private void addProperties(MetaData inputMetadata, InvokeWsdlResolver invokeWsdlResolver, MetaDataPropertyScope metaDataPropertyScope) {
        final List<SOAPHeader> outputHeaders = invokeWsdlResolver.getOperationHeaders();
        for (SOAPHeader soapHeader : outputHeaders) {
            final Message message = invokeWsdlResolver.getDefinition().getMessage(soapHeader.getMessage());
            if (message != null) {
                final Part part = message.getPart(soapHeader.getPart());
                inputMetadata.addProperty(metaDataPropertyScope, SOAP_PREFIX + soapHeader.getPart(),
                        new DefaultXmlMetaDataModel(invokeWsdlResolver.getSchemas(), part.getElementName(), charset()));
            }
        }
    }

    private Charset charset() {
        return Charset.defaultCharset();
    }

    private DataType getDataTypeFromTypeName(Part part) {
        String localPart = part.getTypeName().getLocalPart();

        HashMap<String, DataType> types = new HashMap<String, DataType>();
        types.put("string", DataType.STRING);
        types.put("boolean", DataType.BOOLEAN);
        types.put("date", DataType.DATE);
        types.put("decimal", DataType.DECIMAL);
        types.put("byte", DataType.BYTE);
        types.put("unsignedByte", DataType.BYTE);
        types.put("dateTime", DataType.DATE_TIME);
        types.put("int", DataType.INTEGER);
        types.put("integer", DataType.INTEGER);
        types.put("unsignedInt", DataType.INTEGER);
        types.put("short", DataType.INTEGER);
        types.put("unsignedShort", DataType.INTEGER);
        types.put("long", DataType.LONG);
        types.put("unsignedLong", DataType.LONG);
        types.put("double", DataType.DOUBLE);

        DataType dataType = types.get(localPart);
        return dataType != null ? dataType : DataType.STRING;
    }
}
