/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.ws.model;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.ImmutableMap;
import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.devkit.api.lifecycle.LifeCycleManager;
import org.mule.devkit.api.ws.definition.ServiceDefinition;
import org.mule.devkit.api.ws.exception.WrongParametrizationWsdlException;
import org.mule.devkit.internal.ws.common.EnhancedServiceDefinition;
import org.mule.devkit.internal.ws.common.WSResolver;
import org.mule.devkit.internal.ws.common.WsdlAdapter;
import org.mule.devkit.internal.ws.model.cache.*;
import org.mule.devkit.internal.ws.model.cache.exception.WSLifecycleException;
import org.mule.module.ws.consumer.WSConsumer;
import org.mule.module.ws.consumer.WSConsumerConfig;

import java.util.concurrent.ExecutionException;

public class DefaultWSResolver implements WSResolver {

    private ImmutableMap<String, ServiceDefinition> definitionsCache;
    private Cache<WSConsumerConfigKey, WSConsumerConfig> wsConfigCache;
    private Cache<WSConsumerKey, WSConsumer> wsConsumerCache;
    private Cache<EnhancedServiceDefinitionKey, EnhancedServiceDefinition> enhancedServiceDefinitionCache;

    public DefaultWSResolver(WsdlAdapter wsdlAdapter) throws Exception {
        initialize(wsdlAdapter);
    }

    @Override
    public ServiceDefinition serviceDefinition(String id) {
        if (!definitionsCache.containsKey(id)) {
            throw new WrongParametrizationWsdlException("The connector is being invoked with an id ([" + id + "]) that cannot be resolved.");
        }
        return definitionsCache.get(id);
    }

    @Override
    public EnhancedServiceDefinition enhancedServiceDefinition(String id, WsdlAdapter wsdlAdapter, String operation) throws Exception {
        ServiceDefinition serviceDefinition = serviceDefinition(id);
        EnhancedServiceDefinitionKey enhancedServiceDefinitionKey = new EnhancedServiceDefinitionKey(id, operation);
        return enhancedServiceDefinitionCache.get(enhancedServiceDefinitionKey, new EnhancedServiceDefinitionCallable(wsdlAdapter, serviceDefinition, operation));
    }

    @Override
    public WSConsumer wsConsumer(EnhancedServiceDefinition enhancedServiceDefinition, MuleContext muleContext) throws ExecutionException {
        WSConsumerConfigKey wsConsumerConfigKey = new WSConsumerConfigKey(enhancedServiceDefinition.getId(), enhancedServiceDefinition.getServiceAddress(),
                enhancedServiceDefinition.getService(), enhancedServiceDefinition.getPort());
        WSConsumerConfig wsConsumerConfig = getWsConsumerConfig(wsConsumerConfigKey, enhancedServiceDefinition, muleContext);
        WSConsumerKey wsConsumerKey = new WSConsumerKey(wsConsumerConfigKey, enhancedServiceDefinition.getOperation());
        return wsConsumerCache.get(wsConsumerKey, new WSConsumerCallable(muleContext, wsConsumerConfig, enhancedServiceDefinition.getOperation()));
    }

    @Override
    public void dispose() {
        wsConfigCache.invalidateAll();
        wsConsumerCache.invalidateAll();
        definitionsCache = null;
        enhancedServiceDefinitionCache.invalidateAll();
    }

    @Override
    public ImmutableMap<String, ServiceDefinition> serviceDefinitions() {
        return definitionsCache;
    }

    private void initialize(WsdlAdapter wsdlAdapter) throws Exception {
        wsConfigCache = CacheBuilder.newBuilder().removalListener(getRemovalListener()).build();
        wsConsumerCache = CacheBuilder.newBuilder().removalListener(getRemovalListener()).build();
        enhancedServiceDefinitionCache = CacheBuilder.newBuilder().removalListener(getRemovalListener()).build();

        ImmutableMap.Builder<String, ServiceDefinition> builder = ImmutableMap.builder();
        for (ServiceDefinition serviceDefinition : wsdlAdapter.serviceDefinitions()) {
            builder.put(serviceDefinition.getId(), serviceDefinition);
        }
        definitionsCache = builder.build();
    }

    private WSConsumerConfig getWsConsumerConfig(WSConsumerConfigKey wsConsumerConfigKey, EnhancedServiceDefinition enhancedServiceDefinition, MuleContext muleContext)
            throws ExecutionException {
        return wsConfigCache.get(wsConsumerConfigKey, new WSConsumerConfigCallable(muleContext, enhancedServiceDefinition));
    }

    private <T, K> RemovalListener<T, K> getRemovalListener() {
        return new RemovalListener<T, K>() {

            public void onRemoval(RemovalNotification<T, K> removal) {
                executeLifecycleEnding(removal.getKey(), removal.getValue());
            }
        };
    }

    private void executeLifecycleEnding(Object key, Object value) {
        try {
            LifeCycleManager.executeStopAndDispose(value);
        } catch (MuleException e) {
            throw new WSLifecycleException("There was an issue while trying to close the config referenced by " + key.toString(), e);
        }
    }
}
