/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.streaming.processor;

import org.mule.api.MuleException;
import org.mule.streaming.ProviderAwarePagingDelegate;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

/**
 * @author MuleSoft, Inc.
 *         Used when an @Paged @Processor has also been annotated with @OnException, we delegate each method of the contract
 *         {@link ProviderAwarePagingDelegate} to the inner element (custom code of the connector's developer), but if an error
 *         arises when fetching the page, then we propagate it to the concrete @Handler object by using reflection.
 */
public class ExceptionHandlerProviderAwarePagingDelegate extends ProviderAwarePagingDelegate {

    private ProviderAwarePagingDelegate innerPagination;
    private Class<?> handlerClass;
    private String handlerMethodName;

    public ExceptionHandlerProviderAwarePagingDelegate(ProviderAwarePagingDelegate innerPagination, Class<?> handlerClass, String handlerMethodName) {
        this.innerPagination = innerPagination;
        this.handlerClass = handlerClass;
        this.handlerMethodName = handlerMethodName;
    }

    @Override
    public List getPage(Object provider) throws Exception {
        try {
            return innerPagination.getPage(provider);
        } catch (Exception e) {
            handleException(e);
            throw e;
        }
    }

    @Override
    public int getTotalResults(Object provider) throws Exception {
        return innerPagination.getTotalResults(provider);
    }

    @Override
    public void close() throws MuleException {
        innerPagination.close();
    }

    /**
     * By reflection, we will invoke the handler if the getPage have had an exception to the existing @Handler object
     *
     * @param e exception to analyze
     * @throws Exception if: (a) the reflection mechanism failed because of bad parametrization, then the {@code e} will
     *                   be propagated, (b), the custom throwable of the @Handle method within the @Handler object.
     */
    private void handleException(Exception e) throws Exception {
        try {
            Object handler = handlerClass.newInstance();
            Method handlerMethod = handlerClass.getMethod(handlerMethodName, Exception.class);
            handlerMethod.invoke(handler, e);
        } catch (InstantiationException ie) {
            throw e;
        } catch (IllegalAccessException iae) {
            throw e;
        } catch (NoSuchMethodException nsme) {
            throw e;
        } catch (InvocationTargetException ite) {
            //in this scenario, we need to propagate the same exception that was thrown by the developer if present
            if (ite.getCause() != null && ite.getCause() instanceof Exception) {
                throw (Exception) ite.getCause();
            }
            //if not, then returning the same exception that lead us here: {@code e}
            throw e;
        }
    }
}
