/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.lic.model;

import com.google.common.base.Optional;
import org.apache.commons.lang.StringUtils;
import org.mule.devkit.internal.lic.SecurityUtils;
import org.mule.devkit.internal.lic.exception.InvalidKeyException;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.Key;
import java.security.PublicKey;
import java.util.Properties;

/**
 * @author MuleSoft, Inc
 */
public class LicenseProviderData {

    public static final String CONTACT_EMAIL_KEY = "contact.email";
    public static final String CONTACT_MESSAGE_KEY = "contact.message";
    public static final String CONNECTOR_NAME_KEY = "connector.name";

    private final String name;
    private final PublicKey key;
    private final Properties metadata;
    private final String pubKeyFile;
    private final String bundleFile;
    private String infoFile;

    public LicenseProviderData(String vendorName, String connectorName, Key muleDecryptionKey) throws InvalidKeyException {
        if (StringUtils.isBlank(vendorName)) {
            throw new IllegalArgumentException("Vendor name cannot be blank");
        }

        name = vendorName + "-" + connectorName;
        infoFile = name.concat(".info");
        bundleFile = name.concat(".key");
        pubKeyFile = name.concat(".pub");

        ZippedBundle keyBundle = unzipBundle(bundleFile);

        key = loadPublicKey(muleDecryptionKey, keyBundle);
        metadata = loadMetaData(muleDecryptionKey, keyBundle);

        if (!getProperty(CONNECTOR_NAME_KEY).isPresent()){
            throw new InvalidKeyException("Provided key is incomplete. No connector name information found");
        } else if (!StringUtils.equals(getProperty(CONNECTOR_NAME_KEY).get(), connectorName)){
            throw new InvalidKeyException("Provided key "+ name +" is not valid for this connector. It was created for " + getProperty(CONNECTOR_NAME_KEY).get()
            + " but expected to be " + connectorName);
        }
    }

    public PublicKey getKey() {
        return this.key;
    }

    public String getName() {
        return name;
    }

    public String getEmail() {return getProperty(CONTACT_EMAIL_KEY).get();}

    public String getContactMessage() {return getProperty(CONTACT_MESSAGE_KEY).get();}

    public Optional<String> getProperty(String propertyName) {
        return Optional.fromNullable(metadata.getProperty(propertyName));
    }

    private Properties loadMetaData(Key muleDecryptionKey, ZippedBundle bundleFiles) throws InvalidKeyException {

        Optional<byte[]> info = bundleFiles.get(infoFile);
        if (!info.isPresent()) {
            throw new InvalidKeyException("Failed to decrypt " + pubKeyFile + ". No metadata file named  " + infoFile + "  found in " + bundleFile);
        }

        try {
            Properties metadata = new Properties();
            metadata.load(new ByteArrayInputStream(SecurityUtils.decrypt(info.get(), muleDecryptionKey)));
            return metadata;
        } catch (Exception e) {
            throw new InvalidKeyException("Failed to load " + infoFile + " found in " + bundleFile, e);
        }
    }

    private PublicKey loadPublicKey(Key muleDecryptionKey, ZippedBundle bundleFiles) throws InvalidKeyException {

        Optional<byte[]> pub = bundleFiles.get(pubKeyFile);
        if (!pub.isPresent()) {
            throw new InvalidKeyException("Failed to decrypt " + pubKeyFile + ". No public key name " + pubKeyFile + " found in bundle " + bundleFile);
        }

        try {
            return SecurityUtils.loadPublic(SecurityUtils.decrypt(pub.get(), muleDecryptionKey));
        } catch (Exception e) {
            e.printStackTrace();
            throw new InvalidKeyException("Failed to decrypt " + pubKeyFile + " found in " + bundleFile, e);
        }
    }

    private ZippedBundle unzipBundle(String file) {
        InputStream bundleIS = Thread.currentThread().getContextClassLoader().getResourceAsStream(file);
        if (bundleIS == null) {
            bundleIS = Thread.currentThread().getContextClassLoader().getResourceAsStream("license/" + file);
            if (bundleIS == null) {
                throw new IllegalArgumentException("Vendor key bundle for name [" + file + "] not found as resource");
            }
        }

        return new ZippedBundle(bundleIS);
    }
}
