/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.lic.validator;

import org.apache.log4j.Logger;
import org.mule.devkit.internal.lic.SecurityUtils;
import org.mule.devkit.internal.lic.exception.InvalidKeyException;
import org.mule.devkit.internal.lic.exception.InvalidLicenseException;
import org.mule.devkit.internal.lic.model.CustomLicense;
import org.mule.devkit.internal.lic.model.Entitlement;
import org.mule.devkit.internal.lic.model.LicenseProviderData;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.PublicKey;
import java.util.Calendar;
import java.util.Date;

/**
 * @author Mulesoft, Inc
 */
public class DevkitLicenseValidator implements LicenseValidator {

    private static final Logger logger = Logger.getLogger(DevkitLicenseValidator.class);

    private static final String LICENSE_MANAGEMENT_FACTORY = "com.mulesource.licm.LicenseManagementFactory";
    private static final String ENTERPRISE_LICENSE_KEY = "com.mulesource.licm.EnterpriseLicenseKey";
    private static final String EXPIRED_LICENSE_MSG = "Your license has expired";
    private static final String FEATURE_VALIDATOR = "com.mulesource.licm.feature.FeatureValidator";
    private static final String LICENSE_MANAGER = "com.mulesource.licm.LicenseManager";
    private static final String FEATURE = "com.mulesource.licm.feature.Feature";
    private static final String MULE_EE = "mule-ee";
    private static final String MULE_PUB_KEY = "mule.pub";

    private PublicKey mulePublicKey;
    private String DEFAULT_EXCEPTION_MSG;
    private String EVALUATION_LICENSE_MSG;
    private String MISSING_ENTITLEMENT_MSG;

    private String moduleName;
    private Object license; // com.mulesource.licm.EnterpriseLicenseKey
    private Constructor<?> featureConstructor;
    private Constructor<?> featureValidatorConstructor;
    private Method hasFeature;
    private Method getFeatures;
    private Method licenseIsEvaluation;
    private Method licenseGetExpirationDate;
    private Method validateFeature;
    private Method setFeature;

    public DevkitLicenseValidator(String moduleName) {
        DEFAULT_EXCEPTION_MSG = String.format("The Module %s requires an Enterprise License. Switch to a Mule-EE runtime to enable it. ", moduleName);
        EVALUATION_LICENSE_MSG = String.format("The Module %s does not allow Evaluation Licenses", moduleName);
        MISSING_ENTITLEMENT_MSG = String.format("The Module %s requires a license with entitlement for ", moduleName);
        this.moduleName = moduleName;

        try {
            Class<?> licManagerFactory = Class.forName(LICENSE_MANAGEMENT_FACTORY);
            Object licFactory = invoke(licManagerFactory.getMethod("getInstance"), null);
            Object licenseManager = invoke(licManagerFactory.getMethod("createLicenseManager", String.class), licFactory, MULE_EE);

            // EnterpriseLicenseKey license = LicenseManagementFactory.getInstance().createLicenseManager(MULE_EE).validate(MULE_EE)
            license = invoke(Class.forName(LICENSE_MANAGER).getMethod("validate", String.class), licenseManager, MULE_EE);
            initializeReflectiveMethods();

        } catch (Exception e) {
            throw new InvalidLicenseException(DEFAULT_EXCEPTION_MSG.concat(e.getMessage()));
        }
    }

    private void initializeReflectiveMethods() throws NoSuchMethodException, ClassNotFoundException {
        Class<?> enterpriseLicenseKeyClass = Class.forName(ENTERPRISE_LICENSE_KEY);
        getFeatures = enterpriseLicenseKeyClass.getMethod("getFeatures");
        Class<?> featureClass = Class.forName(FEATURE);
        hasFeature = Class.forName("com.mulesource.licm.feature.FeatureSet").getMethod("hasFeature", featureClass);
        featureConstructor = featureClass.getConstructor(String.class, String.class);
        featureValidatorConstructor = Class.forName(FEATURE_VALIDATOR).getConstructor(featureClass);
        licenseGetExpirationDate = enterpriseLicenseKeyClass.getMethod("getExpirationDate");
        licenseIsEvaluation = enterpriseLicenseKeyClass.getMethod("isEvaluation");
        validateFeature = Class.forName(FEATURE_VALIDATOR).getMethod("validate", enterpriseLicenseKeyClass);
        setFeature = enterpriseLicenseKeyClass.getMethod("setFeature", featureClass);
    }

    @Override
    public void checkEnterpriseLicense(boolean allowEvaluation) {

        logger.debug("Checking EE license. Allows Evaluation ["+allowEvaluation+"]");

        Calendar expirationDate = Calendar.getInstance();
        Object expirationTime = invoke(licenseGetExpirationDate, license);
        if (expirationTime != null) {
            expirationDate.setTime((Date) expirationTime);
            if (expirationDate.after(new Date())) {
                throw new InvalidLicenseException(EXPIRED_LICENSE_MSG);
            }
        }

        Boolean isEvaluation = (Boolean) invoke(licenseIsEvaluation, license);
        if (!allowEvaluation && isEvaluation) {
            throw new InvalidLicenseException(EVALUATION_LICENSE_MSG);
        }
    }

    @Override
    public void checkEntitlement(Entitlement requiredEntitlement) {
        logger.debug("Entitlement is third party " + requiredEntitlement.isThirdParty());
        if (requiredEntitlement.isThirdParty()) {
            addCustomEntitlement(requiredEntitlement);
        }
        logger.debug("Verify License for entitlement " + requiredEntitlement.id());
        verifyLicenseEntitlements(requiredEntitlement);
    }

    private void verifyLicenseEntitlements(Entitlement requiredEntitlement) {
        // feature = new Feature(id, description)
        Object feature = getInstance(featureConstructor, requiredEntitlement.id(), requiredEntitlement.description());
        // new FeatureValidator(feature).validate()
        logger.debug("Validating feature: " + requiredEntitlement.id());
        invoke(MISSING_ENTITLEMENT_MSG + requiredEntitlement.id(), validateFeature, getInstance(featureValidatorConstructor, feature), license);
    }

    private void addCustomEntitlement(Entitlement entitlement) {

        if (!isPresentInLicense(entitlement)){
            logger.debug(String.format("Entitlement [%s] is missing in current License. "
                    + "We'll check for external entitlements", entitlement.id()));
            LicenseProviderData licenseProviderData = null;
            CustomLicense customLicense;

            try {
                logger.debug("Loading provider data for entitlement " + entitlement.id());
                licenseProviderData = new LicenseProviderData(entitlement.provider(),
                        entitlement.licenseName(),
                        getMulePubicKey());

                logger.debug("Loading custom license information from license file " + entitlement.licenseName());
                customLicense = new CustomLicense(entitlement.id(), entitlement.licenseName(), licenseProviderData);
            } catch (InvalidKeyException e) {
                String email = licenseProviderData != null ? licenseProviderData.getEmail() : "";
                String msg = licenseProviderData != null ? licenseProviderData.getContactMessage() : "";
                throw new InvalidLicenseException(MISSING_ENTITLEMENT_MSG + entitlement.id() + ". Contact email: " + email + ". " + msg);
            }

            logger.debug("Validating custom Entitlement " + entitlement.id());
            if (!customLicense.isValid(entitlement)) {
                logInvalidLicenseError(entitlement, licenseProviderData, customLicense);
                throw new InvalidLicenseException(MISSING_ENTITLEMENT_MSG + entitlement.id() + ". "
                        + licenseProviderData.getContactMessage() + " Please Contact: " + licenseProviderData.getEmail());
            }

            logger.debug("Adding custom entitlement " + entitlement.id());
            addFeature(entitlement.id(), entitlement.description());
        } else {
            logger.debug(String.format("Entitlement [%s] already present in current License", entitlement.id()));
        }
    }

    private PublicKey getMulePubicKey(){
        if (mulePublicKey == null) {
            try {
                mulePublicKey = SecurityUtils.loadPublic(MULE_PUB_KEY);
            } catch (Exception e) {
                throw new InvalidLicenseException("A custom license is required but MuleSoft's public key [mule.pub] is missing.",
                        e);
            }
        }

        return mulePublicKey;
    }

    private Boolean isPresentInLicense(Entitlement entitlement) {
        boolean isCloudHub = true;
        try {
            // Ugly workaround for CloudHub not validating licenses with Licm
            Class.forName("com.cloudhub.extensions.tracking.NotificationHandler");
        } catch (Exception e) {
            isCloudHub = false;
        }

        logger.debug("Environment is CloudHub " + isCloudHub);
        return !isCloudHub && (Boolean) invoke(hasFeature, invoke(getFeatures, license), getInstance(featureConstructor, entitlement.id(), entitlement.description()));
    }

    private void addFeature(String id, String description) {
        // ((EnterpriseLicenseKey)license).setFeature(new Feature(id, description))
        invoke(setFeature, license, getInstance(featureConstructor, id, description));
    }

    private Object invoke(java.lang.reflect.Method method, Object instance, Object... args) {
        return invoke(DEFAULT_EXCEPTION_MSG, method, instance, args);
    }

    private Object invoke(String msgOnException, java.lang.reflect.Method method, Object instance, Object... args) {
        try {
            return method.invoke(instance, args);
        } catch (IllegalAccessException e) {
            return new InvalidLicenseException(msgOnException, e);
        } catch (InvocationTargetException e) {
            throw loggedException(e.getTargetException().getMessage(), new InvalidLicenseException(msgOnException));
        }
    }

    private void logInvalidLicenseError(Entitlement entitlement, LicenseProviderData licenseProviderData, CustomLicense customLicense) {
        String licName = entitlement.licenseName().concat(".lic");
        if (!customLicense.hasValidVersion(entitlement.version())) {
            logger.error("Your license " + licName + " is not valid for this connector version: " + entitlement.version());
        } else if (!customLicense.hasValidFeature()) {
            logger.error("Your license " + licName + "does not enable the feature [" + entitlement.id() + "] required by the module " + moduleName);
        } else {
            logger.error("Your license " + licName + " has expired on the " + customLicense.getProperty(CustomLicense.EXPIRATION_DATE_KEY).get());
        }

        logger.error("Please get in contact with your Vendor " + licenseProviderData.getName() +
                " using the following address: " + licenseProviderData.getProperty(LicenseProviderData.CONTACT_EMAIL_KEY).get());
        if (licenseProviderData.getProperty(LicenseProviderData.CONTACT_MESSAGE_KEY).isPresent()) {
            logger.error(licenseProviderData.getProperty(LicenseProviderData.CONTACT_MESSAGE_KEY).get());
        }
    }

    public Object getInstance(Constructor<?> constructor, Object... args) {
        try {
            return constructor.newInstance(args);
        } catch (InstantiationException e) {
            throw new InvalidLicenseException(DEFAULT_EXCEPTION_MSG, e);
        } catch (IllegalAccessException e) {
            throw new InvalidLicenseException(DEFAULT_EXCEPTION_MSG, e);
        } catch (InvocationTargetException e) {
            throw loggedException(e.getTargetException().getMessage(), new InvalidLicenseException(DEFAULT_EXCEPTION_MSG.concat(e.getMessage())));
        }
    }

    private RuntimeException loggedException(String loggedMessage, RuntimeException e) {
        logger.debug(loggedMessage);
        return e;
    }
}