/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.ws.model.cache;

import org.mule.devkit.api.ws.authentication.WsdlSecurityStrategy;
import org.mule.devkit.api.ws.definition.ServiceDefinition;
import org.mule.devkit.api.ws.exception.WrongParametrizationWsdlException;
import org.mule.devkit.api.ws.transport.WsdlTransport;
import org.mule.devkit.internal.ws.common.DefaultEnhancedServiceDefinition;
import org.mule.devkit.internal.ws.common.EnhancedServiceDefinition;
import org.mule.devkit.internal.ws.common.WsdlAdapter;
import org.mule.devkit.internal.ws.common.WsdlUtils;
import org.mule.devkit.internal.ws.model.WsdlIntrospecterUtils;
import org.mule.util.StringUtils;

import javax.wsdl.Definition;
import javax.wsdl.Service;
import java.util.List;
import java.util.concurrent.Callable;

public class EnhancedServiceDefinitionCallable implements Callable<EnhancedServiceDefinition> {

    private WsdlAdapter wsdlAdapter;
    private ServiceDefinition serviceDefinition;
    private String operation;

    public EnhancedServiceDefinitionCallable(WsdlAdapter wsdlAdapter, ServiceDefinition serviceDefinition, String operation) {
        this.wsdlAdapter = wsdlAdapter;
        this.serviceDefinition = serviceDefinition;
        this.operation = operation;
    }

    @Override
    public EnhancedServiceDefinition call() throws Exception {
        String serviceAddress = endpoint();
        WsdlTransport transport = wsdlAdapter.transport(serviceDefinition);
        List<WsdlSecurityStrategy> strategies = wsdlAdapter.security(serviceDefinition);
        return enhanceServiceDefinition(serviceDefinition, serviceAddress, strategies, transport, operation);
    }

    private String endpoint() throws Exception {
        String serviceAddress = wsdlAdapter.endpoint(serviceDefinition);
        if (StringUtils.isBlank(serviceAddress)) {
            throw new WrongParametrizationWsdlException("Service address obtained from @WsdlServiceEndpoint must not be null, nor empty.");
        }
        if (!(StringUtils.startsWith(serviceAddress, "http") || StringUtils.startsWith(serviceAddress, "https"))) {
            throw new WrongParametrizationWsdlException(
                    String.format("Service address obtained from @WsdlServiceEndpoint must start with 'http'/'https', but [%s] was found.", serviceAddress));
        }
        return serviceAddress;
    }

    private EnhancedServiceDefinition enhanceServiceDefinition(ServiceDefinition serviceDefinition, String serviceAddress, List<WsdlSecurityStrategy> strategies,
            WsdlTransport transport, String operation) {
        //Responsible of iterating over the darn wsdl file, and populate the service and port properly in the enhanced (and internal) model
        String serviceName = serviceDefinition.getService().orNull();
        String portName = serviceDefinition.getPort().orNull();

        if (!serviceDefinition.getService().isPresent()) {
            //fill service if needed
            Definition definition = WsdlUtils.parseWSDL(serviceDefinition.getWsdlUrl().toString());
            serviceName = WsdlIntrospecterUtils.resolveServiceName(serviceDefinition, definition);
        }

        if (!serviceDefinition.getPort().isPresent()) {
            //fill port if needed
            Definition definition = WsdlUtils.parseWSDL(serviceDefinition.getWsdlUrl().toString());
            Service service = WsdlIntrospecterUtils.resolveService(serviceDefinition, definition);
            portName = WsdlIntrospecterUtils.resolvePortName(serviceDefinition, service);
        }

        return new DefaultEnhancedServiceDefinition(serviceDefinition.getId(), serviceDefinition.getWsdlUrl(), serviceName, portName, serviceAddress, strategies, transport,
                operation);
    }
}
