/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.api.annotations.oauth;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * A class annotated with @{@link OAuth} is applied to a {@link org.mule.api.annotations.Connector} that uses the OAuth 1.0a protocol for authentication.
 * Basic information about the Service Provider is required to go through the OAuth 1.0a flow.
 *
 * @see OAuth2
 * @see org.mule.api.annotations.components.ConnectionManagement
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.SOURCE)
@Documented
public @interface OAuth {

  /**
   * The signature method to be used in the OAuth 1.0a flow. This method will be included in the auth_signature_method parameter.
   */
  OAuthMessageSigner messageSigner() default OAuthMessageSigner.HMAC_SHA1;

  /**
   * Defines where the OAuth 1.0a parameters should be included.
   */
  OAuthSigningStrategy signingStrategy() default OAuthSigningStrategy.AUTHORIZATION_HEADER;

  /**
   * Default SignPost OAuth provider
   */
  Class<?> provider() default Object.class;

  /**
   * The URL defined by the Service Provider used to obtain an un-authorized Request Token
   */
  String requestTokenUrl();

  /**
   * The URL defined by the Service Provider to obtain an Access Token
   */
  String accessTokenUrl();

  /**
   * The URL defined by the Service Provider where the Resource Owner will be redirected to grant
   * authorization to the Consumer
   */
  String authorizationUrl();

  /**
   * A Java regular expression used to extract the verifier from the Service Provider response as a result
   * of the Resource Owner authorizing the Consumer.
   */
  String verifierRegex() default "oauth_verifier=([^&]+)";

  /**
   * Extra authorization parameters that will be appended as a query string arguments when building
   * the authorization URL.
   */
  OAuthAuthorizationParameter[] authorizationParameters() default {};
}
