/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.mule.devkit.model.code;


/**
 * TypeReference for generating expressions containing operators
 */

abstract public class Op {

  private Op() {}


  /**
   * Determine whether the top level of an expression involves an
   * operator.
   */
  static boolean hasTopOp(GeneratedExpression e) {
    return (e instanceof UnaryOp) || (e instanceof BinaryOp);
  }

  /* -- Unary operators -- */

  static private class UnaryOp extends AbstractExpression {

    protected String op;
    protected GeneratedExpression e;
    protected boolean opFirst = true;

    UnaryOp(String op, GeneratedExpression e) {
      this.op = op;
      this.e = e;
    }

    UnaryOp(GeneratedExpression e, String op) {
      this.op = op;
      this.e = e;
      opFirst = false;
    }

    public void generate(Formatter f) {
      if (opFirst) {
        f.p('(').p(op).g(e).p(')');
      } else {
        f.p('(').g(e).p(op).p(')');
      }
    }

  }

  public static GeneratedExpression minus(GeneratedExpression e) {
    return new UnaryOp("-", e);
  }

  /**
   * Logical not <tt>'!x'</tt>.
   */
  public static GeneratedExpression not(GeneratedExpression e) {
    if (e == ExpressionFactory.TRUE) {
      return ExpressionFactory.FALSE;
    }
    if (e == ExpressionFactory.FALSE) {
      return ExpressionFactory.TRUE;
    }
    return new UnaryOp("!", e);
  }

  public static GeneratedExpression complement(GeneratedExpression e) {
    return new UnaryOp("~", e);
  }

  static private class TightUnaryOp extends UnaryOp {

    TightUnaryOp(GeneratedExpression e, String op) {
      super(e, op);
    }

    public void generate(Formatter f) {
      if (opFirst) {
        f.p(op).g(e);
      } else {
        f.g(e).p(op);
      }
    }

  }

  public static GeneratedExpression incr(GeneratedExpression e) {
    return new TightUnaryOp(e, "++");
  }

  public static GeneratedExpression decr(GeneratedExpression e) {
    return new TightUnaryOp(e, "--");
  }


  /* -- Binary operators -- */

  static private class BinaryOp extends AbstractExpression {

    String op;
    GeneratedExpression left;
    Generable right;

    BinaryOp(String op, GeneratedExpression left, Generable right) {
      this.left = left;
      this.op = op;
      this.right = right;
    }

    public void generate(Formatter f) {
      f.p('(');
      if (left instanceof Assignment) {
        f.p('(').g(left).p(')');
      } else {
        f.g(left);
      }
      f.p(op).g(right).p(')');
    }

  }

  public static GeneratedExpression plus(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("+", left, right);
  }

  public static GeneratedExpression minus(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("-", left, right);
  }

  public static GeneratedExpression mul(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("*", left, right);
  }

  public static GeneratedExpression div(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("/", left, right);
  }

  public static GeneratedExpression mod(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("%", left, right);
  }

  public static GeneratedExpression shl(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("<<", left, right);
  }

  public static GeneratedExpression shr(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp(">>", left, right);
  }

  public static GeneratedExpression shrz(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp(">>>", left, right);
  }

  public static GeneratedExpression band(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("&", left, right);
  }

  public static GeneratedExpression bor(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("|", left, right);
  }

  public static GeneratedExpression cand(GeneratedExpression left, GeneratedExpression right) {
    if (left == ExpressionFactory.TRUE) {
      return right;
    }
    if (right == ExpressionFactory.TRUE) {
      return left;
    }
    if (left == ExpressionFactory.FALSE) {
      return left; // ExpressionFactory.FALSE
    }
    if (right == ExpressionFactory.FALSE) {
      return right; // ExpressionFactory.FALSE
    }
    return new BinaryOp("&&", left, right);
  }

  public static GeneratedExpression cor(GeneratedExpression left, GeneratedExpression right) {
    if (left == ExpressionFactory.TRUE) {
      return left; // ExpressionFactory.TRUE
    }
    if (right == ExpressionFactory.TRUE) {
      return right; // ExpressionFactory.FALSE
    }
    if (left == ExpressionFactory.FALSE) {
      return right;
    }
    if (right == ExpressionFactory.FALSE) {
      return left;
    }
    return new BinaryOp("||", left, right);
  }

  public static GeneratedExpression xor(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("^", left, right);
  }

  public static GeneratedExpression lt(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("<", left, right);
  }

  public static GeneratedExpression lte(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("<=", left, right);
  }

  public static GeneratedExpression gt(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp(">", left, right);
  }

  public static GeneratedExpression gte(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp(">=", left, right);
  }

  public static GeneratedExpression eq(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("==", left, right);
  }

  public static GeneratedExpression ne(GeneratedExpression left, GeneratedExpression right) {
    return new BinaryOp("!=", left, right);
  }

  public static GeneratedExpression _instanceof(GeneratedExpression left, Type right) {
    return new BinaryOp("instanceof", left, right);
  }

  /* -- Ternary operators -- */

  static private class TernaryOp extends AbstractExpression {

    String op1;
    String op2;
    GeneratedExpression e1;
    GeneratedExpression e2;
    GeneratedExpression e3;

    TernaryOp(String op1, String op2,
              GeneratedExpression e1, GeneratedExpression e2, GeneratedExpression e3) {
      this.e1 = e1;
      this.op1 = op1;
      this.e2 = e2;
      this.op2 = op2;
      this.e3 = e3;
    }

    public void generate(Formatter f) {
      f.p('(').g(e1).p(op1).g(e2).p(op2).g(e3).p(')');
    }

  }

  public static GeneratedExpression cond(GeneratedExpression cond,
                                         GeneratedExpression ifTrue, GeneratedExpression ifFalse) {
    return new TernaryOp("?", ":", cond, ifTrue, ifFalse);
  }

}
