package com.mulesoft.exchange.mavenfacade;

import com.mulesoft.exchange.mavenfacade.utils.ZipUtils;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

public class ExchangeDocsDeployer {
    private final MavenPluginParameters mavenPluginParameters;
    private final DeployFilePlugin deployFilePlugin;
    public static final String EXCHANGE_DOCS_DIR = "exchange-docs";
    public static final String ZIPPED_EXCHANGE_DOCS_FILE = "exchange-docs.zip";

    public ExchangeDocsDeployer(MavenPluginParameters mavenPluginParameters, DeployFilePlugin deployFilePlugin) {
        this.mavenPluginParameters = mavenPluginParameters;
        this.deployFilePlugin = deployFilePlugin;
    }

    public void deploy() {
        String exchangeDocsPath = getExchangeDocsPath();
        if (directoryExists(exchangeDocsPath)) {
            if(directoryIsEmpty(exchangeDocsPath)) {
                throw new RuntimeException("The " + EXCHANGE_DOCS_DIR + " (path: " + exchangeDocsPath + ") directory can not be empty.");
            }
            deployExchangeDocs(exchangeDocsPath);
        }
    }

    private boolean directoryIsEmpty(String exchangeDocsPath) {
        File file = new File(exchangeDocsPath);
        return file.isDirectory() && file.list().length == 0;
    }

    private String getExchangeDocsPath() {
        String projectBasePath = mavenPluginParameters.getMavenProjectAbsolutePath();
        return projectBasePath + "/" + EXCHANGE_DOCS_DIR;
    }

    private boolean directoryExists(String directoryPath) {
        Path path = Paths.get(directoryPath);
        return Files.isDirectory(path);
    }

    private void deployExchangeDocs(String exchangeDocsPath) {
        String zippedDocsPath = zipExchangeDocs(exchangeDocsPath);

        mavenPluginParameters.getLogger().info("Executing Exchange Docs deployer");
        try {
            deployFilePlugin.deployFile(zippedDocsPath, "docs", "zip");
        } catch (Exception e) {
            throw new RuntimeException("Failed uploading "+ ZIPPED_EXCHANGE_DOCS_FILE + " file", e);
        }
    }

    private String zipExchangeDocs(String exchangeDocsPath) {
        try {
            Path tmpDirectory = Files.createTempDirectory(EXCHANGE_DOCS_DIR);
            String tmpDirectoryPath = tmpDirectory.toAbsolutePath().toString();
            String zippedDocsPath = tmpDirectoryPath + "/" + ZIPPED_EXCHANGE_DOCS_FILE;

            ZipUtils.compressDirectoryContent(exchangeDocsPath, zippedDocsPath);
            return zippedDocsPath;
        } catch (IOException e) {
            throw new RuntimeException("Failed creating " + ZIPPED_EXCHANGE_DOCS_FILE + " file", e);
        }
    }
}
