package com.mulesoft.exchange.mavenfacade;

import com.mulesoft.exchange.mavenfacade.exceptions.InvalidIconExtensionException;
import com.mulesoft.exchange.mavenfacade.utils.DeployerUtils;
import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Arrays;
import java.util.List;

public class ExchangeIconDeployer {
    private final DeployerUtils deployerUtils;
    private final Log logger;
    private static final String ICON_FILE_NAME = "icon";
    private static final String FILE_NAME_EXTENSION_REGEX = "\\.(?=[^.]+$)";
    public ExchangeIconDeployer(MavenPluginParameters mavenPluginParameters, DeployFilePlugin deployFilePlugin) {
        deployerUtils = new DeployerUtils(mavenPluginParameters, deployFilePlugin);
        logger = mavenPluginParameters.getLogger();
    }

    public void deploy() {
        String exchangeDocsPath = deployerUtils.getExchangeDocsPath();
        if (!deployerUtils.directoryExists(exchangeDocsPath)) {
            return;
        }
        File[] iconFiles = new File(exchangeDocsPath).listFiles(new IconFilenameFilter());
        if (iconFiles == null || iconFiles.length == 0) {
            return;
        }
        File icon = iconFiles[0]; // take the first icon found, ignore the rest
        String packaging =  icon.getName().split(FILE_NAME_EXTENSION_REGEX)[1];
        logger.info("Attempting icon upload of file " + icon.getAbsolutePath() + " with packaging " + packaging);
        deployerUtils.deploy(icon.getAbsolutePath(), "icon", packaging);
    }

    static class IconFilenameFilter implements FilenameFilter {
        private static final List<String> VALID_EXTENSIONS = Arrays.asList("jpg", "jpeg", "png", "gif", "svg");

        @Override
        public boolean accept(File dir, String fullName) {
            boolean isDirectory = new File(dir.getAbsolutePath(), fullName).isDirectory();
            String[] nameAndExtension = fullName.split(FILE_NAME_EXTENSION_REGEX);
            boolean isFileWithoutExtension = nameAndExtension.length < 2;
            if (isDirectory || isFileWithoutExtension) {
                return false;
            }
            String name = nameAndExtension[0];
            String extension = nameAndExtension[1];
            if(!name.equals(ICON_FILE_NAME)) {
                return false;
            }
            if(VALID_EXTENSIONS.contains(extension)) {
                return true;
            }
            throw new InvalidIconExtensionException("File detected as icon but with invalid extension: " + fullName +
                    ". Valid extensions are " + VALID_EXTENSIONS);
        }
    }
}
