package com.mulesoft.exchange.mavenfacade;
import com.google.common.io.Files;
import com.google.gson.Gson;
import com.mulesoft.exchange.mavenfacade.utils.ExchangeUriChecker;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.model.Plugin;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.twdata.maven.mojoexecutor.MojoExecutor;
import java.io.File;
import java.io.FileReader;
import java.lang.reflect.Type;
import java.util.*;

import static org.twdata.maven.mojoexecutor.MojoExecutor.*;

abstract public class AbstractDynamicJSONFileProvider {

    // Get from POM.xml
    private String GET_FILE_PLUGIN_GROUP_ID = "com.googlecode.maven-download-plugin";
    private String GET_FILE_PLUGIN_ARTIFACT_ID = "download-maven-plugin";
    private String GET_FILE_PLUGIN_VERSION = "1.5.0";

    private MavenPluginParameters mavenPluginParameters;
    private ExchangeUriChecker exchangeUriChecker;

    public AbstractDynamicJSONFileProvider(MavenPluginParameters mavenPluginParameters, ExchangeUriChecker exchangeUriChecker) {
        this.mavenPluginParameters = mavenPluginParameters;
        this.exchangeUriChecker = exchangeUriChecker;
    }

    public MavenPluginParameters getMavenPluginParameters() {
        return mavenPluginParameters;
    }

    abstract protected String getClassifier();

    abstract protected Type getDtoClass();

    private HashMap<ArtifactRepository, String> replaceMavenV3Repository(String remoteArtifactUrl, List<ArtifactRepository> remoteArtifactsRepositories){
        HashMap<ArtifactRepository,String> repositoriesToRestore = new HashMap<ArtifactRepository,String>();
        for (ArtifactRepository artifactRepository : remoteArtifactsRepositories) {
            if(exchangeUriChecker.isMavenV3UriWithoutOrganization(artifactRepository.getUrl())){
                repositoriesToRestore.put(artifactRepository,artifactRepository.getUrl());
                artifactRepository.setUrl(remoteArtifactUrl);
            }
        }
        return repositoriesToRestore;
    }

    private void restoreMavenV3Repository(HashMap<ArtifactRepository, String> repositoriesToRestore){
        Iterator<Map.Entry<ArtifactRepository, String> >
                iterator = repositoriesToRestore.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<ArtifactRepository, String> entry = iterator.next();
            entry.getKey().setUrl(entry.getValue());
        }
    }

    public <T> T getStatus() {
        MavenProject mavenProject = mavenPluginParameters.getMavenProject();

        File tempDir = Files.createTempDir();
        File tempFile = null;

        try {
            String groupId = mavenProject.getGroupId();
            String assetId = mavenProject.getArtifactId();
            String version = mavenProject.getVersion();
            String classifier = this.getClassifier() + "-" + new Date().getTime();
            String packaging = "json";
            String fileName = classifier + "." + packaging;

            Plugin deployFilePlugin = plugin(groupId(GET_FILE_PLUGIN_GROUP_ID), artifactId(GET_FILE_PLUGIN_ARTIFACT_ID), version(GET_FILE_PLUGIN_VERSION));
            String goal = goal("artifact");

            Xpp3Dom pluginConfiguration = configuration(
                    element(name("groupId"), groupId),
                    element(name("artifactId"), assetId),
                    element(name("version"), version),
                    element(name("classifier"), classifier),
                    element(name("type"), "json"),
                    element(name("outputDirectory"), tempDir.getAbsolutePath()),
                    element(name("outputFileName"), fileName)
            );

            List<ArtifactRepository> remoteArtifactsRepositories = mavenPluginParameters.getMavenProject().getRemoteArtifactRepositories();
            String remoteArtifactUrl = mavenPluginParameters.getMavenProject().getDistributionManagementArtifactRepository().getUrl();
            HashMap<ArtifactRepository, String> remoteArtifactsToRestore = replaceMavenV3Repository(remoteArtifactUrl, remoteArtifactsRepositories);

            ExecutionEnvironment executionEnvironment = executionEnvironment(mavenPluginParameters.getMavenProject(), mavenPluginParameters.getMavenSession(), mavenPluginParameters.getPluginManager());

            MojoExecutor.executeMojo(deployFilePlugin, goal, pluginConfiguration, executionEnvironment);

            restoreMavenV3Repository(remoteArtifactsToRestore);

            tempFile = new File(tempDir.getAbsolutePath(), fileName);

            Gson gson = new Gson();
            return gson.fromJson(new FileReader(tempFile), getDtoClass());
        } catch (Exception e) {
            throw new RuntimeException("Unexpected error while processing the publication", e);
        } finally {
            if (tempFile != null) {
                tempFile.delete();
            }

            tempDir.delete();
        }
    }
}
