package com.mulesoft.exchange.mavenfacade;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.mulesoft.exchange.mavenfacade.dto.Status;
import com.mulesoft.exchange.mavenfacade.dto.Step;
import com.mulesoft.exchange.mavenfacade.utils.ExchangeUriChecker;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

import java.util.List;

public class PublicationStatusChecker {

    private static final Integer GET_STATUS_MAX_RETRIES = 50;
    private static final Integer GET_STATUS_DELAY = 3000;

    private PublicationStatusProvider publicationStatusProvider;
    private ExchangeUriChecker exchangeUriChecker;
    private Log logger;

    public PublicationStatusChecker(PublicationStatusProvider publicationStatusProvider, ExchangeUriChecker exchangeUriChecker, Log logger) {
        this.publicationStatusProvider = publicationStatusProvider;
        this.exchangeUriChecker = exchangeUriChecker;
        this.logger = logger;
    }

    public void checkStatus() throws InterruptedException {
        Status status = new Status("initial");
        int attempt = 0;

        while (!status.getStatus().equals("completed") && !status.getStatus().equals("error")) {
            logger.info("Getting publication status");

            if (attempt > GET_STATUS_MAX_RETRIES) {
                throw new RuntimeException("Asset publication didn't finish in the expected time");
            }

            try {
                status = publicationStatusProvider.getStatus();

                logger.info("  ------------------------------------------------------------");
                logger.info("  Publication status: " + status.getStatus());
                logger.info("  ------------------------------------------------------------");
                logger.info("    Steps: ");

                for (Step step : status.getSteps()) {
                    logger.info("    - Description: " + step.getDescription());
                    logger.info("    - Status: " + step.getStatus());

                    if (step.getErrors() != null && !step.getErrors().isEmpty()) {
                        logger.info("    - Errors: " + step.getErrors());
                    }

                    logger.info("    .........................................");
                }
            } catch (Exception e) {
                logger.warn("Failed to get publication status", e);
            }

            Thread.sleep(GET_STATUS_DELAY);
            attempt++;
        }

        // Check error step
        if (status.getStatus().equals("error")) {
            List<Step> steps = status.getSteps();
            Step lastStep = steps.get(steps.size() - 1);

            throw new RuntimeException("Publication ended with errors: " + lastStep.getErrors());
        }

        if (status.getStatus().equals("completed")) {
            MavenProject mavenProject = publicationStatusProvider.getMavenPluginParameters().getMavenProject();

            String groupId = mavenProject.getGroupId();
            String assetId = mavenProject.getArtifactId();
            String version = mavenProject.getVersion();
            String basePlatformUri = exchangeUriChecker.extractBaseAnypointUriFromMavenRepositoryUri(mavenProject.getDistributionManagementArtifactRepository().getUrl());

            List<String> elements = Lists.newArrayList(basePlatformUri.split("/"));
            elements.add("exchange");
            elements.add(groupId);
            elements.add(assetId);
            elements.add(version);

            String assetUri = Joiner.on("/").skipNulls().join(elements);

            logger.info("");
            logger.info("  Your asset has been successfully published to Exchange.");
            logger.info("  You can check it at: " + assetUri);
        }
    }
}
