package com.mulesoft.exchange.mavenfacade;

import com.google.common.collect.Lists;
import com.mulesoft.exchange.mavenfacade.utils.ExchangeHelper;
import com.mulesoft.exchange.mavenfacade.utils.ExchangeUriChecker;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryPolicy;
import org.apache.maven.artifact.repository.MavenArtifactRepository;
import org.apache.maven.artifact.repository.layout.DefaultRepositoryLayout;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.DeploymentRepository;
import org.apache.maven.model.DistributionManagement;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.BuildPluginManager;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class AbstractExchangeMojo extends AbstractMojo {

    private static final Pattern ALT_REPO_SYNTAX_PATTERN = Pattern.compile("(.+)::(.+)::(.+)");

    @Component
    protected MavenProject mavenProject;

    @Component
    protected MavenSession mavenSession;

    @Component
    protected BuildPluginManager pluginManager;

    @Parameter
    protected List<String> customDomains = new ArrayList<>();

    @Parameter( property = "altDeploymentRepository" )
    protected String altDeploymentRepository;

    @Parameter
    protected boolean skip = false;

    ExchangeUriChecker exchangeUriChecker;
    ExchangeHelper exchangeHelper;

    public void execute() throws MojoExecutionException {
        handlingAltDeploymentRepository();

        if (!skip) {
            initMojo();
            doExecute();
        }
    }

    private void handlingAltDeploymentRepository() throws MojoExecutionException {
        if (altDeploymentRepository != null) {
            getLog().info("Using alternate deployment repository " + altDeploymentRepository);

            Matcher matcher = ALT_REPO_SYNTAX_PATTERN.matcher(altDeploymentRepository);

            if (!matcher.matches()) {
                throw new MojoExecutionException(altDeploymentRepository, "Invalid syntax for repository.",
                        "Invalid syntax for alternative repository. Use \"id::layout::url\".");
            } else {
                String id = matcher.group(1).trim();
                String layout = matcher.group(2).trim();
                String url = matcher.group(3).trim();

                DistributionManagement distributionManagement = new DistributionManagement();
                DeploymentRepository deploymentRepository = new DeploymentRepository();
                deploymentRepository.setId(id);
                deploymentRepository.setUrl(url);
                deploymentRepository.setLayout(layout);
                distributionManagement.setRepository(deploymentRepository);

                ArtifactRepositoryPolicy releasesPolicy = new ArtifactRepositoryPolicy(true,"daily","warn");
                ArtifactRepositoryPolicy snapshotPolicy = new ArtifactRepositoryPolicy(true,"daily","warn");
                ArtifactRepository artifactRepository = new MavenArtifactRepository(id,url, new DefaultRepositoryLayout(), snapshotPolicy, releasesPolicy);
                mavenProject.setReleaseArtifactRepository(artifactRepository);
                mavenProject.setDistributionManagement(distributionManagement);
            }
        }
    }

    public abstract void doExecute() throws MojoExecutionException;

    private void initMojo() {
        Set<String> storedCustomDomains = new ExchangeHelper().retrieveCustomDomains(mavenSession.getSystemProperties());
        storedCustomDomains.addAll(customDomains);

        List<String> storedCustomDomainsList = Lists.newArrayList(storedCustomDomains);

        exchangeUriChecker = new ExchangeUriChecker(storedCustomDomainsList);
        exchangeHelper = new ExchangeHelper(storedCustomDomainsList);
    }
}
